#include "rt3050.h"
#include "../../../../fs/proc/internal.h"

static int get_num_from_parent_parent_dir_name(struct file *file, int max_value)
{
	struct proc_dir_entry *pde = PDE(file_inode(file));
	int num = pde->parent->parent->name[0] - '0';

	if ((num > max_value) || (num < 0))
		return -EINVAL;
	return num;
}

static struct rt3050_priv * get_rt3050_priv_from_file(struct file *file)
{
	struct proc_dir_entry *pde = PDE(file_inode(file));

	return  (struct rt3050_priv *)(pde->data);
}

static ssize_t write_qos_x(struct file *file, const char __user *buf, size_t size,
	loff_t *pos, void (*set_qos_x)(struct rt3050_priv *, int, u32), struct rt3050_priv *priv)
{
	char *data, *endp;
	unsigned long value;
	int queue = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	if (queue < 0)
		return -EINVAL;
	if (size <= 1)
		return -EINVAL;
	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	value = simple_strtoul(data, &endp, 0);
	if (value > 0xFFFF)
		return -ERANGE;

	mutex_lock(&priv->mtx);
	set_qos_x(priv, queue, (u32)value);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_qos_x(struct file *file, char __user *buf, size_t size,
	loff_t *pos, u32 (*get_qos_x)(struct rt3050_priv *, int), struct rt3050_priv *priv)
{
	u32 value;
	ssize_t written;
	char resp[64];
	int queue = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	if (queue < 0)
		return -EINVAL;

	mutex_lock(&priv->mtx);
	value = get_qos_x(priv, queue);
	mutex_unlock(&priv->mtx);
	written = snprintf(resp, sizeof(resp), "%d\n", value);

	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static ssize_t write_qos_x_many(struct file *file, const char __user *buf, size_t size,
	loff_t *pos, void (*set_qos_x)(struct rt3050_priv *, int, u32*, int), 
	u32 *values, int maxvalues, struct rt3050_priv *priv)
{
	char *data;
	int queue, count = 0, i;

	queue = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);
	if (queue < 0)
		return -EINVAL;

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	for (i = 0; i < size; i++) {
		if ((count >= maxvalues) || (data[i] == '\n') || (data[i] == '\0'))
			break;
		if ((data[i] >= '0') && (data[i] <= '9'))
			values[count++] = (u32)(data[i] - '0');
	}

	mutex_lock(&priv->mtx);
	set_qos_x(priv, queue, values, count);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_qos_x_many(struct file *file, char __user *buf, size_t size,
	loff_t *pos, u32* (*get_qos_x)(struct rt3050_priv *, int, u32*, int*), 
	u32 *values, struct rt3050_priv *priv)
{
	char resp[512], *ptr = resp;
	size_t remain = sizeof(resp);
	int count, queue = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	if (queue < 0)
		return -EINVAL;

	mutex_lock(&priv->mtx);
	get_qos_x(priv, queue, values, &count);
	mutex_unlock(&priv->mtx);

	for (int i = 0; i < count; i++) {
		ssize_t written = snprintf(ptr, remain, "%d ", values[i]);
		ptr += written;
		remain -= written;
	}

	return simple_read_from_buffer(buf, size, pos, resp, ptr - resp + 1);
}


static ssize_t write_queue_weight(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_queue_weight, priv);
}

static ssize_t read_queue_weight(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_queue_weight, priv);
}

static const struct proc_ops queue_num_proc_ops = {
	.proc_write	= write_queue_weight,
	.proc_read	= read_queue_weight,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_queue_threshold(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_queue_threshold, priv);
}

static ssize_t read_queue_threshold(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_queue_threshold, priv);
}

static const struct proc_ops queue_threshold_proc_ops = {
	.proc_write	= write_queue_threshold,
	.proc_read	= read_queue_threshold,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_queue_tags(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	u32 tags[RT3050_NUM_TAGS];

	return write_qos_x_many(file, buf, size, pos, rt3050_set_queue_tags, tags, RT3050_NUM_TAGS, priv);
}

static ssize_t read_queue_tags(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	u32 tags[RT3050_NUM_TAGS];

	return read_qos_x_many(file, buf, size, pos, rt3050_get_queue_tags, tags, priv);
}

static const struct proc_ops queue_tags_proc_ops = {
	.proc_write	= write_queue_tags,
	.proc_read	= read_queue_tags,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_queue_ports(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	u32 ports[RT3050_NUM_PORTS];

	return write_qos_x_many(file, buf, size, pos, rt3050_set_queue_ports, ports, RT3050_NUM_PORTS, priv);
}

static ssize_t read_queue_ports(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	u32 ports[RT3050_NUM_PORTS];

	return read_qos_x_many(file, buf, size, pos, rt3050_get_queue_ports, ports, priv);
}

static const struct proc_ops queue_ports_proc_ops = {
	.proc_write	= write_queue_ports,
	.proc_read	= read_queue_ports,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_threshold(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_port_threshold, priv);
}

static ssize_t read_port_threshold(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_port_threshold, priv);
}

static const struct proc_ops port_threshold_proc_ops = {
	.proc_write	= write_port_threshold,
	.proc_read	= read_port_threshold,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_turnoff(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_port_turnoff, priv);
}

static ssize_t read_port_turnoff(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_port_turnoff, priv);
}

static const struct proc_ops port_turnoff_proc_ops = {
	.proc_write	= write_port_turnoff,
	.proc_read	= read_port_turnoff,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_tos(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_port_tos, priv);
}

static ssize_t read_port_tos(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_port_tos, priv);
}

static const struct proc_ops port_tos_proc_ops = {
	.proc_write	= write_port_tos,
	.proc_read	= read_port_tos,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_vlan(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_port_vlan, priv);
}

static ssize_t read_port_vlan(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_port_vlan, priv);
}

static const struct proc_ops port_vlan_proc_ops = {
	.proc_write	= write_port_vlan,
	.proc_read	= read_port_vlan,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_priority(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_port_priority, priv);
}

static ssize_t read_port_priority(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_port_priority, priv);
}

static const struct proc_ops port_priority_proc_ops = {
	.proc_write	= write_port_priority,
	.proc_read	= read_port_priority,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_secured_mode(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	char *data;
	int count = 0;
	struct rt3050_sa_secured_mac_addr_entry values[RT3050_NUM_VLANS];
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	int port = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	if (port < 0)
		return -EINVAL;

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	if (size <= ETH_ALEN || data[0] == '\0') {
		mutex_lock(&priv->mtx);
		rt3050_set_port_secured_mode(priv, port, values, 0);
		mutex_unlock(&priv->mtx);
		return size;
	}

	for (int i = 0; i < size; i++) {
		if (count >= RT3050_NUM_VLANS)
			break;
		if ((data[i] == '\n') || (data[i] == '\0'))
			break;
		if (data[i] == ' ')
			continue;
		if ((sscanf(&data[i], "%2hhx:%2hhx:%2hhx:%2hhx:%2hhx:%2hhx", 
			values[count].mac, 
			&values[count].mac[1], 
			&values[count].mac[2], 
			&values[count].mac[3], 
			&values[count].mac[4], 
			&values[count].mac[5]) == 6) && 
			(is_unicast_ether_addr(values[count].mac) || is_multicast_ether_addr(values[count].mac))) {
			count ++;
			i += 17;
		}
	}

	mutex_lock(&priv->mtx);
	rt3050_set_port_secured_mode(priv, port, values, count);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_port_secured_mode(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	char resp[512], *ptr = resp;
	size_t remain = sizeof(resp);
	struct rt3050_sa_secured_mac_addr_entry values[RT3050_NUM_VLANS];
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	int count, port = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	if (port < 0)
		return -EINVAL;

	mutex_lock(&priv->mtx);
	rt3050_get_port_secured_mode(priv, port, values, &count);
	mutex_unlock(&priv->mtx);

	for (int i = 0; i < count; i++) {
		ssize_t written = snprintf(ptr, remain, "%pM ", 
				values[i].mac);
		ptr += written;
		remain -= written;
	}

	return simple_read_from_buffer(buf, size, pos, resp, ptr - resp + 1);
}

static const struct proc_ops port_secured_mode_proc_ops = {
	.proc_write	= write_port_secured_mode,
	.proc_read	= read_port_secured_mode,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_led_state(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	char *data;
	u32 led;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	int port = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	if (!strncmp(data, "off", 3))
		led = RT3050_LED_OFF;
	else if (!strncmp(data, "on", 2))
		led = RT3050_LED_ON;
	else if (!strncmp(data, "default", 7))
		led = RT3050_LED_LINKACT;
	else
		return -EINVAL;

	mutex_lock(&priv->mtx);
	rt3050_set_port_led_state(priv, port, led);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_port_led_state(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	char resp[8];
	int led;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	int port = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	mutex_lock(&priv->mtx);
	led = rt3050_get_port_led_state(priv, port);
	mutex_unlock(&priv->mtx);
	switch (led) {
		case RT3050_LED_OFF:
			snprintf(resp, sizeof(resp), "off");
			break;
		case RT3050_LED_ON:
			snprintf(resp, sizeof(resp), "on");
			break;
		case RT3050_LED_LINKACT:
			snprintf(resp, sizeof(resp), "default");
			break;
		default:
			snprintf(resp, sizeof(resp), "?");
	}

	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static const struct proc_ops port_led_state_proc_ops = {
	.proc_write	= write_port_led_state,
	.proc_read	= read_port_led_state,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_port_doubletag(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	char *data;
	u32 doubletag;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	int port = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	if (!strncmp(data, "yes", 3))
		doubletag = 1;
	else if (!strncmp(data, "no", 2))
		doubletag = 0;
	else
		return -EINVAL;

	mutex_lock(&priv->mtx);
	rt3050_set_port_doubletag(priv, port, doubletag);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_port_doubletag(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	char resp[4];
	int doubletag;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);
	int port = get_num_from_parent_parent_dir_name(file, RT3050_NUM_PORTS);

	mutex_lock(&priv->mtx);
	doubletag = rt3050_get_port_doubletag(priv, port);
	mutex_unlock(&priv->mtx);
	if (doubletag) {
		snprintf(resp, sizeof(resp), "yes");
	} else {
		snprintf(resp, sizeof(resp), "no");
	}

	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static const struct proc_ops port_doubletag_proc_ops = {
	.proc_write	= write_port_doubletag,
	.proc_read	= read_port_doubletag,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_queue_priority_tag(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return write_qos_x(file, buf, size, pos, rt3050_set_queue_priority_tag, priv);
}

static ssize_t read_queue_priority_tag(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	return read_qos_x(file, buf, size, pos, rt3050_get_queue_priority_tag, priv);
}

static const struct proc_ops queue_priority_tag_proc_ops = {
	.proc_write	= write_queue_priority_tag,
	.proc_read	= read_queue_priority_tag,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_pref_queue(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	char *data;
	u32 pref;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	if (!strncmp(data, "tos", 4))
		pref = 0x1;
	else if (!strncmp(data, "pcp", 4))
		pref = 0x0;
	else
		return -EINVAL;

	mutex_lock(&priv->mtx);
	rt3050_set_pref_queue(priv, pref);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_pref_queue(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	char resp[4];
	int pref;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	mutex_lock(&priv->mtx);
	pref = rt3050_get_pref_queue(priv);
	mutex_unlock(&priv->mtx);

	if (pref)
		snprintf(resp, sizeof(resp), "tos");
	else
		snprintf(resp, sizeof(resp), "pcp");

	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static const struct proc_ops pref_queue_proc_ops = {
	.proc_write	= write_pref_queue,
	.proc_read	= read_pref_queue,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_fct0(struct file *file, const char __user *buf, size_t size,
	loff_t *pos, int is_fc)
{
	char *data, *endp;
	unsigned long val;
	int start, opposite_val;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	if (is_fc)
		start = RT3050_FCT0_FC_RLS_TH_S;
	else
		start = RT3050_FCT0_DROP_RLS_TH_S;

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	val = simple_strtoul(data, &endp, 0);
	if (val > 0xFFFF)
		return -EINVAL;

	mutex_lock(&priv->mtx);

	if (is_fc) {
		opposite_val = (int)rt3050_get_fct0_reg(priv, RT3050_FCT0_DROP_RLS_TH_S);
		if ((int)val > opposite_val) {
			rt3050_set_fct0_reg(priv, RT3050_FCT0_DROP_RLS_TH_S, (int)val);
			rt3050_set_fct0_reg(priv, RT3050_FCT0_DROP_SET_TH_S, (int)val);
		}
		rt3050_set_fct0_reg(priv, RT3050_FCT0_FC_RLS_TH_S, (int)val);
		rt3050_set_fct0_reg(priv, RT3050_FCT0_FC_SET_TH_S, (int)val);
	}
	else {
		opposite_val = (int)rt3050_get_fct0_reg(priv, RT3050_FCT0_FC_RLS_TH_S);
		if ((int)val < opposite_val) {
			rt3050_set_fct0_reg(priv, RT3050_FCT0_FC_RLS_TH_S, (int)val);
			rt3050_set_fct0_reg(priv, RT3050_FCT0_FC_SET_TH_S, (int)val);
		}
		rt3050_set_fct0_reg(priv, RT3050_FCT0_DROP_RLS_TH_S, (int)val);
		rt3050_set_fct0_reg(priv, RT3050_FCT0_DROP_SET_TH_S, (int)val);
	}

	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_fct0(struct file *file, char __user *buf, size_t size,
	loff_t *pos, int is_fc)
{
	char resp[5];
	int val, start;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	if (is_fc)
		start = RT3050_FCT0_FC_RLS_TH_S;
	else
		start = RT3050_FCT0_DROP_RLS_TH_S;

	mutex_lock(&priv->mtx);
	val = rt3050_get_fct0_reg(priv, start);
	mutex_unlock(&priv->mtx);

	snprintf(resp, sizeof(resp), "%d", val);

	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static ssize_t write_fc(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	return write_fct0(file, buf, size, pos, 1);
}

static ssize_t read_fc(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{

	return read_fct0(file, buf, size, pos, 1);
}

static const struct proc_ops fc_proc_ops = {
	.proc_write	= write_fc,
	.proc_read	= read_fc,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_drop(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	return write_fct0(file, buf, size, pos, 0);
}

static ssize_t read_drop(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	return read_fct0(file, buf, size, pos, 0);
}

static const struct proc_ops drop_proc_ops = {
	.proc_write	= write_drop,
	.proc_read	= read_drop,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_bc_storm(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	char *data;
	u32 bc;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	if (!strncmp(data, "none", 4))
		bc = 0x0;
	else if (!strncmp(data, "soft", 4))
		bc = 0x3;
	else if (!strncmp(data, "medium", 6))
		bc = 0x2;
	else if (!strncmp(data, "strong", 6))
		bc = 0x1;
	else
		return -EINVAL;

	mutex_lock(&priv->mtx);
	priv->bc_storm_protect = bc;
	rt3050_set_bc_storm_control(priv, bc);
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_bc_storm(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	char resp[7];
	int val;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	mutex_lock(&priv->mtx);
	val = rt3050_get_bc_storm_control(priv);
	mutex_unlock(&priv->mtx);
	switch (val) {
		case 0x1:
			snprintf(resp, sizeof(resp), "strong");
			break;
		case 0x2:
			snprintf(resp, sizeof(resp), "medium");
			break;
		case 0x3:
			snprintf(resp, sizeof(resp), "soft");
			break;
		default:
			snprintf(resp, sizeof(resp), "none");
	}

	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static const struct proc_ops bc_storm_proc_ops = {
	.proc_write	= write_bc_storm,
	.proc_read	= read_bc_storm,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static ssize_t write_offloading(struct file *file, const char __user *buf, size_t size,
	loff_t *pos)
{
	char *data;
	bool val;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	data = simple_transaction_get(file, buf, size);
	if (IS_ERR(data))
		return PTR_ERR(data);

	if (!strncmp(data, "yes", 3))
		val = true;
	else if (!strncmp(data, "no", 2))
		val = false;
	else
		return -EINVAL;

	mutex_lock(&priv->mtx);
	if (val != priv->do_bridge_offloading) {
		for (int i = 0; i <= RT3050_PORT5; i++) {
			if (priv->global_vlan_enable) {
				if (dsa_is_user_port(priv->ds, i)) {
					if (val && !priv->ports[i].filtering) {
						for (int vlan = 0; vlan < RT3050_NUM_VLANS; vlan++) {
							if (priv->vlans[vlan].bridge_ptr && (priv->vlans[vlan].ports & BIT(i))) {
								rt3050_set_pvid(priv, i, (u16)vlan);
								break;
							}
						}
					} else rt3050_set_pvid(priv, i, dsa_8021q_rt3050_port_to_vid(i));
				}
			} else {
				rt3050_set_pvid(priv, i, 0);
			}
		}
		priv->do_bridge_offloading = val;
	}
	mutex_unlock(&priv->mtx);

	return size;
}

static ssize_t read_offloading(struct file *file, char __user *buf, size_t size,
	loff_t *pos)
{
	char resp[4];
	bool val;
	struct rt3050_priv *priv = get_rt3050_priv_from_file(file);

	mutex_lock(&priv->mtx);
	val = priv->do_bridge_offloading;
	mutex_unlock(&priv->mtx);
	snprintf(resp, sizeof(resp), val ? "yes" : "no");
	return simple_read_from_buffer(buf, size, pos, resp, strlen(resp));
}

static const struct proc_ops offloading_proc_ops = {
	.proc_write	= write_offloading,
	.proc_read	= read_offloading,
	.proc_lseek	= default_llseek,
	.proc_release	= simple_transaction_release,
};

static int remove_procfs_numbered_dirs(struct proc_dir_entry *parent, int count)
{
	for (int i = 0; i < count; i++) {
		char buf[16];

		snprintf(buf, sizeof(buf), "%d", i);
		remove_proc_entry(buf, parent);
	}
	proc_remove(parent);
	return -EINVAL;
}

static int rt3050_queues_create_procfs(struct rt3050_priv *priv, struct proc_dir_entry *upentry)
{
	struct proc_dir_entry *qentry;

	qentry = proc_mkdir("queues", upentry);
	if (!qentry)
		return -ENOMEM;

	if (!proc_create_data("preference_to_select_queue", 0666, qentry, &pref_queue_proc_ops, priv))
		return -EINVAL;

	for (int i = 0; i < RT3050_NUM_QUEUES; i++) {
		char buf[16];
		struct proc_dir_entry *dentry, *fentry, *qsentry;

		snprintf(buf, sizeof(buf), "%d", i);
		dentry = proc_mkdir(buf, qentry);
		if (!dentry)
			return remove_procfs_numbered_dirs(qentry, i);

		qsentry = proc_mkdir("qos", dentry);
		if (!qsentry)
			return remove_procfs_numbered_dirs(qentry, i);

		fentry = proc_create_data("tagmap", 0666, qsentry, &queue_tags_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(qentry, i);

		fentry = proc_create_data("weight", 0666, qsentry, &queue_num_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(qentry, i);

		fentry = proc_create_data("flow_control_threshold", 0666, qsentry, &queue_threshold_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(qentry, i);

		fentry = proc_create_data("portmap", 0666, qsentry, &queue_ports_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(qentry, i);

		fentry = proc_create_data("priority_tag", 0666, qsentry, &queue_priority_tag_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(qentry, i);
	}
	return 0;
}

static int rt3050_ports_create_procfs(struct rt3050_priv *priv, struct proc_dir_entry *upentry)
{
	struct proc_dir_entry *pentry;

	pentry = proc_mkdir("ports", upentry);
	if (!pentry)
		return -ENOMEM;
	for (int i = 0; i < RT3050_NUM_PORTS; i++) {
		char buf[16];
		struct proc_dir_entry *dentry, *fentry, *qsentry;

		snprintf(buf, sizeof(buf), "%d", i);
		dentry = proc_mkdir(buf, pentry);
		if (!dentry)
			return remove_procfs_numbered_dirs(pentry, i);

		qsentry = proc_mkdir("qos", dentry);
		if (!qsentry)
			return remove_procfs_numbered_dirs(pentry, i);

		fentry = proc_create_data("enable_flow_control_on_threshold", 0666, qsentry, &port_threshold_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

		fentry = proc_create_data("auto_turn_off_flow_control", 0666, qsentry, &port_turnoff_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

		fentry = proc_create_data("use_tos_to_select_queue", 0666, qsentry, &port_tos_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

		fentry = proc_create_data("use_pcp_to_select_queue", 0666, qsentry, &port_vlan_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

		fentry = proc_create_data("default_queue", 0666, qsentry, &port_priority_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);
		
		qsentry = proc_mkdir("security", dentry);
		if (!qsentry)
			return remove_procfs_numbered_dirs(pentry, i);
		
		fentry = proc_create_data("addresses", 0666, qsentry, &port_secured_mode_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

		qsentry = proc_mkdir("led", dentry);
		if (!qsentry)
			return remove_procfs_numbered_dirs(pentry, i);
		
		fentry = proc_create_data("state", 0666, qsentry, &port_led_state_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

		qsentry = proc_mkdir("mgmt", dentry);
		if (!qsentry)
			return remove_procfs_numbered_dirs(pentry, i);
		
		fentry = proc_create_data("doubletagging", 0666, qsentry, &port_doubletag_proc_ops, priv);
		if (!fentry)
			return remove_procfs_numbered_dirs(pentry, i);

	}
	return 0;
}

static int rt3050_fct_create_procfs(struct rt3050_priv *priv, struct proc_dir_entry *dentry)
{
	struct proc_dir_entry *fentry;

	fentry = proc_create_data("flow_control", 0666, dentry, &fc_proc_ops, priv);
	if (!fentry)
		return -EINVAL;
	fentry = proc_create_data("drop", 0666, dentry, &drop_proc_ops, priv);
	if (!fentry)
		return -EINVAL;

	return 0;
}

static int rt3050_mgmt_create_procfs(struct rt3050_priv *priv, struct proc_dir_entry *dentry)
{
	if (!proc_create_data("bcast_storm_protection", 0666, dentry, &bc_storm_proc_ops, priv))
		return -EINVAL;

	if (!proc_create_data("offloading", 0666, dentry, &offloading_proc_ops, priv))
		return -EINVAL;

	dentry = proc_mkdir("thresholds", dentry);
	if (!dentry)
		return -EINVAL;

	if (rt3050_fct_create_procfs(priv, dentry))
		return -EINVAL;

	return 0;
}

void rt3050_switch_remove_procfs(struct rt3050_priv *priv)
{
	if (priv->procentry) {
		proc_remove(priv->procentry);
		priv->procentry = NULL;
	}
}

int rt3050_switch_create_procfs(struct rt3050_priv *priv)
{
	struct proc_dir_entry	*dentry;

	priv->procentry = proc_mkdir("switch", NULL);
	if (!priv->procentry) {
		dev_err(priv->dev, "Could not create procfs SWITCH entry\n");
		return -ENOMEM;
	}

	if (rt3050_queues_create_procfs(priv, priv->procentry)) {
		rt3050_switch_remove_procfs(priv);
		return -ENOMEM;
	}
	if (rt3050_ports_create_procfs(priv, priv->procentry)) {
		rt3050_switch_remove_procfs(priv);
		return -ENOMEM;
	}

	dentry = proc_mkdir("mgmt", priv->procentry);
	if (!dentry) {
		dev_err(priv->dev, "Could not create procfs Management entry\n");
		rt3050_switch_remove_procfs(priv);
		return -ENOMEM;
	}
	if (rt3050_mgmt_create_procfs(priv, dentry)) {
		rt3050_switch_remove_procfs(priv);
		return -ENOMEM;
	}

	return 0;
}
