#include <linux/module.h>
#include <linux/platform_device.h>
#include <asm/mach-ralink/ralink_regs.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/if_bridge.h>
#include <linux/dsa/8021q.h>
#include <linux/switch.h>
#include <linux/reset.h>

#include "rt3050.h"

static int rt3050_setup_irq(struct rt3050_priv *priv)
{
	struct device *dev = priv->dev;
	struct device_node *np = dev->of_node;
	int ret;

	rt3050_disable_irq(priv);
	priv->irq = irq_of_parse_and_map(np, 0);

	if (priv->irq <= 0) {
		dev_err(dev, "Failed to get IRQ: %d from OF\n", priv->irq);
		return -EINVAL;
	}

	ret = devm_request_irq(dev, priv->irq, rt3050_irq_handler, 0, RT3050_DRIVER_NAME,
			priv);

	if (ret)
		dev_err(dev, "Could not assign IRQ %d handler\n", priv->irq);

	return ret;
}

static int rt3050_phy_write(struct dsa_switch *ds, int addr, int reg, u16 val)
{
	struct rt3050_priv *priv = ds->priv;
	int ret;

	mutex_lock(&priv->mtx);
	ret = rt3050_mii_write(priv, addr, reg, (u32)val);
	mutex_unlock(&priv->mtx);
	return ret;
}

static int rt3050_phy_read(struct dsa_switch *ds, int addr, int reg)
{
	struct rt3050_priv *priv = ds->priv;
	int ret;

	mutex_lock(&priv->mtx);
	ret = rt3050_mii_read(priv, addr, reg);
	mutex_unlock(&priv->mtx);
	return ret;
}

static int rt3050_get_vlan_from_bridge(struct rt3050_priv *priv, struct net_device *bridge)
{
	int empty_vlan = 0;

	for (int vlan = 0; vlan < RT3050_NUM_VLANS; vlan++) {
		if (!empty_vlan && !priv->vlans[vlan].ports)
			empty_vlan = vlan;
		if (priv->vlans[vlan].bridge_ptr == (void *)bridge)
			return vlan;
	}
	return empty_vlan;
}

static void rt3050_port_bridge_vlan_flush(struct rt3050_priv *priv, int port)
{
	struct rt3050_br_port_vlan_entry *entry, *tmp;

	list_for_each_entry_safe(entry, tmp, &priv->ports[port].br_vlan_list, list) {
		list_del(&entry->list);
		kfree(entry);
	}
}

static int rt3050_port_vlan_add(struct dsa_switch *ds, int port,
		     const struct switchdev_obj_port_vlan *vlan,
		     struct netlink_ext_ack *extack)
{
	struct rt3050_priv *priv = ds->priv;
	struct dsa_port *dp = dsa_to_port(ds, port);
	int ivlan, ret = -EINVAL;
	struct rt3050_br_port_vlan_entry *entry;

	if (port == priv->cpu_port)
		return 0;

	mutex_lock(&priv->mtx);
	ivlan = rt3050_get_vlan_from_bridge(priv, dp->bridge_dev);
	if (!ivlan) {
		netdev_err(dp->master, "(%s). Attempt to assign VLAN %d to non bridged port %d.\n", __func__, vlan->vid, port);
		goto out;
	}
	entry = rt3050_lookup_port_bridge_vlan(&priv->ports[port], vlan->vid);
	if (!entry) {
		if ((entry = kmalloc(sizeof(*entry), GFP_ATOMIC))) {
			entry->pvid = vlan->vid;
			INIT_LIST_HEAD(&entry->list);
			list_add(&entry->list, &priv->ports[port].br_vlan_list);
		} else {
			netdev_err(dp->master, "(%s). No memory to assign VLAN %d to port %d\n", __func__, vlan->vid, port);
			goto out;
		}
	}
	ret = 0;
out:
	mutex_unlock(&priv->mtx);
	return ret;
}

static int rt3050_port_vlan_del(struct dsa_switch *ds, int port,
		     const struct switchdev_obj_port_vlan *vlan)
{
	struct rt3050_priv *priv = ds->priv;
	struct dsa_port *dp = dsa_to_port(ds, port);
	int ivlan, ret = -EINVAL;
	struct rt3050_br_port_vlan_entry *entry;

	if (port == priv->cpu_port)
		return 0;

	mutex_lock(&priv->mtx);
	ivlan = rt3050_get_vlan_from_bridge(priv, dp->bridge_dev);
	if (!ivlan) {
		netdev_err(dp->master, "(%s). Attempt to remove VLAN %d from non bridged port %d.\n", __func__, vlan->vid, port);
		goto out;
	}
	entry = rt3050_lookup_port_bridge_vlan(&priv->ports[port], vlan->vid);
	if (entry) {
		list_del(&entry->list);
		kfree(entry);
	}
	ret = 0;
out:
	mutex_unlock(&priv->mtx);
	return 0;
}

/* Because ds->ops->port_enable is calling before ds->ops->setup, we must have switch basically initiaited at _probe call */
static int rt3050_port_enable(struct dsa_switch *ds, int port, struct phy_device *phy_dev)
{
	u32 disableMask, val = BIT(port);
	struct rt3050_priv *priv = ds->priv;

	mutex_lock(&priv->mtx);
	disableMask = rt3050_get_port_disable(priv);
	if (disableMask & val) {
		disableMask &= ~val;
		rt3050_set_port_disable(priv, disableMask);
	}
	priv->ports[port].disabled = false;
	mutex_unlock(&priv->mtx);
	return 0;
}

/* Because ds->ops->port_disable is calling before ds->ops->setup, we must have switch basically initiaited at _probe call */
static void rt3050_port_disable(struct dsa_switch *ds, int port)
{
	u32 disableMask, val = BIT(port);
	struct rt3050_priv *priv = ds->priv;

	mutex_lock(&priv->mtx);
	disableMask = rt3050_get_port_disable(priv);
	if (!(disableMask & val)) {
		disableMask |= val;
		rt3050_set_port_disable(priv, disableMask);
	}
	priv->ports[port].disabled = true;
	mutex_unlock(&priv->mtx);
}

static enum dsa_tag_protocol rt3050_get_tag_protocol(struct dsa_switch *ds, int port, enum dsa_tag_protocol mprot)
{
	return DSA_TAG_PROTO_RT3050_8021Q;
}

static int rt3050_change_mtu(struct dsa_switch *ds, int port, int new_mtu)
{
	return 0;
}

static int rt3050_port_max_mtu(struct dsa_switch *ds, int port)
{
	struct rt3050_priv *priv = ds->priv;
	u32 val;

	mutex_lock(&priv->mtx);
	val = rt3050_r32(priv, RT3050_REG_SGC);
	val >>= RT3050_SGC_PKT_MAX_LEN_S;
	val &= RT3050_SGC_PKT_MAX_LEN_M;
	mutex_unlock(&priv->mtx);
	if (dsa_is_cpu_port(ds, port)) {
		switch(val) {
			case 0:
				return 1518;
			break;
			case 2:
				return 1508;
			break;
			default:
				return 1504;
			break;
		}
	} else {
		switch(val) {
			case 0:
				return 1514;
			break;
			case 2:
				return 1504;
			break;
			default:
				return 1500;
			break;
		}
	}
	return 1500;
}

static void rt3050_phylink_mac_config(struct dsa_switch *ds, int port, unsigned int mode,
			  const struct phylink_link_state *state)
{
	struct rt3050_priv *priv = ds->priv;
	u32 fpa;

	if (port == priv->cpu_port) {
		if (state->interface != PHY_INTERFACE_MODE_RGMII)
			return;

		mutex_lock(&priv->mtx);
		rt3050_cpu_port_config(priv, true);
		mutex_unlock(&priv->mtx);
	} else if (port != 5) {
		if (state->interface != PHY_INTERFACE_MODE_INTERNAL)
			return;

		mutex_lock(&priv->mtx);
		fpa = rt3050_r32(priv, RT3050_REG_FPA);
		if (state->duplex == DUPLEX_FULL) {
			fpa |= BIT(RT3050_FPA_FORCE_DPX_S + port);
		} else
			fpa &= ~(BIT(RT3050_FPA_FORCE_DPX_S + port));
		if ((mode == MLO_AN_PHY) || state->an_enabled) {
			fpa &= ~(BIT(RT3050_FPA_FORCE_MODE_S + port) | 
			    BIT(RT3050_FPA_FORCE_XFC_S + port) | 
			    BIT(RT3050_FPA_FORCE_SPD_S + port) | 
			    BIT(RT3050_FPA_FORCE_LINK_S + port));
		} else {
			fpa |= BIT(RT3050_FPA_FORCE_MODE_S + port);
			if (state->link)
				fpa |= BIT(RT3050_FPA_FORCE_LINK_S + port);
			else
				fpa &= ~(BIT(RT3050_FPA_FORCE_LINK_S + port));
			if (state->speed == SPEED_100) {
				fpa |= BIT(RT3050_FPA_FORCE_SPD_S + port);
			} else
				fpa &= ~(BIT(RT3050_FPA_FORCE_SPD_S + port));
			if (state->pause != MLO_PAUSE_NONE)
				fpa |= BIT(RT3050_FPA_FORCE_XFC_S + port);
			else
				fpa &= ~(BIT(RT3050_FPA_FORCE_XFC_S + port));
		}
		rt3050_w32(priv, fpa, RT3050_REG_FPA);
		mutex_unlock(&priv->mtx);
	}
}

static void rt3050_phylink_mac_link_down(struct dsa_switch *ds, int port,
					 unsigned int mode,
					 phy_interface_t interface)
{
	struct rt3050_priv *priv = ds->priv;

	if (port == priv->cpu_port) {
		if (interface != PHY_INTERFACE_MODE_RGMII)
			return;

		mutex_lock(&priv->mtx);
		rt3050_cpu_port_config(priv, false);
		mutex_unlock(&priv->mtx);
	} else if (port != RT3050_PORT5) {
		if (interface != PHY_INTERFACE_MODE_INTERNAL)
			return;

		mutex_lock(&priv->mtx);
		rt3050_rmw(priv, RT3050_REG_FPA,
			BIT(RT3050_FPA_FORCE_LINK_S + port),
			BIT(RT3050_FPA_FORCE_MODE_S + port));
		mutex_unlock(&priv->mtx);
	}
}

static void rt3050_phylink_mac_link_up(struct dsa_switch *ds, int port,
				       unsigned int mode,
				       phy_interface_t interface,
				       struct phy_device *phydev,
				       int speed, int duplex,
				       bool tx_pause, bool rx_pause)
{
	struct rt3050_priv *priv = ds->priv;
	u32 fpa;

	if (port == priv->cpu_port) {
		if (interface != PHY_INTERFACE_MODE_RGMII)
			return;

		mutex_lock(&priv->mtx);
		rt3050_cpu_port_config(priv, true);
		mutex_unlock(&priv->mtx);
	} else if (port != RT3050_PORT5) {
		if (interface != PHY_INTERFACE_MODE_INTERNAL)
			return;

		mutex_lock(&priv->mtx);
		fpa = rt3050_r32(priv, RT3050_REG_FPA);
		if (duplex == DUPLEX_FULL) {
			fpa |= BIT(RT3050_FPA_FORCE_DPX_S + port);
		} else {
			fpa &= ~(BIT(RT3050_FPA_FORCE_DPX_S + port));
		}
		if ((mode == MLO_AN_PHY) && phydev) {
			fpa &= ~(BIT(RT3050_FPA_FORCE_MODE_S + port) | 
			    BIT(RT3050_FPA_FORCE_XFC_S + port) | 
			    BIT(RT3050_FPA_FORCE_SPD_S + port) | 
			    BIT(RT3050_FPA_FORCE_LINK_S + port));
		} else {
			fpa |= BIT(RT3050_FPA_FORCE_MODE_S + port) | BIT(RT3050_FPA_FORCE_LINK_S + port);
			if (speed == SPEED_100) {
				fpa |= BIT(RT3050_FPA_FORCE_SPD_S + port);
			} else
				fpa &= ~(BIT(RT3050_FPA_FORCE_SPD_S + port));
			if (tx_pause || rx_pause)
				fpa |= BIT(RT3050_FPA_FORCE_XFC_S + port);
			else
				fpa &= ~(BIT(RT3050_FPA_FORCE_XFC_S + port));
		}
		rt3050_w32(priv, fpa, RT3050_REG_FPA);
		mutex_unlock(&priv->mtx);
	}
}

static void rt3050_phylink_get_caps(struct dsa_switch *ds, int port,
				    struct phylink_config *config)
{
	/* This switch only supports full-duplex at 1Gbps */
	config->mac_capabilities = MAC_ASYM_PAUSE | MAC_SYM_PAUSE |
				   MAC_10 | MAC_100 | MAC_1000FD;

	switch (port) {
	case 0 ... 4: /* Internal phy */
		__set_bit(PHY_INTERFACE_MODE_INTERNAL,
			  config->supported_interfaces);
		break;

	case RT3050_PORT5: /* external phy */
		__set_bit(PHY_INTERFACE_MODE_MII,
			  config->supported_interfaces);
		break;

	case RT3050_PORT6: /* 1st cpu port */
		__set_bit(PHY_INTERFACE_MODE_RGMII,
			  config->supported_interfaces);
		break;
	}
}

static int rt3050_port_fdb_add(struct dsa_switch *ds, int port,
	const unsigned char *addr, u16 vid)
{
	struct rt3050_priv *priv = ds->priv;
	int ret;

	mutex_lock(&priv->mtx);
	ret = rt3050_port_xdb_op(priv, port, addr, vid, true, false);
	mutex_unlock(&priv->mtx);
	return ret;
}

static int rt3050_port_fdb_del(struct dsa_switch *ds, int port,
	const unsigned char *addr, u16 vid)
{
	struct rt3050_priv *priv = ds->priv;

	mutex_lock(&priv->mtx);
	rt3050_port_xdb_op(priv, port, addr, vid, false, false);
	mutex_unlock(&priv->mtx);
	return 0;
}

int rt3050_port_fdb_dump(struct dsa_switch *ds, int port,
	 dsa_fdb_dump_cb_t *cb, void *data)
{
	u32 ats_entry;
	struct rt3050_priv *priv = ds->priv;
	bool first_time = true;

	mutex_lock(&priv->mtx);
	while (1) {
		unsigned long t_start = jiffies;

		while (1) {
			ats_entry = rt3050_r32(priv, RT3050_REG_ATS);
			if (ats_entry & RT3050_ATS_AT_LKUP_IDLE) {
				break;
			}
			if (time_after(jiffies, t_start + RT3050_ATS_TIMEOUT)) {
				dev_err(priv->dev, "(%s) ATS IDLE timeout\n", __func__);
				mutex_unlock(&priv->mtx);
				return -EINVAL;
			}
		}
		if (first_time) {
			first_time = false;
			rt3050_w32(priv, BIT(0), RT3050_REG_ATS);
		} else
			rt3050_w32(priv, BIT(1), RT3050_REG_ATS);
		t_start = jiffies;
		while (1) {
			ats_entry = rt3050_r32(priv, RT3050_REG_ATS0);
			if ((ats_entry & (RT3050_ATS0_SEARCH_RDY_M << RT3050_ATS0_SEARCH_RDY_S)) || 
			    ((ats_entry >> RT3050_ATS0_AT_TABLE_END_S) & RT3050_ATS0_AT_TABLE_END_M)) {
				break;
			}
			if (time_after(jiffies, t_start + RT3050_ATS_TIMEOUT)) {
				dev_err(priv->dev, "(%s) ATS0 result timeout\n", __func__);
				mutex_unlock(&priv->mtx);
				return -EINVAL;
			}
		}
		if (ats_entry & (RT3050_ATS0_SEARCH_RDY_M << RT3050_ATS0_SEARCH_RDY_S)) {
			u8 mac[6];
			u32 mac_raw, ivlan, port_map, age;

			mac_raw = rt3050_r32(priv, RT3050_REG_ATS1);
			mac[5] = mac_raw & 0xff;
			mac[4] = (mac_raw >> 8) & 0xff;
			mac_raw = rt3050_r32(priv, RT3050_REG_ATS2);
			mac[3] = mac_raw & 0xff;
			mac[2] = (mac_raw >> 8) & 0xff;
			mac[1] = (mac_raw >> 16) & 0xff;
			mac[0] = (mac_raw >> 24) & 0xff;
			ivlan = (ats_entry >> RT3050_ATS0_R_VLD_S) & RT3050_ATS0_R_VLD_M;
			port_map = (ats_entry >> RT3050_ATS0_R_PORT_MAP_S) & RT3050_ATS0_R_PORT_MAP_M;
			age = (ats_entry >> RT3050_ATS0_R_AGE_FIELD_S) & RT3050_ATS0_R_AGE_FIELD_M;

			if (port_map & BIT(port)) {
				if (cb && (cb(mac, (u16)ivlan, !!(age == 7), data) < 0))
					break;
			}
		}
		if ((ats_entry >> RT3050_ATS0_AT_TABLE_END_S) & RT3050_ATS0_AT_TABLE_END_M)
			break;
	}
	mutex_unlock(&priv->mtx);
	return 0;
}

static int rt3050_port_mdb_add(struct dsa_switch *ds, int port,
	const struct switchdev_obj_port_mdb *mdb)
{
	struct rt3050_priv *priv = ds->priv;
	int ret;

	mutex_lock(&priv->mtx);
	ret = rt3050_port_xdb_op(priv, port, mdb->addr, mdb->vid, true, true);
	mutex_unlock(&priv->mtx);
	return ret;
}

static int rt3050_port_mdb_del(struct dsa_switch *ds, int port,
	const struct switchdev_obj_port_mdb *mdb)
{
	struct rt3050_priv *priv = ds->priv;

	mutex_lock(&priv->mtx);
	rt3050_port_xdb_op(priv, port, mdb->addr, mdb->vid, false, true);
	mutex_unlock(&priv->mtx);
	return 0;
}

static const struct rt3050_stat_entity rt3050_stats[RT3050_COUNTERS] = {
	{.name = "bad_pkt_cnt_rx", .reg = 0xe8, .up = true},
	{.name = "good_pkt_cnt_rx", .reg = 0xe8, .up = false},
	{.name = "bad_pkt_cnt_tx", .reg = 0x150, .up = true},
	{.name = "good_pkt_cnt_tx", .reg = 0x150, .up = false},
	{.name = "switch_to_frame_engine", .reg = 0xe0, .up = true},
	{.name = "frame_engine_to_switch", .reg = 0xe0, .up = false},
};

static void rt3050_get_strings(struct dsa_switch *ds, int port, u32 stringset, u8 *data)
{
	int i;

	if (stringset != ETH_SS_STATS)
		return;

	for (i = 0; i < ARRAY_SIZE(rt3050_stats); i++)
		strncpy(data + i * ETH_GSTRING_LEN, rt3050_stats[i].name,
			ETH_GSTRING_LEN);
}

static void rt3050_get_ethtool_stats(struct dsa_switch *ds, int port, uint64_t *data)
{
	struct rt3050_priv *priv = ds->priv;
	u32 val;

	mutex_lock(&priv->mtx);
	for (int i = 0; i < ARRAY_SIZE(rt3050_stats); i ++) {
		val = rt3050_r32(priv, RT3050_REG_COUNTER_PER_PORT(rt3050_stats[i].reg, port));
		if (rt3050_stats[i].up)
			data[i] = (val >> RT3050_COUNTER_S) & RT3050_COUNTER_M;
		else
			data[i] = val & RT3050_COUNTER_M;
	}
	mutex_unlock(&priv->mtx);
}

static int rt3050_get_sset_count(struct dsa_switch *ds, int port, int sset)
{
	if (sset != ETH_SS_STATS)
		return 0;

	return ARRAY_SIZE(rt3050_stats);
}

static int rt3050_port_pre_bridge_flags(struct dsa_switch *ds, int port,
			     struct switchdev_brport_flags flags,
			     struct netlink_ext_ack *extack)
{
	if (flags.mask & ~(BR_LEARNING | BR_FLOOD | BR_MCAST_FLOOD |
			   BR_BCAST_FLOOD))
		return -EINVAL;

	return 0;
}

static int rt3050_port_bridge_flags(struct dsa_switch *ds, int port,
			 struct switchdev_brport_flags flags,
			 struct netlink_ext_ack *extack)
{
	struct rt3050_priv *priv = ds->priv;

	mutex_lock(&priv->mtx);
	if (flags.mask & BR_LEARNING)
		rt3050_rmw(priv, RT3050_REG_POC1, RT3050_STP_STATE_BLOCK(port) | 
			RT3050_STP_STATE_LEARN(port) | RT3050_STP_SECURED_PORT(port),
			flags.val & BR_LEARNING ? RT3050_STP_STATE_BLOCK(port) | 
			(!list_empty(&priv->ports[port].sa_secured_mac_addr_list) ? RT3050_STP_SECURED_PORT(port) | RT3050_STP_DIS_LEARN(port) : 0) : 
			RT3050_STP_STATE_BLOCK(port) | RT3050_STP_STATE_LEARN(port) | 
			RT3050_STP_SECURED_PORT(port));

	if (flags.mask & BR_FLOOD)
		rt3050_rmw(priv, RT3050_REG_SOCPC, RT3050_SOCPC_DISUN2CPU(port),
			   flags.val & BR_FLOOD ? 0 : RT3050_SOCPC_DISUN2CPU(port));

	if (flags.mask & BR_MCAST_FLOOD)
		rt3050_rmw(priv, RT3050_REG_SOCPC, RT3050_SOCPC_DISMC2CPU(port),
			   flags.val & BR_MCAST_FLOOD ? 0 : RT3050_SOCPC_DISMC2CPU(port));

	if (flags.mask & BR_BCAST_FLOOD)
		rt3050_rmw(priv, RT3050_REG_SOCPC, RT3050_SOCPC_DISBC2CPU(port),
			   flags.val & BR_BCAST_FLOOD ? 0 : RT3050_SOCPC_DISBC2CPU(port));
	mutex_unlock(&priv->mtx);
	return 0;
}

static int rt3050_port_bridge_join(struct dsa_switch *ds, int port,
			struct net_device *bridge)
{
	struct rt3050_sa_secured_mac_addr_entry *entry;
	struct rt3050_priv *priv = ds->priv;
	int vlan;

	mutex_lock(&priv->mtx);
	vlan = rt3050_get_vlan_from_bridge(priv, bridge);
	if (!vlan) {
		/* No VLAN to use for bridging */
		mutex_unlock(&priv->mtx);
		return -EINVAL;
	}
	if (priv->vlans[vlan].bridge_ptr != (void *)bridge) {
		rt3050_set_vlan_id(priv, vlan, vlan);
		priv->vlans[vlan].vid = (u16)vlan;
		priv->vlans[vlan].bridge_ptr = (void*)bridge;
		rt3050_add_vlan_to_port(priv, (u16)vlan, priv->cpu_port);
	}
	rt3050_add_vlan_to_port(priv, (u16)vlan, port);
	if (priv->do_bridge_offloading && !priv->ports[port].filtering) /* Software bridging */
		rt3050_set_pvid(priv, port, (u16)vlan);

	list_for_each_entry(entry, &priv->ports[port].sa_secured_mac_addr_list, list)
		rt3050_port_xdb_op(priv, port, entry->mac, vlan, true, is_multicast_ether_addr(entry->mac));

	mutex_unlock(&priv->mtx);
	return 0;
}

static void rt3050_port_bridge_leave(struct dsa_switch *ds, int port,
			 struct net_device *bridge)
{
	struct rt3050_priv *priv = ds->priv;
	int vlan;
	struct rt3050_sa_secured_mac_addr_entry *entry;

	mutex_lock(&priv->mtx);
	vlan = rt3050_get_vlan_from_bridge(priv, bridge);
	if (priv->vlans[vlan].bridge_ptr != (void *)bridge) {
		mutex_unlock(&priv->mtx);
		return;
	}
	rt3050_del_vlan_from_port(priv, (u16)vlan, port);
	if (rt3050_get_vmsc(priv, vlan) == BIT(priv->cpu_port))
		rt3050_del_vlan_from_port(priv, (u16)vlan, priv->cpu_port);
	if (!priv->vlans[vlan].ports)
		priv->vlans[vlan].bridge_ptr = NULL;
	rt3050_port_bridge_vlan_flush(priv, port);
	priv->ports[port].filtering = false;
	rt3050_set_pvid(priv, port, dsa_8021q_rt3050_port_to_vid(port));

	list_for_each_entry(entry, &priv->ports[port].sa_secured_mac_addr_list, list)
		rt3050_port_xdb_op(priv, port, entry->mac, vlan, false, is_multicast_ether_addr(entry->mac));

	mutex_unlock(&priv->mtx);
}

static void rt3050_stp_state_set(struct dsa_switch *ds, int port, u8 state)
{
	struct rt3050_priv *priv = ds->priv;

	mutex_lock(&priv->mtx);
	priv->ports[port].forwarding = false;
	switch (state) {
		case BR_STATE_DISABLED:
		case BR_STATE_BLOCKING:
			rt3050_rmw(priv, RT3050_REG_POC1, RT3050_STP_STATE_BLOCK(port) | RT3050_STP_STATE_LEARN(port) | RT3050_STP_SECURED_PORT(port),
				RT3050_STP_STATE_BLOCK(port) | RT3050_STP_STATE_LEARN(port) | RT3050_STP_SECURED_PORT(port));
			break;
		case BR_STATE_LISTENING:
		case BR_STATE_LEARNING:
			rt3050_rmw(priv, RT3050_REG_POC1, RT3050_STP_STATE_BLOCK(port) | RT3050_STP_STATE_LEARN(port) | RT3050_STP_SECURED_PORT(port),
				RT3050_STP_STATE_BLOCK(port) | (!list_empty(&priv->ports[port].sa_secured_mac_addr_list) ? RT3050_STP_SECURED_PORT(port) | RT3050_STP_DIS_LEARN(port) : 0));
			break;
		case BR_STATE_FORWARDING:
		default:
			priv->ports[port].forwarding = true;
			rt3050_rmw(priv, RT3050_REG_POC1, RT3050_STP_STATE_BLOCK(port) | RT3050_STP_STATE_LEARN(port) | RT3050_STP_SECURED_PORT(port),
				!list_empty(&priv->ports[port].sa_secured_mac_addr_list) ? RT3050_STP_SECURED_PORT(port) | RT3050_STP_DIS_LEARN(port) : 0);
			break;
	}
	mutex_unlock(&priv->mtx);
}

static struct rt3050_pcs *pcs_to_rt3050_pcs(struct phylink_pcs *pcs)
{
	return container_of(pcs, struct rt3050_pcs, pcs);
}

static int rt3050_pcs_validate(struct phylink_pcs *pcs,
			       unsigned long *supported,
			       const struct phylink_link_state *state)
{
	struct rt3050_priv *priv = pcs_to_rt3050_pcs(pcs)->priv;
	int port = pcs_to_rt3050_pcs(pcs)->port;
	__ETHTOOL_DECLARE_LINK_MODE_MASK(mask) = { 0, };

	switch (port) {
	case RT3050_PORT6:
		if (state->interface != PHY_INTERFACE_MODE_RGMII)
			goto unsupported;

		phylink_set(mask, 1000baseT_Full);
		phylink_clear(supported, Autoneg);
		break;
	case 0 ... 4:
		if (state->interface != PHY_INTERFACE_MODE_INTERNAL)
			goto unsupported;
		phylink_set(mask, Autoneg);
		break;
	default:
		linkmode_zero(supported);
		dev_err(priv->dev, "(%s) Unsupported port: %i\n", __func__, port);
		return -EINVAL;
	}

	phylink_set_port_modes(mask);
	phylink_set(mask, Pause);
	phylink_set(mask, Asym_Pause);
	phylink_set(mask, 10baseT_Half);
	phylink_set(mask, 10baseT_Full);
	phylink_set(mask, 100baseT_Half);
	phylink_set(mask, 100baseT_Full);

	linkmode_and(supported, supported, mask);

	return 0;

unsupported:
	linkmode_zero(supported);
	dev_err(priv->dev, "Unsupported interface: %d, port: %d\n",
		state->interface, port);
	return -EINVAL;
}

static void rt3050_pcs_get_state(struct phylink_pcs *pcs,
				 struct phylink_link_state *state)
{
	struct rt3050_priv *priv = pcs_to_rt3050_pcs(pcs)->priv;
	int port = pcs_to_rt3050_pcs(pcs)->port;

	if (port == RT3050_PORT5)
		return;

	mutex_lock(&priv->mtx);
	rt3050_get_phylink_port_state(priv, port, state);
	mutex_unlock(&priv->mtx);
}

static int rt3050_pcs_config(struct phylink_pcs *pcs, unsigned int mode,
			     phy_interface_t interface,
			     const unsigned long *advertising,
			     bool permit_pause_to_mac)
{
	return 0;
}

static void rt3050_pcs_an_restart(struct phylink_pcs *pcs)
{
}

static const struct phylink_pcs_ops rt3050_pcs_ops = {
	.pcs_validate = rt3050_pcs_validate,
	.pcs_get_state = rt3050_pcs_get_state,
	.pcs_config = rt3050_pcs_config,
	.pcs_an_restart = rt3050_pcs_an_restart,
};

static struct phylink_pcs *
rt3050_phylink_mac_select_pcs(struct dsa_switch *ds, int port,
			      phy_interface_t interface)
{
	struct rt3050_priv *priv = ds->priv;

	switch (interface) {
	case PHY_INTERFACE_MODE_INTERNAL:
	case PHY_INTERFACE_MODE_RGMII:
		return &priv->pcs[port].pcs;
	default:
		return NULL;
	}
}

static int rt3050_setup(struct dsa_switch *ds)
{
	struct rt3050_priv *priv = ds->priv;
	int i, err;
	u32 queue_phy_ports[] = {0, 1, 2, 3, 4, 5};
	u32 queue_cpu_ports[] = {6};

	if (!priv->global_vlan_enable)
		rt3050_set_vlan_id(priv, 0, 0);
	for (i = 0; i < ds->num_ports; i++) {
		if (i == RT3050_PORT5)
			continue;
		if (priv->global_vlan_enable) {
			if (dsa_is_user_port(ds, i)) {
				rt3050_set_port_untagged(priv, i);
				rt3050_set_vlan_id(priv, dsa_8021q_vid_to_rt3050_vlan(dsa_8021q_rt3050_port_to_vid(i)), dsa_8021q_rt3050_port_to_vid(i));
				rt3050_add_vlan_to_port(priv, dsa_8021q_vid_to_rt3050_vlan(dsa_8021q_rt3050_port_to_vid(i)), i);
				rt3050_add_vlan_to_port(priv, dsa_8021q_vid_to_rt3050_vlan(dsa_8021q_rt3050_port_to_vid(i)), priv->cpu_port);
				rt3050_set_pvid(priv, i, dsa_8021q_rt3050_port_to_vid(i));
			} else if (dsa_is_cpu_port(ds, i)) {
				rt3050_set_port_tagged(priv, i);
				rt3050_set_pvid(priv, i, 0);
			}
		} else {
			rt3050_set_port_untagged(priv, i);
			rt3050_add_vlan_to_port(priv, 0, i);
			rt3050_set_pvid(priv, i, 0);
		}
		if (dsa_is_user_port(ds, i)) {
			rt3050_set_port_led_state(priv, i, RT3050_LED_LINKACT);
			rt3050_port_disable(ds, i);
		}
	}

	rt3050_rmw(priv, RT3050_REG_SGC,
		 RT3050_SGC_PKT_MAX_LEN_M << RT3050_SGC_PKT_MAX_LEN_S, 
		 (0x1 & RT3050_SGC_PKT_MAX_LEN_M) << RT3050_SGC_PKT_MAX_LEN_S);

	rt3050_set_queue_ports(priv, RT3050_QUEUE_BACKGROUND, queue_phy_ports, sizeof(queue_phy_ports) / sizeof(u32));
	rt3050_set_queue_ports(priv, RT3050_QUEUE_BESTEFFORT, queue_cpu_ports, sizeof(queue_cpu_ports) / sizeof(u32));
	rt3050_set_bc_storm_control(priv, 0);

	err = rt3050_setup_irq(priv);
	if (err)
		return err;
	rtnl_lock();
	err = dsa_tag_8021q_register(ds, htons(ETH_P_8021AD));
	rtnl_unlock();
	if (err) {
		rt3050_free_irq(priv);
		return err;
	}
#ifdef CONFIG_NET_DSA_RT3050_DEBUGFS
	rt3050_dbgfs_init(priv);
#endif
#if CONFIG_PROC_FS
	if (rt3050_switch_create_procfs(priv))
		dev_err(priv->dev, "(%s) Error in procfs!\n", __func__);
#endif
	rt3050_enable_irq(priv);

	return 0;
}

static int rt3050_port_vlan_filtering(struct dsa_switch *ds, int port, bool vlan_filtering,
			   struct netlink_ext_ack *extack)
{
	struct rt3050_priv *priv = ds->priv;

	priv->ports[port].filtering = vlan_filtering;
	if (vlan_filtering)
		rt3050_set_pvid(priv, port, dsa_8021q_rt3050_port_to_vid(port));
	return 0;
}

const struct dsa_switch_ops rt3050_switch_ops = {
	.get_tag_protocol	= rt3050_get_tag_protocol,
	.setup			= rt3050_setup,
	.phy_read		= rt3050_phy_read,
	.phy_write		= rt3050_phy_write,

	.port_enable		= rt3050_port_enable,
	.port_disable		= rt3050_port_disable,
	.port_vlan_filtering	= rt3050_port_vlan_filtering,
	.port_vlan_add		= rt3050_port_vlan_add,
	.port_vlan_del		= rt3050_port_vlan_del,
	.port_max_mtu		= rt3050_port_max_mtu,
	.port_change_mtu	= rt3050_change_mtu,

	.phylink_get_caps	= rt3050_phylink_get_caps,
	.phylink_mac_config	= rt3050_phylink_mac_config,
	.phylink_mac_link_down	= rt3050_phylink_mac_link_down,
	.phylink_mac_link_up	= rt3050_phylink_mac_link_up,
	.phylink_mac_select_pcs	= rt3050_phylink_mac_select_pcs,

	.port_fdb_add		= rt3050_port_fdb_add,
	.port_fdb_del		= rt3050_port_fdb_del,
	.port_fdb_dump		= rt3050_port_fdb_dump,
	.port_mdb_add		= rt3050_port_mdb_add,
	.port_mdb_del		= rt3050_port_mdb_del,
	
	.get_strings		= rt3050_get_strings,
	.get_ethtool_stats	= rt3050_get_ethtool_stats,
	.get_sset_count		= rt3050_get_sset_count,

	.port_pre_bridge_flags	= rt3050_port_pre_bridge_flags,
	.port_bridge_flags	= rt3050_port_bridge_flags,
	.port_stp_state_set	= rt3050_stp_state_set,
	.port_bridge_join	= rt3050_port_bridge_join,
	.port_bridge_leave	= rt3050_port_bridge_leave,

};


static int rt3050_probe_common(struct rt3050_priv *priv)
{
	struct device *dev = priv->dev;

	priv->ds = devm_kzalloc(dev, sizeof(*priv->ds), GFP_KERNEL);

	if (!priv->ds)
		return -ENOMEM;

	priv->cpu_port = RT3050_PORT6;
	priv->ds->dev = dev;
	priv->ds->num_ports = RT3050_NUM_PORTS;
	priv->dev = dev;
	priv->ds->priv = priv;
	priv->ds->ops = &rt3050_switch_ops;

	spin_lock_init(&priv->reg_rw_lock);
	mutex_init(&priv->mtx);
	dev_set_drvdata(priv->dev, priv);

	for (int i = 0; i <= priv->cpu_port; i++) {
		priv->pcs[i].priv = priv;
		priv->pcs[i].port = i;
		priv->pcs[i].pcs.ops = &rt3050_pcs_ops;
	}

	rt3050_reset_switch(priv);

	return 0;
}

static int rt3050_probe(struct platform_device *pdev)
{
	struct rt3050_priv *priv;
	int ret, port_num;
	struct device_node *ports, *port;
	u32 value;

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->dev = &pdev->dev;

	priv->rst_esw = devm_reset_control_get(&pdev->dev, "esw");
	if (IS_ERR(priv->rst_esw)) {
		dev_err(&pdev->dev, "Couldn't get 'esw' reset line\n");
		return PTR_ERR(priv->rst_esw);
	}

	priv->rst_ephy = devm_reset_control_get(&pdev->dev, "ephy");

	priv->base = devm_platform_ioremap_resource(pdev, 0);
	if (IS_ERR(priv->base)) {
		dev_err(&pdev->dev, "Cannot request I/O memory space\n");
		return -ENXIO;
	}
	priv->do_bridge_offloading = of_property_read_bool(priv->dev->of_node, "bridge_offloading");
	priv->global_vlan_enable = !of_property_read_bool(priv->dev->of_node, "engineering_mode");

	if (!of_property_read_u32(priv->dev->of_node, "led_polarity", &value))
		priv->reg_led_polarity = value;
	if (!of_property_read_u32(priv->dev->of_node, "led_source", &value))
		priv->reg_led_source = value;
	if (!of_property_read_u32(priv->dev->of_node, "portmap", &value))
		priv->port_map = (u8)value & 0xF;

	memset(priv->ports, 0, sizeof(priv->ports));

	ports = of_get_child_by_name(priv->dev->of_node, "ports");

	for_each_child_of_node(ports, port) {
		if (strcmp(port->name, "port"))
			continue;
		if (of_property_read_u32(port, "reg", &port_num) || port_num >= RT3050_PORT5)
			continue;

		priv->ports[port_num].doubletag = of_property_read_bool(port, "doubletagging");
	}

	of_node_put(ports);


	ret = rt3050_probe_common(priv);
	if (ret) 
		return ret;

	return dsa_register_switch(priv->ds);
}

static int rt3050_remove_common(struct rt3050_priv *priv)
{
	rt3050_free_irq(priv);
#ifdef CONFIG_NET_DSA_RT3050_DEBUGFS
	rt3050_dbgfs_cleanup(priv);
#endif
#if CONFIG_PROC_FS
	if (priv->procentry)
		rt3050_switch_remove_procfs(priv);
#endif

	return 0;
}

static int rt3050_remove(struct platform_device *pdev)
{
	struct rt3050_priv *priv = platform_get_drvdata(pdev);

	if (priv) {
		rt3050_remove_common(priv);
		platform_set_drvdata(pdev, NULL);
	}

	return 0;
}

static void rt3050_shutdown(struct platform_device *pdev)
{
	struct rt3050_priv *priv = platform_get_drvdata(pdev);

	if (!priv)
		return;

	dsa_switch_shutdown(priv->ds);
	dev_set_drvdata(&pdev->dev, NULL);
}

static const struct of_device_id ralink_rt3050_match[] = {
	{ .compatible = "mediatek,rt3050-dsa-switch"},
	{}
};

static struct platform_driver rt3050_driver = {
	.probe = rt3050_probe,
	.remove = rt3050_remove,
	.shutdown = rt3050_shutdown,
	.driver = {
		.name = RT3050_DRIVER_NAME,
		.owner = THIS_MODULE,
		.of_match_table = ralink_rt3050_match,
	},
};

module_platform_driver(rt3050_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("John Crispin <blogic@openwrt.org>, Egor Smirnov <e.smirnov@netshe-lab.ru>, Stanislav Korsakov <sta@stasoft.net>");
MODULE_DESCRIPTION("DSA Switch driver for RT305X/MT76x8 SoCs");
MODULE_VERSION(MTK_FE_DRV_VERSION);
