#include <linux/dsa/tag_rt3050_8021q.h>
#include "dsa_priv.h"
#include "../../drivers/net/dsa/rt3050/rt3050.h"

static int rt3050_mac_to_port(struct rt3050_priv *priv,
	 const u8 src[ETH_ALEN])
{
	int port = -1;
	u32 ats_entry;
	bool first_time = true;

	mutex_lock(&priv->mtx);
	while (1) {
		unsigned long t_start = jiffies;

		while (1) {
			ats_entry = rt3050_r32(priv, RT3050_REG_ATS);
			if (ats_entry & RT3050_ATS_AT_LKUP_IDLE) {
				break;
			}
			if (time_after(jiffies, t_start + RT3050_ATS_TIMEOUT)) {
				dev_err(priv->dev, "(%s) ATS IDLE timeout\n", __func__);
				goto out;
			}
		}
		if (first_time) {
			first_time = false;
			rt3050_w32(priv, BIT(0), RT3050_REG_ATS);
		} else
			rt3050_w32(priv, BIT(1), RT3050_REG_ATS);
		t_start = jiffies;
		while (1) {
			ats_entry = rt3050_r32(priv, RT3050_REG_ATS0);
			if ((ats_entry & (RT3050_ATS0_SEARCH_RDY_M << RT3050_ATS0_SEARCH_RDY_S)) || 
			    ((ats_entry >> RT3050_ATS0_AT_TABLE_END_S) & RT3050_ATS0_AT_TABLE_END_M)) {
				break;
			}
			if (time_after(jiffies, t_start + RT3050_ATS_TIMEOUT)) {
				dev_err(priv->dev, "(%s) ATS0 result timeout\n", __func__);
				goto out;
			}
		}
		if (ats_entry & (RT3050_ATS0_SEARCH_RDY_M << RT3050_ATS0_SEARCH_RDY_S)) {
			u8 mac[ETH_ALEN];
			u32 mac_raw;

			mac_raw = rt3050_r32(priv, RT3050_REG_ATS1);
			mac[5] = mac_raw & 0xff;
			mac[4] = (mac_raw >> 8) & 0xff;
			mac_raw = rt3050_r32(priv, RT3050_REG_ATS2);
			mac[3] = mac_raw & 0xff;
			mac[2] = (mac_raw >> 8) & 0xff;
			mac[1] = (mac_raw >> 16) & 0xff;
			mac[0] = (mac_raw >> 24) & 0xff;
			if (ether_addr_equal(mac, src)) {
				u32 port_map = (ats_entry >> RT3050_ATS0_R_PORT_MAP_S) & RT3050_ATS0_R_PORT_MAP_M;

				for (int i = 0; i < priv->cpu_port; i++) {
					if (port_map & BIT(i)) {
						port = i;
						break;
					}
				}
			}
		}
		if ((ats_entry >> RT3050_ATS0_AT_TABLE_END_S) & RT3050_ATS0_AT_TABLE_END_M)
			break;
	}
out:
	mutex_unlock(&priv->mtx);
	return port;
}

/* TX from pseudo interface / port */
static struct sk_buff *rt3050_xmit(struct sk_buff *skb, struct net_device *netdev)
{
	struct dsa_port *dp = dsa_slave_to_port(netdev);
	u16 tx_vid = dsa_8021q_rt3050_tx_vid(dp);
	u16 queue_mapping = skb_get_queue_mapping(skb);
	u8 pcp = netdev_txq_to_tc(netdev, queue_mapping);

	if (dp->bridge_dev) {
		struct rt3050_priv * priv = dp->ds->priv;

		if (priv->do_bridge_offloading && !priv->ports[dp->index].filtering) {
			u32 mask = BIT(dp->index);

			for (int i = dp->index + 1; i < RT3050_NUM_VLANS; i++) {
				if ((priv->vlans[i].bridge_ptr == dp->bridge_dev) && (priv->vlans[i].ports & mask)) {
					const struct ethhdr *eth = eth_hdr(skb);

					if (!is_unicast_ether_addr(eth->h_dest)) {
						if (!ether_addr_equal(eth->h_source, dp->bridge_dev->dev_addr) && !ether_addr_equal(eth->h_source, netdev->dev_addr)) {
							int src_port = rt3050_mac_to_port(priv, eth->h_source);

							if ((src_port >= 0) && (src_port < priv->cpu_port))
								return NULL;
							if (src_port < 0)
								priv->ds->ops->port_fdb_add(priv->ds, priv->cpu_port, eth->h_source, (u16)i);
						} else {
							for (int j = 0; j < dp->index; j++) {
								if ((priv->vlans[i].ports & BIT(j)) && priv->ports[j].up && priv->ports[j].forwarding) {
									/* Current port is not first active port in this bridge.
									 * No need to TX this skb. */
									return NULL;
								}
							}
						}
					}
					tx_vid = (u16)i;
					break;
				}
			}
		}
	}

	skb->dev = netdev;
	return dsa_8021q_xmit(skb, netdev, ETH_P_8021Q, ((pcp << VLAN_PRIO_SHIFT) | tx_vid));
}

/* RX to pseudo interface / port */
static struct sk_buff *rt3050_rcv(struct sk_buff *skb, struct net_device *netdev)
{
	int src_port, switch_id;

	dsa_8021q_rcv(skb, &src_port, &switch_id);
	if (src_port <= RT3050_PORT5) {
		skb->dev = dsa_master_find_slave(netdev, switch_id, dsa_8021q_port_to_rt3050_port(src_port));
	} else if (src_port < RT3050_NUM_VLANS) {
		struct dsa_port *dp = netdev->dsa_ptr;

		if (!IS_ERR(dp)) {
			struct rt3050_priv * priv = dp->ds->priv;
			const struct ethhdr *eth = eth_hdr(skb);

			if ((src_port = rt3050_mac_to_port(priv, eth->h_source)) >= 0)
				skb->dev = dsa_master_find_slave(netdev, switch_id, src_port);
		}
	}
	if (!skb->dev)
		return NULL;
	return skb;
}


static const struct dsa_device_ops RT3050_8021q_netdev_ops = {
	.name = "rt3050-8021q",
	.proto = DSA_TAG_PROTO_RT3050_8021Q,
	.xmit = rt3050_xmit,
	.rcv = rt3050_rcv,
	.needed_headroom = VLAN_HLEN,
	.promisc_on_master = true,
};

MODULE_ALIAS_DSA_TAG_DRIVER(DSA_TAG_PROTO_RT3050_8021Q);

module_dsa_tag_driver(RT3050_8021q_netdev_ops);
