#include <asm/mach-ralink/ralink_regs.h>
#include <linux/reset.h>
#include "rt3050.h"

static void rt3050_reset(struct rt3050_priv *priv)
{
	if (!priv->rst_esw)
		return;

	reset_control_assert(priv->rst_esw);
	usleep_range(60, 120);
	reset_control_deassert(priv->rst_esw);
	/* the rt3050 takes long to reset otherwise the board hang */
	msleep(10);
}

static void rt3050_reset_ephy(struct rt3050_priv *priv)
{
	if (!priv->rst_ephy)
		return;

	reset_control_assert(priv->rst_ephy);
	usleep_range(60, 120);
	reset_control_deassert(priv->rst_ephy);
	usleep_range(60, 120);
}

void rt3050_rmw(struct rt3050_priv *priv, u32 reg,
		    u32 mask, u32 val)
{
	unsigned long flags;

	spin_lock_irqsave(&priv->reg_rw_lock, flags);
	rt3050_rmw_raw(priv, reg, mask, val);
	spin_unlock_irqrestore(&priv->reg_rw_lock, flags);
}

int rt3050_mii_write(struct rt3050_priv *priv, int phy_addr,
			    int phy_register, u32 write_data)
{
	unsigned long t_start = jiffies;
	int ret = 0;

	while (1) {
		if (!(rt3050_r32(priv, RT3050_REG_PCR1) &
		      RT3050_PCR1_WT_DONE))
			break;
		if (time_after(jiffies, t_start + RT3050_PHY_TIMEOUT)) {
			ret = -EINVAL;
			goto out;
		}
	}

	rt3050_w32(priv, ((write_data & RT3050_PCR0_WT_NWAY_DATA_M) << RT3050_PCR0_WT_NWAY_DATA_S) |
		      ((phy_register & RT3050_PCR0_CPU_PHY_REG_M) << RT3050_PCR0_CPU_PHY_REG_S) |
		      (phy_addr & RT3050_PCR0_PHY_ADDR_M) | RT3050_PCR0_WT_PHY_CMD,
		RT3050_REG_PCR0);

	t_start = jiffies;
	while (1) {
		if (rt3050_r32(priv, RT3050_REG_PCR1) &
			    RT3050_PCR1_WT_DONE)
			break;

		if (time_after(jiffies, t_start + RT3050_PHY_TIMEOUT)) {
			ret = -EINVAL;
			break;
		}
	}
out:
	if (ret)
		dev_err(priv->dev, "%s: MDIO timeout\n", __func__);
	return ret;
}

int rt3050_mii_read(struct rt3050_priv *priv, int phy_addr,
			    int phy_register)
{
	unsigned long t_start = jiffies;
	int ret = 0;

	while (1) {
		if (!(rt3050_r32(priv, RT3050_REG_PCR1) &
		      RT3050_PCR1_RD_READY))
			break;
		if (time_after(jiffies, t_start + RT3050_PHY_TIMEOUT)) {
			ret = -EINVAL;
			goto out;
		}
	}

	rt3050_w32(priv, ((phy_register & RT3050_PCR0_CPU_PHY_REG_M) << RT3050_PCR0_CPU_PHY_REG_S) |
		      (phy_addr & RT3050_PCR0_PHY_ADDR_M) | RT3050_PCR0_RD_PHY_CMD,
		RT3050_REG_PCR0);

	t_start = jiffies;
	while (1) {
		u32 val = rt3050_r32(priv, RT3050_REG_PCR1);

		if (val & RT3050_PCR1_RD_READY) {
			ret = ((val >> RT3050_PCR0_WT_NWAY_DATA_S) & RT3050_PCR0_WT_NWAY_DATA_M);
			break;
		}

		if (time_after(jiffies, t_start + RT3050_PHY_TIMEOUT)) {
			ret = -EINVAL;
			dev_err(priv->dev, "(%s): MDIO timeout\n", __func__);
			break;
		}
	}
out:
	return ret;
}

u16 rt3050_get_vlan_id(struct rt3050_priv *priv, u16 vlan)
{
	u32 s, val;

	s = RT3050_VLANI_VID_S * (vlan % 2);
	val = rt3050_r32(priv, RT3050_REG_VLANI(vlan >> 1));
	val = (val >> s) & RT3050_VLANI_VID_M;

	return (u16)val;
}

void rt3050_set_vlan_id(struct rt3050_priv *priv, u16 vlan, u16 vid)
{
	u32 s;

	s = RT3050_VLANI_VID_S * (vlan % 2);
	rt3050_rmw(priv,
		       RT3050_REG_VLANI(vlan >> 1),
		       RT3050_VLANI_VID_M << s,
		       (vid & RT3050_VLANI_VID_M) << s);
	priv->vlans[vlan].vid = vid;
}

u32 rt3050_get_untagged_port(struct rt3050_priv *priv)
{
	u32 val;

	val = rt3050_r32(priv, RT3050_REG_POC2);
	val = (val >> RT3050_POC2_UNTAG_EN_S) & RT3050_POC2_UNTAG_EN_M;
	return val;
}

void rt3050_set_port_tagged(struct rt3050_priv *priv, int port)
{
	if (port >= RT3050_NUM_PORTS)
		return;
	rt3050_rmw(priv, RT3050_REG_POC2, BIT(port) << RT3050_POC2_UNTAG_EN_S, 0);
	priv->ports[port].untag = false;
}

void rt3050_set_port_untagged(struct rt3050_priv *priv, int port)
{
	if (port >= RT3050_NUM_PORTS)
		return;
	rt3050_rmw(priv, RT3050_REG_POC2, BIT(port) << RT3050_POC2_UNTAG_EN_S,
		BIT(port) << RT3050_POC2_UNTAG_EN_S);
	priv->ports[port].untag = true;
}

void rt3050_add_vlan_to_port(struct rt3050_priv *priv, u16 vlan, int port)
{
	u32 conf, val = BIT(port);

	conf = rt3050_get_vmsc(priv, vlan);
	conf |= val;
	rt3050_set_vmsc(priv, vlan, conf);
	priv->vlans[vlan].ports |= val;
}

void rt3050_del_vlan_from_port(struct rt3050_priv *priv, u16 vlan, int port)
{
	u32 conf, val = ~BIT(port);

	conf = rt3050_get_vmsc(priv, vlan);
	conf &= val;
	rt3050_set_vmsc(priv, vlan, conf);
	priv->vlans[vlan].ports &= val;
}

u16 rt3050_get_pvid(struct rt3050_priv *priv, int port)
{
	u32 s, val;

	s = RT3050_PVIDC_PVID_S * (port % 2);
	val = rt3050_r32(priv, RT3050_REG_PVIDC(port >> 1));
	return (u16)((val >> s) & RT3050_PVIDC_PVID_M);
}

void rt3050_set_pvid(struct rt3050_priv *priv, int port, u16 pvid)
{
	u32 s;

	s = RT3050_PVIDC_PVID_S * (port % 2);
	rt3050_rmw(priv,
		       RT3050_REG_PVIDC(port / 2),
		       RT3050_PVIDC_PVID_M << s,
		       (u32)(pvid & RT3050_PVIDC_PVID_M) << s);
	priv->ports[port].pvid = pvid;
}

u32 rt3050_get_vmsc(struct rt3050_priv *priv, u16 vlan)
{
	u32 s, val;

	s = RT3050_VMSC_MSC_S * (vlan % 4);
	val = rt3050_r32(priv, RT3050_REG_VMSC(vlan >> 2));
	val = (val >> s) & RT3050_VMSC_MSC_M;

	return val;
}

void rt3050_set_vmsc(struct rt3050_priv *priv, u16 vlan, u32 msc)
{
	u32 s;

	s = RT3050_VMSC_MSC_S * (vlan % 4);
	rt3050_rmw(priv,
		       RT3050_REG_VMSC(vlan >> 2),
		       RT3050_VMSC_MSC_M << s,
		       (msc & RT3050_VMSC_MSC_M) << s);
}

u32 rt3050_get_port_disable(struct rt3050_priv *priv)
{
	u32 reg = rt3050_r32(priv, RT3050_REG_POC0);

	return (reg >> RT3050_POC0_DIS_PORT_S) &
	       RT3050_POC0_DIS_PORT_M;
}

void rt3050_set_port_disable(struct rt3050_priv *priv, u32 disable_mask)
{
	u32 old_mask;
	u32 enable_mask;
	u32 changed;
	int i;

	old_mask = rt3050_get_port_disable(priv);
	changed = old_mask ^ disable_mask;
	enable_mask = old_mask & disable_mask;

	/* enable before writing to MII */
	rt3050_rmw(priv, RT3050_REG_POC0,
		       (RT3050_POC0_DIS_PORT_M <<
			RT3050_POC0_DIS_PORT_S),
		       enable_mask << RT3050_POC0_DIS_PORT_S);

	for (i = 0; i < priv->cpu_port; i++) {
		if (!(changed & (BIT(i))))
			continue;
		if (disable_mask & (BIT(i))) {
			/* disable */
			rt3050_mii_write(priv, i, MII_BMCR,
					 BMCR_PDOWN);
		} else {
			/* enable */
			rt3050_mii_write(priv, i, MII_BMCR, BMCR_FULLDPLX |
					 BMCR_ANENABLE |
					 BMCR_ANRESTART |
					 BMCR_SPEED100);
		}
	}

	/* disable after writing to MII */
	rt3050_rmw(priv, RT3050_REG_POC0,
		       (RT3050_POC0_DIS_PORT_M <<
			RT3050_POC0_DIS_PORT_S),
		       disable_mask << RT3050_POC0_DIS_PORT_S);
}

static inline void rt3050_set_gsc(struct rt3050_priv *priv)
{
	rt3050_rmw(priv, RT3050_REG_SGC,
		RT3050_SGC_BC_STORM_M << RT3050_SGC_BC_STORM_S,
		priv->bc_storm_protect << RT3050_SGC_BC_STORM_S);
	rt3050_rmw(priv, RT3050_REG_SGC,
		RT3050_SGC_LED_FREQ_M << RT3050_SGC_LED_FREQ_S,
		priv->led_frequency << RT3050_SGC_LED_FREQ_S);
}


static int rt3050_apply_config(struct rt3050_priv *priv)
{
	int i;
	u32 disable = 0;
	u32 doubletag = 0;
	u32 en_vlan = 0;
	u32 untag = 0;

	for (i = 0; i < RT3050_NUM_VLANS; i++) {
		u32 vid, vmsc;

		if (priv->global_vlan_enable) {
			vid = priv->vlans[i].vid;
			vmsc = priv->vlans[i].ports;
		} else {
			vid = RT3050_VLAN_NONE;
			vmsc = RT3050_PORTS_NONE;
		}
		rt3050_set_vlan_id(priv, i, vid);
		rt3050_set_vmsc(priv, i, vmsc);
	}

	for (i = 0; i < RT3050_NUM_PORTS; i++) {
		u32 pvid;

		INIT_LIST_HEAD(&priv->ports[i].br_vlan_list);
		INIT_LIST_HEAD(&priv->ports[i].sa_secured_mac_addr_list);

		if (priv->ports[i].disabled)
			disable |= BIT(i);
		if (priv->global_vlan_enable) {
			doubletag |= priv->ports[i].doubletag << i;
			en_vlan   |= BIT(i);
			untag     |= priv->ports[i].untag     << i;
			pvid       = priv->ports[i].pvid;
		} else {
			u32 x = priv->alt_vlan_disable ? 0 : 1;

			doubletag |= x << i;
			en_vlan   |= x << i;
			untag     |= x << i;
			pvid       = 0;
		}
		rt3050_set_pvid(priv, i, pvid);
		if (i < RT3050_NUM_LEDS)
			rt3050_w32(priv, priv->ports[i].led,
				      RT3050_REG_P0LED + 4 * i);
	}

	rt3050_set_gsc(priv);
	rt3050_set_port_disable(priv, disable);
	rt3050_rmw(priv, RT3050_REG_SGC2,
		       (RT3050_SGC2_DOUBLE_TAG_M <<
			RT3050_SGC2_DOUBLE_TAG_S),
		       doubletag << RT3050_SGC2_DOUBLE_TAG_S);
	rt3050_rmw(priv, RT3050_REG_PFC1,
		       RT3050_PFC1_EN_VLAN_M << RT3050_PFC1_EN_VLAN_S,
		       en_vlan << RT3050_PFC1_EN_VLAN_S);
	rt3050_rmw(priv, RT3050_REG_POC2,
		       RT3050_POC2_UNTAG_EN_M << RT3050_POC2_UNTAG_EN_S,
		       untag << RT3050_POC2_UNTAG_EN_S);

	if (!priv->global_vlan_enable) {
		/*
		 * Still need to put all ports into vlan 0 or they'll be
		 * isolated.
		 * NOTE: vlan 0 is special, no vlan tag is prepended
		 */
		rt3050_set_vlan_id(priv, 0, 0);
		rt3050_set_vmsc(priv, 0, RT3050_PORTS_ALL);
	}

	return 0;
}

static void rt3050_hw_init(struct rt3050_priv *priv)
{
	int i;
	u32 port_map = RT3050_PMAP_LLLLLL;

	rt3050_reset(priv);

	/* vodoo from original driver */
	rt3050_w32(priv, 0xC8A07850, RT3050_REG_FCT0);
	rt3050_w32(priv, 0x00000000, RT3050_REG_SGC2);
	/* Port priority 1 for all ports, vlan enabled. */
	rt3050_w32(priv, 0x00005555 |
		     (RT3050_PORTS_ALL << RT3050_PFC1_EN_VLAN_S),
		RT3050_REG_PFC1);

	/* Enable all ports, Back Pressure and Flow Control */
	rt3050_w32(priv, ((RT3050_PORTS_ALL << RT3050_POC0_EN_BP_S) |
		      (RT3050_PORTS_ALL << RT3050_POC0_EN_FC_S)),
		RT3050_REG_POC0);

	/* Enable Aging, and VLAN TAG removal */
	rt3050_w32(priv, ((RT3050_PORTS_ALL << RT3050_POC2_ENAGING_S) |
		      (RT3050_PORTS_NOCPU << RT3050_POC2_UNTAG_EN_S)),
		RT3050_REG_POC2);

	if (priv->reg_initval_fct2)
		rt3050_w32(priv, priv->reg_initval_fct2, RT3050_REG_FCT2);
	else
		rt3050_w32(priv, 0x0002500c, RT3050_REG_FCT2);

	/* 300s aging timer, max packet len 1536, broadcast storm prevention
	 * disabled, disable collision abort, mac xor48 hash, 10 packet back
	 * pressure jam, GMII disable was_transmit, back pressure disabled,
	 * 30ms led flash, unmatched IGMP as broadcast, rmc tb fault to all
	 * ports.
	 */
	rt3050_w32(priv, 0x0008a301, RT3050_REG_SGC);

	/* Setup SoC Port control register */
	rt3050_w32(priv,
		(RT3050_SOCPC_CRC_PADDING |
		(RT3050_PORTS_CPU << RT3050_SOCPC_DISUN2CPU_S) |
		(RT3050_PORTS_CPU << RT3050_SOCPC_DISMC2CPU_S) |
		(RT3050_PORTS_CPU << RT3050_SOCPC_DISBC2CPU_S)),
		RT3050_REG_SOCPC);

	/* ext phy base addr 31, enable port 5 polling, rx/tx clock skew 1,
	 * turbo mii off, rgmi 3.3v off
	 * port5: disabled
	 * port6: enabled, gige, full-duplex, rx/tx-flow-control
	 */
	if (priv->reg_initval_fpa2)
		rt3050_w32(priv, priv->reg_initval_fpa2, RT3050_REG_FPA1);
	else
		rt3050_w32(priv, 0x3f502b28, RT3050_REG_FPA1);
	rt3050_w32(priv, 0x00000000, RT3050_REG_FPA);

	/* Force Link/Activity on ports */
	rt3050_w32(priv, RT3050_LED_LINKACT, RT3050_REG_P0LED);
	rt3050_w32(priv, RT3050_LED_LINKACT, RT3050_REG_P1LED);
	rt3050_w32(priv, RT3050_LED_LINKACT, RT3050_REG_P2LED);
	rt3050_w32(priv, RT3050_LED_LINKACT, RT3050_REG_P3LED);
	rt3050_w32(priv, RT3050_LED_LINKACT, RT3050_REG_P4LED);

	/* Disable nonexistent ports by reading the switch config
	 * after having enabled all possible ports above
	 */
	priv->ports[RT3050_PORT5].disabled = true;

	rt3050_reset_ephy(priv);

	/* set the led polarity and led source */
	rt3050_w32(priv, (priv->reg_led_polarity & 0x1F) |
				((priv->reg_led_source << 8) & 0x700),
				RT5350_EWS_REG_LED_CONTROL);

	rt3050_mii_write(priv, 0, 31, 0x2000); /* change G2 page */
	rt3050_mii_write(priv, 0, 26, 0x0020);

	for (i = 0; i < RT3050_PORT5; i++) {
		rt3050_mii_write(priv, i, 31, 0x8000);
		rt3050_mii_write(priv, i,  0, 0x3100);
		rt3050_mii_write(priv, i, 30, 0xa000);
		rt3050_mii_write(priv, i, 31, 0xa000);
		rt3050_mii_write(priv, i, 16, 0x0606);
		rt3050_mii_write(priv, i, 23, 0x0f0e);
		rt3050_mii_write(priv, i, 24, 0x1610);
		rt3050_mii_write(priv, i, 30, 0x1f15);
		rt3050_mii_write(priv, i, 28, 0x6111);
		rt3050_mii_write(priv, i, 31, 0x2000);
		rt3050_mii_write(priv, i, 26, 0x0000);
	}

	/* 100Base AOI setting */
	rt3050_mii_write(priv, 0, 31, 0x5000);
	rt3050_mii_write(priv, 0, 19, 0x004a);
	rt3050_mii_write(priv, 0, 20, 0x015a);
	rt3050_mii_write(priv, 0, 21, 0x00ee);
	rt3050_mii_write(priv, 0, 22, 0x0033);
	rt3050_mii_write(priv, 0, 23, 0x020a);
	rt3050_mii_write(priv, 0, 24, 0x0000);
	rt3050_mii_write(priv, 0, 25, 0x024a);
	rt3050_mii_write(priv, 0, 26, 0x035a);
	rt3050_mii_write(priv, 0, 27, 0x02ee);
	rt3050_mii_write(priv, 0, 28, 0x0233);
	rt3050_mii_write(priv, 0, 29, 0x000a);
	rt3050_mii_write(priv, 0, 30, 0x0000);

	if (priv->port_map)
		port_map = priv->port_map;
	else
		port_map = RT3050_PMAP_LLLLLL;

	/* Unused HW feature, but still nice to be consistent here...
	 * This is also exported to userspace ('lan' attribute) so it's
	 * conveniently usable to decide which ports go into the wan vlan by
	 * default.
	 */
	rt3050_rmw(priv, RT3050_REG_SGC2,
		RT3050_SGC2_LAN_PMAP_M << RT3050_SGC2_LAN_PMAP_S,
		port_map << RT3050_SGC2_LAN_PMAP_S);

	rt3050_apply_config(priv);
}

int rt3050_reset_switch(struct rt3050_priv *priv)
{
	memset(priv->vlans, 0, sizeof(priv->vlans));
	rt3050_hw_init(priv);

	return 0;
}

static void rt3050_netdev_change_carrier(struct rt3050_priv *priv, int port, bool new_state)
{
	if (priv->ports[port].up != new_state) {
		const struct dsa_port *dp = dsa_to_port(priv->ds, port);

		priv->ports[port].up = new_state;
		if (dp && dp->slave) {
			if (new_state) {
				dev_info(priv->dev, "port %d link up: %s\n", port, dp->slave->name);
				netif_carrier_on(dp->slave);
			} else {
				dev_info(priv->dev, "port %d link down: %s\n", port, dp->slave->name);
				netif_carrier_off(dp->slave);
			}
		}
	}
}

static inline void rt3050_toggle_carrier(struct rt3050_priv *priv)
{
	u32 link;

	link = rt3050_r32(priv, RT3050_REG_POA);
	link >>= RT3050_POA_LINK_S;
	rt3050_netdev_change_carrier(priv, priv->cpu_port, !!(link & RT3050_POA_LINK_CPU_M(priv->cpu_port)));
	for (int i = 0; i < RT3050_NUM_PHY_PORTS; i++)
		rt3050_netdev_change_carrier(priv, i, !!(link & BIT(i)));
}

void rt3050_enable_irq(struct rt3050_priv *priv)
{
	rt3050_w32(priv, RT3050_PORT_ST_CHG, RT3050_REG_ISR);
	/* Only unmask the port change interrupt */
	rt3050_w32(priv, ~RT3050_PORT_ST_CHG, RT3050_REG_IMR);
}

irqreturn_t rt3050_irq_handler(int irq, void *_rt3050)
{
	struct rt3050_priv *priv = (struct rt3050_priv *) _rt3050;
	u32 status;

	status = rt3050_r32(priv, RT3050_REG_ISR);
	rt3050_w32(priv, status, RT3050_REG_ISR);
	if (status & RT3050_PORT_ST_CHG)
		rt3050_toggle_carrier(priv);

	rt3050_enable_irq(priv);

	return IRQ_HANDLED;
}

void rt3050_disable_irq(struct rt3050_priv *priv)
{
	rt3050_w32(priv, ~0, RT3050_REG_IMR);
	rt3050_w32(priv, 0, RT3050_REG_ISR);
}

void rt3050_free_irq(struct rt3050_priv *priv)
{
	rt3050_disable_irq(priv);
	if (priv->irq > 0)
		free_irq(priv->irq, priv);
}

void rt3050_cpu_port_config(struct rt3050_priv *priv, bool up)
{
	u32 fpa;

	if (up) {
		fpa = rt3050_r32(priv, RT3050_REG_FPA1);
		fpa |= RT3050_FPA1_FORCE_RGMII_FORCE_M(priv->cpu_port) << RT3050_FPA1_FORCE_RGMII_FORCE_S;
		fpa |= RT3050_FPA1_FORCE_RGMII_LINK_M(priv->cpu_port) << RT3050_FPA1_FORCE_RGMII_LINK_S;
		fpa |= RT3050_FPA1_FORCE_RGMII_SPD_M(priv->cpu_port) << RT3050_FPA1_FORCE_RGMII_SPD_S;
		fpa |= RT3050_FPA1_FORCE_RGMII_DPX_M(priv->cpu_port) << RT3050_FPA1_FORCE_RGMII_DPX_S;
		rt3050_w32(priv, fpa, RT3050_REG_FPA1);
	} else {
		rt3050_rmw(priv, RT3050_REG_FPA1,
			RT3050_FPA1_FORCE_RGMII_LINK_M(priv->cpu_port) << RT3050_FPA1_FORCE_RGMII_LINK_S,
			RT3050_FPA1_FORCE_RGMII_FORCE_M(priv->cpu_port) << RT3050_FPA1_FORCE_RGMII_FORCE_S);
	}
}

void rt3050_get_phylink_port_state(struct rt3050_priv *priv, int port, struct phylink_link_state *state)
{
	u32 poa, fpa, fpa1;

	poa = rt3050_r32(priv, RT3050_REG_POA);
	fpa = rt3050_r32(priv, RT3050_REG_FPA);
	fpa1 = rt3050_r32(priv, RT3050_REG_FPA1);

	if (port == priv->cpu_port) {
		u32 cpu_speed, cpu_flow_mode;

		if ((fpa1 >> RT3050_FPA1_FORCE_RGMII_FORCE_S) & RT3050_FPA1_FORCE_RGMII_FORCE_M(priv->cpu_port)) {
			cpu_speed = (fpa1 >> RT3050_FPA1_FORCE_RGMII_SPD_S) & RT3050_FPA1_FORCE_RGMII_SPD_M(priv->cpu_port);
			cpu_flow_mode = (fpa1 >> RT3050_FPA1_FORCE_RGMII_XFC_S) & RT3050_FPA1_FORCE_RGMII_XFC_M(priv->cpu_port);

			state->duplex = ((fpa1 >> RT3050_FPA1_FORCE_RGMII_DPX_S) & RT3050_FPA1_FORCE_RGMII_DPX_M(priv->cpu_port)) ? DUPLEX_FULL : DUPLEX_HALF;
			state->pause = (cpu_flow_mode == 3 ? MLO_PAUSE_TXRX_MASK : 
							(cpu_flow_mode == 2 ? MLO_PAUSE_TX : (cpu_flow_mode == 1 ? MLO_PAUSE_RX : MLO_PAUSE_NONE)));
			state->link = !!((fpa1 >> RT3050_FPA1_FORCE_RGMII_LINK_S) & RT3050_FPA1_FORCE_RGMII_LINK_M(priv->cpu_port));
			state->an_enabled = false;
		} else {
			cpu_speed = (poa >> RT3050_POA_SPEED_S) & RT3050_POA_SPEED_CPU_M(priv->cpu_port);
			cpu_flow_mode = (poa >> RT3050_POA_XFC_S) & RT3050_POA_XFC_CPU_M(priv->cpu_port);

			state->duplex = ((poa >> RT3050_POA_DUPLEX_S) & RT3050_POA_DUPLEX_CPU_M(priv->cpu_port)) ? DUPLEX_FULL : DUPLEX_HALF;
			state->link = !!((poa >> RT3050_POA_LINK_S) & RT3050_POA_LINK_CPU_M(priv->cpu_port));
			state->an_enabled = true;
		}
		state->interface = PHY_INTERFACE_MODE_RGMII;
		state->pause = (cpu_flow_mode == 3 ? MLO_PAUSE_TXRX_MASK : 
						(cpu_flow_mode == 2 ? MLO_PAUSE_TX : (cpu_flow_mode == 1 ? MLO_PAUSE_RX : MLO_PAUSE_NONE)));
		state->speed = cpu_speed > 1 ? SPEED_1000 : (cpu_speed == 1 ? SPEED_100 : SPEED_10);
		state->an_complete = state->link;
	} else {
		if ((fpa >> RT3050_FPA_FORCE_MODE_S) & BIT(port)) {
			state->speed = ((fpa >> RT3050_FPA_FORCE_SPD_S) & BIT(port)) ? SPEED_100 : SPEED_10;
			state->duplex = ((fpa >> RT3050_FPA_FORCE_DPX_S) & BIT(port)) ? DUPLEX_FULL : DUPLEX_HALF;
			state->pause = ((fpa >> RT3050_FPA_FORCE_XFC_S) & BIT(port)) ? MLO_PAUSE_TXRX_MASK : MLO_PAUSE_NONE;
			state->link = !!((fpa >> RT3050_FPA_FORCE_LINK_S) & BIT(port));
			state->an_enabled = false;
		} else {
			state->speed = ((poa >> RT3050_POA_SPEED_S) & BIT(port)) ? SPEED_100 : SPEED_10;
			state->duplex = ((poa >> RT3050_POA_DUPLEX_S) & BIT(port)) ? DUPLEX_FULL : DUPLEX_HALF;
			state->pause = ((poa >> RT3050_POA_XFC_S) & BIT(port)) ? MLO_PAUSE_TXRX_MASK : MLO_PAUSE_NONE;
			state->link = !!((poa >> RT3050_POA_LINK_S) & BIT(port));
			state->an_enabled = true;
		}
		state->interface = PHY_INTERFACE_MODE_INTERNAL;
		state->an_complete = state->link;
	}
}

void rt3050_set_port_priority(struct rt3050_priv *priv, int port, u32 queue)
{
	rt3050_rmw(priv, RT3050_REG_PFC1, 0,
				RT3050_PFC1_PORT_PRI_VAL_SET(port, queue));
}

u32 rt3050_get_port_priority(struct rt3050_priv *priv, int port)
{
	return RT3050_PFC1_PORT_PRI_VAL_GET(port, rt3050_r32(priv, RT3050_REG_PFC1));
}

void rt3050_set_port_tos(struct rt3050_priv *priv, int port, u32 value)
{
	rt3050_rmw(priv, RT3050_REG_PFC1, (BIT(port) & RT3050_PFC1_EN_TOS_M) << RT3050_PFC1_EN_TOS_S,
				(value ? BIT(RT3050_PFC1_EN_TOS_S + port) : 0));
}

u32 rt3050_get_port_tos(struct rt3050_priv *priv, int port)
{
	return ((rt3050_r32(priv, RT3050_REG_PFC1) >> RT3050_PFC1_EN_TOS_S) & (RT3050_PFC1_EN_TOS_M & BIT(port)) ? 1 : 0);
}

void rt3050_set_port_vlan(struct rt3050_priv *priv, int port, u32 value)
{
	rt3050_rmw(priv, RT3050_REG_PFC1, (BIT(port) & RT3050_PFC1_EN_VLAN_M) << RT3050_PFC1_EN_VLAN_S,
				(value ? BIT(RT3050_PFC1_EN_VLAN_S + port) : 0));
}

u32 rt3050_get_port_vlan(struct rt3050_priv *priv, int port)
{
	return ((rt3050_r32(priv, RT3050_REG_PFC1) >> RT3050_PFC1_EN_VLAN_S) & (RT3050_PFC1_EN_VLAN_M & BIT(port)) ? 1 : 0);
}

void rt3050_set_port_threshold(struct rt3050_priv *priv, int port, u32 value)
{
	rt3050_rmw(priv, RT3050_REG_FCT1, (BIT(port) & RT3050_FCT_M) << RT3050_FCT1_PORT_TH_S,
				(value ? BIT(RT3050_FCT1_PORT_TH_S + port) : 0));
}

u32 rt3050_get_port_threshold(struct rt3050_priv *priv, int port)
{
	return ((rt3050_r32(priv, RT3050_REG_FCT1) >> RT3050_FCT1_PORT_TH_S) & (RT3050_FCT_M & BIT(port)) ? 1 : 0);
}

void rt3050_set_port_turnoff(struct rt3050_priv *priv, int port, u32 value)
{
	rt3050_rmw(priv, RT3050_REG_PFC0, (BIT(port) & RT3050_PFC0_TURN_OFF_FC_M) << RT3050_PFC0_TURN_OFF_FC_S,
				(value ? BIT(RT3050_PFC0_TURN_OFF_FC_S + port) : 0));
}

u32 rt3050_get_port_turnoff(struct rt3050_priv *priv, int port)
{
	return ((rt3050_r32(priv, RT3050_REG_PFC0) >> RT3050_PFC0_TURN_OFF_FC_S) & (RT3050_PFC0_TURN_OFF_FC_M & BIT(port)) ? 1 : 0);
}

void rt3050_set_pref_queue(struct rt3050_priv *priv, u32 pref)
{
	rt3050_rmw(priv, RT3050_REG_PFC1, RT3050_PFC1_PRIORITY_OPTION_M << RT3050_PFC1_PRIORITY_OPTION_S,
		(pref & RT3050_PFC1_PRIORITY_OPTION_M) << RT3050_PFC1_PRIORITY_OPTION_S);
}

u32 rt3050_get_pref_queue(struct rt3050_priv *priv)
{
	return (rt3050_r32(priv, RT3050_REG_PFC1) >> RT3050_PFC1_PRIORITY_OPTION_S) & RT3050_PFC1_PRIORITY_OPTION_M;
}

void rt3050_set_queue_weight(struct rt3050_priv *priv, int queue, u32 weight)
{
	rt3050_rmw(priv, RT3050_REG_PFC0, RT3050_PFC0_NUM_M << RT3050_PFC0_QUEUE_NUM_S(queue),
		(weight & RT3050_PFC0_NUM_M) << RT3050_PFC0_QUEUE_NUM_S(queue));
}

u32 rt3050_get_queue_weight(struct rt3050_priv *priv, int queue)
{
	return (rt3050_r32(priv, RT3050_REG_PFC0) >> RT3050_PFC0_QUEUE_NUM_S(queue)) & RT3050_PFC0_NUM_M;
}

void rt3050_set_queue_threshold(struct rt3050_priv *priv, int queue, u32 threshold)
{
	rt3050_rmw(priv, RT3050_REG_PFC2, RT3050_PFC2_PRI_TH_M << RT3050_PFC2_QUEUE_PRI_TH_S(queue),
		(threshold & RT3050_PFC2_PRI_TH_M) << RT3050_PFC2_QUEUE_PRI_TH_S(queue));
}

u32 rt3050_get_queue_threshold(struct rt3050_priv *priv, int queue)
{
	return (rt3050_r32(priv, RT3050_REG_PFC2) >> RT3050_PFC2_QUEUE_PRI_TH_S(queue)) & RT3050_PFC2_PRI_TH_M;
}

void rt3050_set_queue_priority_tag(struct rt3050_priv *priv, int queue, u32 priority_tag)
{
	rt3050_rmw(priv, RT3050_REG_PVIDC(3), RT3050_PVIDC3_QUE_PRIT_M(queue),
		RT3050_PVIDC3_QUE_PRIT_VAL_SET(queue, priority_tag));
}

u32 rt3050_get_queue_priority_tag(struct rt3050_priv *priv, int queue)
{
	return RT3050_PVIDC3_QUE_PRIT_VAL_GET(queue, rt3050_r32(priv, RT3050_REG_PVIDC(3)));
}

void rt3050_set_queue_tags(struct rt3050_priv *priv, int queue, u32 *tags, int count)
{
	u32 mask = 0, val = 0;

	for (int i = 0; i < count; i++) {
		mask |= RT3050_GQS0_TAG_M << RT3050_GQS0_TAG_S(tags[i]);
		val |= (queue & RT3050_GQS0_TAG_M) << RT3050_GQS0_TAG_S(tags[i]);
	}
	rt3050_rmw(priv, RT3050_REG_GQS0, mask, val);
}

u32 * rt3050_get_queue_tags(struct rt3050_priv *priv, int queue, u32 *tags, int *count)
{
	u32 gqs0;

	*count = 0;
	gqs0 = rt3050_r32(priv, RT3050_REG_GQS0);
	for (int i = 0; i < RT3050_MAX_TAGS; i++) {
		if (((gqs0 >> RT3050_GQS0_TAG_S(i)) & RT3050_GQS0_TAG_M) == queue) {
			tags[*count] = i;
			(*count)++;
		}
	}
	return tags;
}

void rt3050_set_queue_ports(struct rt3050_priv *priv, int queue, u32 *ports, int count)
{
	u32 mask = 0, val = 0;

	for (int i = 0; i < count; i++) {
		mask |= RT3050_PFC1_PORT_PRI_M << RT3050_PFC1_PORT_PRI_S(ports[i]);
		val |= RT3050_PFC1_PORT_PRI_VAL_SET(ports[i], queue);
	}
	rt3050_rmw(priv, RT3050_REG_PFC1, mask, val);
}

u32 * rt3050_get_queue_ports(struct rt3050_priv *priv, int queue, u32 *ports, int *count)
{
	u32 pfc1;

	*count = 0;
	pfc1 = rt3050_r32(priv, RT3050_REG_PFC1);
	for (int i = 0; i < RT3050_NUM_PORTS; i++) {
		if (RT3050_PFC1_PORT_PRI_VAL_GET(i, pfc1) == queue)
			ports[(*count)++] = i;
	}
	return ports;
}

void rt3050_set_fct0_reg(struct rt3050_priv *priv, int start, u32 value)
{
	rt3050_rmw(priv, RT3050_REG_FCT0, RT3050_FCT_M << start,
		(value & RT3050_FCT_M) << start);
}

u32 rt3050_get_fct0_reg(struct rt3050_priv *priv, int start)
{
	return (rt3050_r32(priv, RT3050_REG_FCT0) >> start) & RT3050_FCT_M;
}


void rt3050_set_bc_storm_control(struct rt3050_priv *priv, u32 value)
{
	rt3050_rmw(priv, RT3050_REG_SGC, RT3050_SGC_BC_STORM_M << RT3050_SGC_BC_STORM_S,
				(value & RT3050_SGC_BC_STORM_M) << RT3050_SGC_BC_STORM_S);
}

u32 rt3050_get_bc_storm_control(struct rt3050_priv *priv)
{
	return (rt3050_r32(priv, RT3050_REG_SGC) >> RT3050_SGC_BC_STORM_S) & RT3050_SGC_BC_STORM_M;
}

static int rt3050_port_mac_op(struct rt3050_priv *priv, int port, int vlan, bool add, bool mc, const unsigned char *addr)
{
	unsigned long t_start;
	u32 state, val = ((BIT(port) & RT3050_WMAD0_W_PORT_MAP_M) << RT3050_WMAD0_W_PORT_MAP_S) |
			((vlan & RT3050_WMAD0_W_INDEX_M) << RT3050_WMAD0_W_INDEX_S) |
			((1 & RT3050_WMAD0_W_MAC_CMD_M) << RT3050_WMAD0_W_MAC_CMD_S);

	if (add)
		val |= (RT3050_WMAD0_W_AGE_FIELD_M << RT3050_WMAD0_W_AGE_FIELD_S);
	if (mc)
		val |= ((1 & RT3050_WMAD0_W_MC_INGRESS_M) << RT3050_WMAD0_W_MC_INGRESS_S);

	t_start = jiffies;
	while (1) {
		state = rt3050_r32(priv, RT3050_REG_WMAD0);
		if ((state >> RT3050_WMAD0_AT_CFG_IDLE_S) & RT3050_WMAD0_AT_CFG_IDLE_M) {
			break;
		}
		if (time_after(jiffies, t_start + RT3050_ATS_TIMEOUT)) {
			dev_err(priv->dev, "(%s) MAC enginie IDLE timeout\n", __func__);
			goto out;
		}
	}
	rt3050_w32(priv, val, RT3050_REG_WMAD0);
	rt3050_w32(priv, (addr[4] << 8 | addr[5]) & 0xffff, RT3050_REG_WMAD1);
	rt3050_w32(priv, (addr[0] << 24 | addr[1] << 16 | addr[2] << 8 | addr[3]), RT3050_REG_WMAD2);

	t_start = jiffies;
	while (1) {
		state = rt3050_r32(priv, RT3050_REG_WMAD0);
		if (((state >> RT3050_WMAD0_W_MAC_DONE_S) & RT3050_WMAD0_W_MAC_DONE_M) || 
		    ((state >> RT3050_WMAD0_AT_CFG_IDLE_S) & RT3050_WMAD0_AT_CFG_IDLE_M)) {
			break;
		}
		if (time_after(jiffies, t_start + RT3050_ATS_TIMEOUT)) {
			dev_err(priv->dev, "(%s) MAC enginie IDLE timeout\n", __func__);
			goto out;
		}
	}

	if ((state >> RT3050_WMAD0_W_MAC_DONE_S) & RT3050_WMAD0_W_MAC_DONE_M)
		return 0;

out:
	return -EINVAL;
}

int rt3050_port_xdb_op(struct rt3050_priv *priv, int port, 
	const unsigned char *addr, u16 vid, bool add, bool mc)
{
	int ret = -EINVAL;

	if (!vid) {
		for (int i = 0; i < RT3050_NUM_VLANS; i++) {
			if (priv->vlans[i].bridge_ptr && (priv->vlans[i].ports & BIT(port))) {
				ret = rt3050_port_mac_op(priv, port, i, add, mc, addr);
				if (ret)
					break;
			}
		}
	} else {
		if (port == priv->cpu_port) {
			/* Seems that need to find all ports with such assigned VLAN and add MAC to any VLANs that contains such port */
			for (int i = 0; i < priv->cpu_port; i++) {
				struct rt3050_br_port_vlan_entry *entry = rt3050_lookup_port_bridge_vlan(&priv->ports[i], vid);

				if (entry) {
					for (int j = 0; j < RT3050_NUM_VLANS; j++) {
						if (priv->vlans[j].bridge_ptr && (priv->vlans[j].ports & BIT(i))) {
							ret = rt3050_port_mac_op(priv, port, j, add, mc, addr);
							if (ret)
								break;
						}
					}
				}
			}
		} else {
			struct rt3050_br_port_vlan_entry *entry = rt3050_lookup_port_bridge_vlan(&priv->ports[port], vid);

			if (entry) {
				for (int i = 0; i < RT3050_NUM_VLANS; i++) {
					if (priv->vlans[i].bridge_ptr && (priv->vlans[i].ports & BIT(port))) {
						ret = rt3050_port_mac_op(priv, port, i, add, mc, addr);
						if (ret)
							break;
					}
				}
			}
		}
	}
	return ret;
}

void rt3050_set_port_secured_mode(struct rt3050_priv *priv, int port, struct rt3050_sa_secured_mac_addr_entry *values, int count)
{
	struct rt3050_sa_secured_mac_addr_entry *entry, *tmp;

	list_for_each_entry_safe(entry, tmp, &priv->ports[port].sa_secured_mac_addr_list, list) {
		list_del(&entry->list);
		kfree(entry);
	}

	for (int i = 0; i < count; i++) {
		entry = kmalloc(sizeof(*entry), GFP_ATOMIC);
		if (!entry) {
			dev_err(priv->dev, "(%s). No memory to set secured mode to port %d\n", __func__, port);
			return;
		}
		memcpy(entry->mac, values[i].mac, ETH_ALEN);
		INIT_LIST_HEAD(&entry->list);
		list_add(&entry->list, &priv->ports[port].sa_secured_mac_addr_list);
		rt3050_port_xdb_op(priv, port, entry->mac, priv->ports[port].pvid, true, is_multicast_ether_addr(entry->mac));
		rt3050_port_xdb_op(priv, port, entry->mac, 0, true, is_multicast_ether_addr(entry->mac));
	}
	if (count > 0)
		rt3050_rmw(priv, RT3050_REG_POC1, RT3050_STP_SECURED_PORT(port) | RT3050_STP_DIS_LEARN(port),
			RT3050_STP_SECURED_PORT(port) | RT3050_STP_DIS_LEARN(port));
	else
		rt3050_rmw(priv, RT3050_REG_POC1, RT3050_STP_SECURED_PORT(port) | RT3050_STP_DIS_LEARN(port), 0);
}

u32 rt3050_get_port_secured_mode(struct rt3050_priv *priv, int port, struct rt3050_sa_secured_mac_addr_entry *values, int *count)
{
	struct rt3050_sa_secured_mac_addr_entry *entry;

	*count = 0;
	list_for_each_entry(entry, &priv->ports[port].sa_secured_mac_addr_list, list) {
		memcpy(values[*count].mac, entry->mac, ETH_ALEN);
		(*count)++;
		if (*count >= RT3050_NUM_VLANS)
			break;
	}
	return *count;
}

void rt3050_set_port_led_state(struct rt3050_priv *priv, int port, u32 value)
{
	priv->ports[port].led = value;
	rt3050_w32(priv, value & RT3050_LEDPX_M, RT3050_REG_LEDPX(port));
}

u32 rt3050_get_port_led_state(struct rt3050_priv *priv, int port)
{
	return rt3050_r32(priv, RT3050_REG_LEDPX(port)) & RT3050_LEDPX_M;
}

void rt3050_set_port_doubletag(struct rt3050_priv *priv, int port, u32 value)
{
	priv->ports[port].doubletag = !!value;
	rt3050_rmw(priv, RT3050_REG_SGC2, (BIT(port) & RT3050_SGC2_DOUBLE_TAG_EXTPORTS_M) << RT3050_SGC2_DOUBLE_TAG_S,
				(value ? BIT(port) : 0)  << RT3050_SGC2_DOUBLE_TAG_S);
}

u32 rt3050_get_port_doubletag(struct rt3050_priv *priv, int port)
{
	return ((rt3050_r32(priv, RT3050_REG_SGC2) >> RT3050_SGC2_DOUBLE_TAG_S) & (RT3050_SGC2_DOUBLE_TAG_EXTPORTS_M & BIT(port)) ? 1 : 0);
}
