#include "rt3050.h"

static int mac_table_cb(const unsigned char *mac, u16 vid, bool is_static, void * data)
{
	struct seq_file *m = data;

	seq_printf(m, " %pM\t%d\t%s\n", mac, (int)vid, (is_static ? "static" : "temp"));

	return 0;
}

static int mac_table_show(struct seq_file *m, void *v)
{
	u32 i;
	struct rt3050_priv *priv = m->private;

	seq_puts(m, "MAC address table\n");
	for (i = 0; i < RT3050_NUM_PORTS; i++) {
		seq_printf(m, "Port %d\n", i);
		rt3050_port_fdb_dump(priv->ds, i, mac_table_cb, m);
	}

	return 0;
}

static int mac_table_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, mac_table_show, inode->i_private);
}

static const struct file_operations mac_table_fops = {
	.owner = THIS_MODULE,
	.open = mac_table_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void ports_to_vlan(struct seq_file *m, struct rt3050_priv *priv)
{
	int i, j;
	u32 ports;
	u32 untag;

	untag = rt3050_get_untagged_port(priv);

	for (i = 0; i < RT3050_NUM_PORTS; i++) {
		seq_printf(m, "VLANs per port %d %s: ", i,
			(untag ? "untagged" : "tagged"));
		for (j = 0; j < RT3050_NUM_VLANS; j++) {
			ports = rt3050_get_vmsc(priv, j);
			if (ports & BIT(i))
				seq_printf(m, "%d ", j);
		}
		seq_puts(m, "\n");
	}
	seq_puts(m, "\n");
}

static void port_to_vlan(struct seq_file *m, struct rt3050_priv *priv, int port)
{
	int j;
	u32 ports;
	u32 untag;

	untag = rt3050_get_untagged_port(priv);

	seq_printf(m, "VLANs per port %d %s: ", port,
		(untag ? "untagged" : "tagged"));
	for (j = 0; j < RT3050_NUM_VLANS; j++) {
		ports = rt3050_get_vmsc(priv, j);
		if (ports & BIT(port))
			seq_printf(m, "%d ", j);
	}
	seq_puts(m, "\n\n");
}

static void get_priv_port_info(struct seq_file *m, struct rt3050_priv *priv, int port)
{
	seq_printf(m, "Port %d PRIV INFO: disabled: %s\tup: %s\tdoubletag: %s\tuntag: %s\tPVID: %d\tled: %d\n", port, 
		priv->ports[port].disabled ? " true " : "false",
		priv->ports[port].up ? "true" : "false",
		priv->ports[port].doubletag ? "true" : "false",
		priv->ports[port].untag ? "true" : "false",
		priv->ports[port].pvid,
		priv->ports[port].led);
}

static const char * get_port_stp_state(u32 poc1, int port)
{
	u32 stp_state;

	stp_state = ((poc1 >> RT3050_POC1_BLOCKING_STATE_S) & RT3050_POC1_BLOCKING_STATE_M) & BIT(port);

	return (!stp_state ? "FWD" : "BLK");
}

static int ports_show(struct seq_file *m, void *v)
{
	struct rt3050_priv *priv = m->private;
	struct phylink_link_state state;
	u32 poc0, poc1;

	mutex_lock(&priv->mtx);

	poc0 = rt3050_r32(priv, RT3050_REG_POC0);
	poc1 = rt3050_r32(priv, RT3050_REG_POC1);

	seq_puts(m, "Ports table\n");
	for (int i = 0; i < RT3050_NUM_PHY_PORTS; i++) {
		rt3050_get_phylink_port_state(priv, i, &state);

		seq_printf(m, "User port %d: %s\t%s\t%s\t%s\tmode: %s\tPVID: %d\tSTP: %s\tStatus: %s\n", i, 
				state.link ? " up " : "down",
				state.duplex ? "full dpx" : "half dpx",
				state.speed == SPEED_100 ? " 100M" : "  10M",
				state.pause == MLO_PAUSE_NONE ? " FC OFF " : " FC ON  ",
				state.an_enabled ? " AN  " : "FORCE",
				(int)rt3050_get_pvid(priv, i),
				get_port_stp_state(poc1, i),
				(poc0 >> RT3050_POC0_DIS_PORT_S) & BIT(i) ? "Disabled" : " Enabled");

		get_priv_port_info(m, priv, i);
		port_to_vlan(m, priv, i);
	}

	rt3050_get_phylink_port_state(priv, priv->cpu_port, &state);
	seq_printf(m, "CPU port  %d: %s\t%s\t%s\t%s\tmode: %s\tPVID: %d\tSTP: %s\tStatus: %s\n", priv->cpu_port, 
		state.link ? " up " : "down",
		state.duplex ? "full dpx" : "half dpx",
		(state.speed == SPEED_1000 ? "1000M" : (state.speed == SPEED_100 ? " 100M" : " 10M")),
		(state.pause == MLO_PAUSE_TXRX_MASK ? "rx/tx FC" : 
			(state.pause == MLO_PAUSE_TX ? "  tx FC " : 
				(state.pause == MLO_PAUSE_RX ? "  rx FC " : " FC OFF "))),
		state.an_enabled ? " AN  " : "FORCE",
		(int)rt3050_get_pvid(priv, priv->cpu_port),
		get_port_stp_state(poc1, priv->cpu_port),
		(poc0 >> RT3050_POC0_DIS_PORT_S) & BIT(priv->cpu_port) ? "Disabled" : " Enabled");
	get_priv_port_info(m, priv, priv->cpu_port);
	port_to_vlan(m, priv, priv->cpu_port);

	mutex_unlock(&priv->mtx);

	return 0;
}

static int ports_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, ports_show, inode->i_private);
}

static const struct file_operations ports_fops = {
	.owner = THIS_MODULE,
	.open = ports_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int vlans_show(struct seq_file *m, void *v)
{
	struct rt3050_priv *priv = m->private;
	u32 i, j, untag, ports;
	u16 vid;

	mutex_lock(&priv->mtx);

	untag = rt3050_get_untagged_port(priv);
	
	seq_puts(m, "VLANs table\n");
	for (i = 0; i < RT3050_NUM_VLANS; i++) {
		vid = rt3050_get_vlan_id(priv, i);
		ports = rt3050_get_vmsc(priv, i);
		if (!priv->vlans[i].ports && !ports)
			continue;

		seq_printf(m, "VLAN %d: vid - %d, ports(0-6) - ", i,
			vid);
		for (j = 0; j < RT3050_NUM_PORTS; ++j)
			if (ports & BIT(j))
				seq_printf(m, "%d%s ", j,
					(untag & BIT(j) ? "u" : "t"));
		seq_printf(m, "\n");

		seq_printf(m, "VLAN %d PRIV INFO: vid - %d, ports(0-6) - ", i,
			priv->vlans[i].vid);
		for (j = 0; j < RT3050_NUM_PORTS; ++j)
			if (priv->vlans[i].ports & BIT(j))
				seq_printf(m, "%d%s ", j,
					(priv->ports[j].untag ? "u" : "t"));
		seq_printf(m, "\n");
	}
	ports_to_vlan(m, priv);

	mutex_unlock(&priv->mtx);

	return 0;
}

static int vlans_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, vlans_show, inode->i_private);
}

static const struct file_operations vlans_fops = {
	.owner = THIS_MODULE,
	.open = vlans_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int global_show(struct seq_file *m, void *v)
{
	struct rt3050_priv *priv = m->private;
	u32 sgc, sgc2;
	u8 bc_storm, hash, bp_jam, bp_mode, led, rmc_rule, ipm_rule, lan_map, i;
	bool slot4to1, arbit56, cpu_tpid, arbit04;

	mutex_lock(&priv->mtx);

	sgc = rt3050_r32(priv, RT3050_REG_SGC);
	sgc2 = rt3050_r32(priv, RT3050_REG_SGC2);
	bc_storm = (sgc >> RT3050_SGC_BC_STORM_S) & RT3050_SGC_BC_STORM_M;
	hash = (sgc >> RT3050_SGC_ADDRESS_HASH_ALG_S) & RT3050_SGC_ADDRESS_HASH_ALG_M;
	bp_jam = (sgc >> RT3050_SGC_BP_JAM_CNT_S) & RT3050_SGC_BP_JAM_CNT_M;
	bp_mode = (sgc >> RT3050_SGC_BP_MODE_S) & RT3050_SGC_BP_MODE_M;
	led = (sgc >> RT3050_SGC_LED_FREQ_S) & RT3050_SGC_LED_FREQ_M;
	rmc_rule = (sgc >> RT3050_SGC_RMC_RULE_S) & RT3050_SGC_RMC_RULE_M;
	ipm_rule = (sgc >> RT3050_SGC_IP_MULT_RULE_S) & RT3050_SGC_IP_MULT_RULE_M;
	slot4to1 = (sgc2 >> RT3050_SGC2_SLOT_4TO1_S) & RT3050_SGC2_SLOT_4TO1_M;
	arbit56 = (sgc2 >> RT3050_SGC2_ARBITER_GPT_EN_S) & RT3050_SGC2_ARBITER_GPT_EN_M;
	cpu_tpid = (sgc2 >> RT3050_SGC2_CPU_TPID_EN_S) & RT3050_SGC2_CPU_TPID_EN_M;
	arbit04 = (sgc2 >> RT3050_SGC2_ARBITER_LAN_EN_S) & RT3050_SGC2_ARBITER_LAN_EN_M;
	lan_map = (sgc2 >> RT3050_SGC2_LAN_PMAP_S) & RT3050_SGC2_LAN_PMAP_M;
	seq_puts(m, "Global switch setings\n");
	seq_printf(m, "CPU port: %d\n", priv->cpu_port);
	seq_printf(m, "Broadcast Storm Protect: %s\n",
		(bc_storm == 3 ? "128" : (bc_storm == 2 ? "96" : (bc_storm == 1 ? "64" : "disable"))));
	seq_printf(m, "MAC address Hash alg: %s\n",
		(hash == 2 ? "XOR32" : (hash == 1 ? "XOR48" : "last 10-bit")));
	seq_printf(m, "Back pressure jam number: %d\n", bp_jam);
	seq_printf(m, "Back pressure mode: %s\n",
		(bp_mode == 3 ? "BP carrier" : (bp_mode == 2 ? "BP jamALL" : (bp_mode == 1 ? "BP jam" : "disable"))));
	seq_printf(m, "The frequency of LED flash: %s\n",
		(led == 3 ? "480ms" : (led == 2 ? "240ms" : (led == 1 ? "60ms" : "30ms"))));
	seq_printf(m, "Unknown reserved MC frame forward rule: %s\n",
		(rmc_rule == 2 ? "drop" : (rmc_rule == 1 ? "to cpu" : "to all port")));
	seq_printf(m, "Unknown IP MC frame forward rule: %s\n",
		(ipm_rule == 2 ? "drop" : (ipm_rule == 1 ? "to cpu" : "BC")));
	seq_printf(m, "Memory arbiter ratio: %s\n",
		(slot4to1 ? "(P5,P6) : (P0,P4) = 3:2" : "(P5,P6) : (P0,P4) = 4:1"));
	seq_printf(m, "Memory arbiter only for P5 and P6: %s\n",
		(arbit56 ? "Enable" : "Default"));
	seq_printf(m, "CPU TPID(81xx): %s\n",
		(cpu_tpid ? "Enable. CPU TPID=810x" : "Disable. CPU TPID=810x"));
	seq_printf(m, "Memory arbiter only for P0-P4: %s\n",
		(arbit04 ? "Enable" : "Default"));
	seq_puts(m, "Lan port bit map (0-6): ");
	for (i = 0; i < RT3050_NUM_PORTS; i++)
		seq_printf(m, "%s", (lan_map & BIT(i) ? "L" : "W"));
	seq_puts(m, "\n");
	mutex_unlock(&priv->mtx);

	for (int i = 0; i < RT3050_NUM_PHY_PORTS; i++)
		seq_printf(m, "Port %d Max MTU: %d\n", i, priv->ds->ops->port_max_mtu(priv->ds, i));

	return 0;
}

static int global_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, global_show, inode->i_private);
}

static const struct file_operations global_fops = {
	.owner = THIS_MODULE,
	.open = global_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int stats_show(struct seq_file *m, void *v)
{
	int count, i, j;
	struct rt3050_priv *priv = m->private;
	u8 names[ETH_GSTRING_LEN * RT3050_COUNTERS];
	u64 values[RT3050_COUNTERS];

	count = priv->ds->ops->get_sset_count(priv->ds, 0, ETH_SS_STATS);
	if (count > RT3050_COUNTERS) {
		seq_printf(m, "Wrong counters: %d-%d\n", count, RT3050_COUNTERS);
		count = RT3050_COUNTERS;
	}

	seq_puts(m, "Switch statistics\n");
	for (i = 0; i < RT3050_NUM_PORTS; i++) {
		const char *ptr = names;

		priv->ds->ops->get_strings(priv->ds, i, ETH_SS_STATS, names);
		priv->ds->ops->get_ethtool_stats(priv->ds, i, values);
		seq_printf(m, "Port %d:\n", i);
		for (j = 0; j < count; j++) {
			seq_printf(m, "\t%s: %lld\n", ptr, values[j]);
			ptr += ETH_GSTRING_LEN;
		}
	}

	return 0;
}

static int stats_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, stats_show, inode->i_private);
}

static const struct file_operations stats_fops = {
	.owner = THIS_MODULE,
	.open = stats_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int stp_show(struct seq_file *m, void *v)
{
	int i;
	u32 poc1, blk, dis_lrn, secured;
	struct rt3050_priv *priv = m->private;

	mutex_lock(&priv->mtx);

	poc1 = rt3050_r32(priv, RT3050_REG_POC1);

	blk = (poc1 >> RT3050_POC1_BLOCKING_STATE_S) & RT3050_POC1_BLOCKING_STATE_M;
	dis_lrn = (poc1 >> RT3050_POC1_DIS_LRNING_S) & RT3050_POC1_DIS_LRNING_M;
	secured = (poc1 >> RT3050_POC1_SA_SECURED_PORT_S) & RT3050_POC1_SA_SECURED_PORT_M;

	seq_puts(m, "STP table\n");
	for (i = 0; i < RT3050_NUM_PORTS; i++) {
		seq_printf(m, "Port %d:\tSTATE: %s\tLEARNING: %s\tSECURITY: %s\n", i, 
			blk & BIT(i) ? "blocking" : "normal",
			dis_lrn & BIT(i) ? "disable" : "enable",
			secured & BIT(i) ? "SA need match" : "don't care SA match");
	}

	mutex_unlock(&priv->mtx);

	return 0;
}

static int stp_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, stp_show, inode->i_private);
}

static const struct file_operations stp_fops = {
	.owner = THIS_MODULE,
	.open = stp_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

void rt3050_dbgfs_init(struct rt3050_priv *priv)
{
	struct dentry *dbg_dir;

	dbg_dir = debugfs_lookup(RT3050_DRIVER_NAME, NULL);
	if (!dbg_dir)
		dbg_dir = debugfs_create_dir(RT3050_DRIVER_NAME, NULL);

	priv->dbgfs_dir = dbg_dir;

	debugfs_create_file("mac_table", 0400, dbg_dir, priv, &mac_table_fops);
	debugfs_create_file("ports", 0400, dbg_dir, priv, &ports_fops);
	debugfs_create_file("vlans", 0400, dbg_dir, priv, &vlans_fops);
	debugfs_create_file("global", 0400, dbg_dir, priv, &global_fops);
	debugfs_create_file("statistics", 0400, dbg_dir, priv, &stats_fops);
	debugfs_create_file("stp", 0400, dbg_dir, priv, &stp_fops);
}

void rt3050_dbgfs_cleanup(struct rt3050_priv *priv)
{
	debugfs_remove_recursive(priv->dbgfs_dir);
}
