#include <plib/utils.h>
#include <plib/exec.h>
#include <plib/fs.h>
#include <plib/conf.h>
#include <plib/osutil.h>
#include <plib/crypt.h>

PLEX BOOL sched_reload(void) {
    return os_kill_by_name("cron", SIGHUP);
}

PLEX BOOL sched_remove_jobs(const char * user) {
    if (!SEMPTY(user)) {
	return os_system("crontab -u %s -r", user);
    }
    return FALSE;
}

static int __del_user_jobs(const char * name, BOOL isdir, unsigned long flags, void * data) {
    BOOL res = FALSE;

    if (!isdir) {
	const char	* cptr = name, * uptr = name;

	while (*cptr != 0) {
	    if (*cptr == '/')
		uptr = cptr + 1;
	    cptr ++;
	}
	if (!sched_remove_jobs(uptr)) {
	    log_errno("Could not remove jobs for %s", uptr);
	}
    }
    return 0;
}

PLEX void sched_remove_all_jobs(void) {
    fs_dir_walk_through("/etc/crontabs", __del_user_jobs, NULL, NULL);
    sched_reload();
}

PLEX val_t * sched_list_jobs(const char * user) {
    val_t	* root = NULL, * output;
    char	params[256];

    snprintf(params, sizeof(params), "-u %s -l", user);
    if ((output = os_exec_as_dict("crontab", params)) != NULL) {
	int	i;

	FOREACH_IN_DICT(output, i) {
	    val_t	* v = dict_find_node(V_CHILD(output, i), "id");

	    if (v != NULL) {
		char	* ptr = val_as_str_simple(v);

		if (!SEMPTY(ptr) && (ptr[0] != '#') && (strncmp(ptr, "no", 2) != 0) && (strncmp(ptr, "crontab", 7) != 0)) {
		    size_t	szm = 2, k;
		    char	* mwords[2];

		    split(ptr, mwords, &szm, "#");
		    if (szm >= 1) {
			size_t	sz = 20;
			char	* words[20];

			split(mwords[0], words, &sz, " ");
			if (sz >= 6) {
			    if (root == NULL) {
				root = val_new(NULL);
			    }
			    if (root != NULL) {
				val_t	* nstr, * nv;

				if ((nstr = val_new(NULL)) != NULL) {
				    char	* nptr = NULL;

				    nstr->type = VAL_ARRAY;
				    val_dict_add(root, nstr);
				    for (k = 0; k < 5; k ++) {
					if ((nv = val_new(NULL)) != NULL) {
					    val_from_str(nv, words[k]);
					    val_dict_add(nstr, nv);
					}
				    }
				    for (k = 5; k < sz; k ++)
					os_asprintf(&nptr, "%s ", words[k]);
				    if (nptr != NULL) {
					char	* sptr = strstrip(NULL, 0, nptr);

					free(nptr);
					if (sptr != NULL) {
					    if ((nv = val_new(NULL)) != NULL) {
						val_from_str(nv, sptr);
						val_dict_add(nstr, nv);
					    }
					    ZFREE(sptr);
					}
				    }
				    if (szm > 1) {
					if ((nv = val_new(NULL)) != NULL) {
					    strreplace(mwords[1], ' ', '_');
					    val_from_str(nv, mwords[1]);
					    val_dict_add(nstr, nv);
					}
				    }
				}
			    }
			}
			for (k = 0; k < sz; k ++)
			    ZFREE(words[k]);
		    }
		    for (k = 0; k < szm; k ++)
			ZFREE(mwords[k]);
		}
		ZFREE(ptr);
	    }
	}
	val_free(output);
    }
    return root;
}

PLEX BOOL sched_restore_jobs(void) {
    BOOL	res = FALSE;
    val_t	* cfg = parse_jsonfile(SYS_CONFIG_FILE), * node;

    sched_remove_all_jobs();
    if ((node = dict_find_node_near(cfg, "schedule", TRUE)) != NULL) {
	int	i;

	FOREACH_IN_DICT(node, i) {
	    val_t	* child = V_CHILD(node, i);
	    char	buf[256], filename[512], * ptr = val_as_str_simple(child);

	    str_unescape(ptr, strlen(ptr), ptr);
	    fs_tmp_name(NULL, buf, sizeof(buf));
	    fs_put_data(buf, "%s\n", ptr);
	    ZFREE(ptr);
	    snprintf(filename, sizeof(filename), "/etc/crontabs/%s", val_name(child));
	    if (!fs_exists(filename)) {
		fs_mkdir("/etc/crontabs", 0755);
		fs_touch_simple(filename);
	    }
	    res = os_system("crontab -u %s %s", val_name(child), buf);
	    fs_remove(buf);
	}
    }
    val_free(cfg);
    sched_reload();
    return res;
}

PLEX BOOL sched_install_jobs(const char * user, val_t * jobs) {
    BOOL	res = FALSE;

    if (!SEMPTY(user)) {
	char	buf[BUFSIZ];
	int	i;
	ssize_t	written;
	char	* dptr = buf;
	size_t	remain = sizeof(buf);

	written = snprintf(dptr, remain, "# Automatically generated by NETSHe \n");
	dptr += written;
	remain -= written;
	FOREACH_IN_DICT(jobs, i) {
	    int		j;
	    val_t	* child = V_CHILD(jobs, i);

	    if (V_DICT_SIZE(child) < 6)
		continue;
	    FOREACH_IN_DICT(child, j) {
		char	* ptr = NULL;

		if (j < 6) {
		    ptr = val_as_str_simple(V_CHILD(child, j));
		    written = snprintf(dptr, remain, "%s ", ptr);
		    dptr += written;
		    remain -= written;
		} else if (j == 6) {
		    ptr = val_as_str_simple(V_CHILD(child, j));
		    strreplace(ptr, '_', ' ');
		    written = snprintf(dptr, remain, "#%s", ptr);
		    dptr += written;
		    remain -= written;
		} else {
		    continue;
		}
		ZFREE(ptr);
	    }
	    written = snprintf(dptr, remain, "\n");
	    dptr += written;
	    remain -= written;
	}
	dptr = json_escape(buf, strlen(buf));
	write_simple_event("setval schedule:%s %s", user, buf);
	ZFREE(dptr);
	os_sleep(3, 0, 0);
	res = sched_restore_jobs();
    }
    return res;
}

PLEX BOOL sched_install_job(const char * user, const char * job) {
    BOOL	found = FALSE;
    val_t	* rule = parse_jsonstr(job), * rules = sched_list_jobs(user);

    if ((V_IS_OBJ(rule) || V_IS_ARR(rule)) && (V_DICT_SIZE(rule) >= 6)) {
	char	crule[ASCIILINESZ];

	if (dict_as_jsonbuf(crule, sizeof(crule), rule) != NULL) {
	    char	* hash;

	    if ((hash = P_SHA1_txt(crule, strlen(crule))) != NULL) {
		int	j;

		FOREACH_IN_DICT(rules, j) {
		    char	txt[ASCIILINESZ], * thash;

		    if (found)
			break;
		    if (dict_as_jsonbuf(txt, sizeof(txt), V_CHILD(rules, j)) != NULL) {
			if ((thash = P_SHA1_txt(txt, strlen(txt))) != NULL) {
			    if (strcmp(hash, thash) == 0)
				found = TRUE;
			    ZFREE(thash);
			}
		    }
		}
		ZFREE(hash);
	    }
	}
    }
    if (!found) {
	if (rules == NULL)
	    rules = val_new(NULL);
	if (rules != NULL) {
	    val_dict_add(rules, rule);
	    sched_install_jobs(user, rules);
	} else
	    val_free(rule);
    } else
	val_free(rule);
    val_free(rules);
    return !found;
}

PLEX BOOL sched_remove_job_uniq(const char * user, const char * job) {
    BOOL	res = FALSE;
    val_t	* jobs = sched_list_jobs(user);

    if (jobs != NULL) {
	val_t	* rule = parse_jsonstr(job);

	if ((V_IS_OBJ(rule) || V_IS_ARR(rule)) && (V_DICT_SIZE(rule) >= 6)) {
	    BOOL	found = FALSE;
	    char	* cmd = val_as_str_simple(V_CHILD(rule, 5));
	    int		j;

to_begin_of_cycle:
	    FOREACH_IN_DICT(jobs, j) {
		val_t	* child = V_CHILD(jobs, j);

		if ((V_IS_OBJ(rule) || V_IS_ARR(rule)) && (V_DICT_SIZE(rule) >= 6)) {
		    char	* tptr = val_as_str_simple(V_CHILD(child, 5));
		    BOOL	match = FALSE;

		    match = (BOOL)(strcmp(cmd, tptr) == 0);
		    ZFREE(tptr);
		    if (match) {
			found = TRUE;
			val_free(child);
			goto to_begin_of_cycle;
		    }
		}
	    }
	    ZFREE(cmd);
	    if (found)
		res = sched_install_jobs(user, jobs);
	}
	val_free(rule);
	val_free(jobs);
    }
    return res;
}

PLEX BOOL sched_is_job_installed_uniq(const char * user, const char * job) {
    BOOL	res = FALSE;
    val_t	* jobs = sched_list_jobs(user);

    if (jobs != NULL) {
	val_t	* rule = parse_jsonstr(job);

	if ((V_IS_OBJ(rule) || V_IS_ARR(rule)) && (V_DICT_SIZE(rule) >= 6)) {
	    BOOL	found = FALSE;
	    char	* cmd = val_as_str_simple(V_CHILD(rule, 5));
	    int		j;

	    FOREACH_IN_DICT(jobs, j) {
		val_t	* child = V_CHILD(jobs, j);

		if ((V_IS_OBJ(child) || V_IS_ARR(child)) && (V_DICT_SIZE(child) >= 6)) {
		    char	* tptr = val_as_str_simple(V_CHILD(child, 5));

		    res = (BOOL)(strcmp(cmd, tptr) == 0);
		    ZFREE(tptr);
		    if (res) {
			break;
		    }
		}
	    }
	    ZFREE(cmd);
	}
	val_free(rule);
	val_free(jobs);
    }
    return res;
}
