/**
 * @file
 * @brief SMTP client library.
 * @author James Humphrey (mail@somnisoft.com)
 * @version 1.00
 *
 * This SMTP client library allows the user to send emails to an SMTP server.
 * The user can include custom headers and MIME attachments.
 *
 * This software has been placed into the public domain using CC0.
 */

/**
 * @mainpage smtp-client
 *
 * This section contains documentation generated directly from the source
 * code.
 *
 * To view the repository details, visit the main smtp-client page at
 * <a href='https://www.somnisoft.com/smtp-client'>
 * www.somnisoft.com/smtp-client
 * </a>.
 */

#if defined(_WIN32) || defined(WIN32)
# define SMTP_IS_WINDOWS
#endif /* SMTP_IS_WINDOWS */

#ifdef SMTP_IS_WINDOWS
# include <winsock2.h>
# include <ws2tcpip.h>
#else /* POSIX */
# include <netinet/in.h>
# include <sys/select.h>
# include <sys/socket.h>
# include <netdb.h>
# include <unistd.h>
#endif /* SMTP_IS_WINDOWS */

#include <errno.h>
#include <limits.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <plib/fs.h>
#include <plib/uri.h>
#include <plib/template.h>

/**
 * Get access to the @ref smtp_result_code and @ref smtp_command definitions.
 */
#define SMTP_INTERNAL_DEFINE

#include "internal.h"

/*
 * The SMTP_TEST converts some library routines into special test seams which
 * allows the test program to control whether they fail. For example, we can
 * control when malloc() fails under certain conditions with an out of
 * memory condition.
 */
#ifdef SMTP_TEST
/**
 * Declare extern linkage on some functions so we can redefine their behavior
 * in the external test suite.
 */
# define SMTP_LINKAGE extern
# include "../test/seams.h"
#else /* !(SMTP_TEST) */
/**
 * When not testing, all functions should have static linkage except for those
 * in the header.
 */
# define SMTP_LINKAGE static
#endif /* SMTP_TEST */

/**
 * Increment the read buffer size by this amount if the delimiter
 * has not been found.
 */
#define SMTP_GETDELIM_READ_SZ 1000


/**
 * Find and return the location of the delimiter character in the
 * search buffer.
 *
 * This function gets used by the main socket parsing function which
 * continually reads from the socket and expands the buffer until it
 * encounters the expected delimiter. This function provides the logic
 * to check for the delimiter character in order to simplify the code
 * in the main parse function.
 *
 * @param[in]  buf       Search buffer used to find the delimiter.
 * @param[in]  buf_len   Number of bytes to search for in buf.
 * @param[in]  delim     The delimiter to search for in buf.
 * @param[out] delim_pos If delimiter found in buf, return the delimiter
 *                       position in this parameter.
 * @retval 1 If the delimiter character found.
 * @retval 0 If the delimiter character not found.
 */
static BOOL
smtp_str_getdelimfd_search_delim(const char *const buf,
                                 size_t buf_len,
                                 int delim,
                                 size_t *const delim_pos) {
    *delim_pos = 0;

    for (size_t i = 0; i < buf_len; i++) {
	if(buf[i] == delim) {
	    *delim_pos = i;
	    return TRUE;
	}
    }
    return FALSE;
}

/**
 * Set the internal line buffer to the number of bytes specified.
 *
 * @param[in] gdfd     See @ref str_getdelimfd.
 * @param[in] copy_len Number of bytes to copy to the internal line buffer.
 * @retval  0 Successfully allocated and copied data over to the new
 *            line buffer.
 * @retval -1 Failed to allocate memory for the new line buffer.
 */
SMTP_LINKAGE int
smtp_str_getdelimfd_set_line_and_buf(struct str_getdelimfd *const gdfd,
                                     size_t copy_len) {
    if (gdfd->line != NULL) {
	ZFREE(gdfd->line);
	gdfd->line = NULL;
	gdfd->line_len = 0;
    }
    if (copy_len == 0)
	return 0;
    if ((gdfd->line = ZALLOC(copy_len + 1)) == NULL) {
	return -1;
    }
    blob_delete(gdfd->blob, gdfd->line, copy_len + 1);
    blob_seek(gdfd->blob, - (ssize_t)(copy_len + 1), SEEK_CUR);
    gdfd->line_len = copy_len;
    return 0;
}

/**
 * Free memory in the @ref str_getdelimfd data structure.
 *
 * @param[in] gdfd Frees memory stored in this socket parsing structure.
 */
SMTP_LINKAGE void
smtp_str_getdelimfd_free(struct str_getdelimfd *const gdfd) {
    blob_free(gdfd->blob);
    ZFREE(gdfd->line);
    gdfd->blob = NULL;
    gdfd->line = NULL;
    gdfd->line_len = 0;
}

/**
 * Free the @ref str_getdelimfd and return the @ref STRING_GETDELIMFD_ERROR
 * error code.
 *
 * @param[in] gdfd See @ref str_getdelimfd.
 * @return See @ref str_getdelim_retcode.
 */
static enum str_getdelim_retcode
smtp_str_getdelimfd_throw_error(struct str_getdelimfd *const gdfd) {
    smtp_str_getdelimfd_free(gdfd);
    return STRING_GETDELIMFD_ERROR;
}

/**
 * Read and parse a delimited string using a custom socket read function.
 *
 * This interface handles all of the logic for expanding the buffer,
 * parsing the delimiter in the buffer, and returning each "line"
 * to the caller for handling.
 *
 * @param[in] gdfd See @ref str_getdelimfd.
 * @return See @ref str_getdelim_retcode.
 */
SMTP_LINKAGE enum str_getdelim_retcode
smtp_str_getdelimfd(struct str_getdelimfd *const gdfd) {
    size_t	delim_pos;
    ssize_t	bytes_read = -1;

    if (gdfd->getdelimfd_read == NULL) {
	return STRING_GETDELIMFD_ERROR;
    }
    while (TRUE) {
	if (smtp_str_getdelimfd_search_delim((const char *)blob_data(gdfd->blob),
                                        blob_left(gdfd->blob),
                                        gdfd->delim,
                                        &delim_pos)) {
	    if (smtp_str_getdelimfd_set_line_and_buf(gdfd, delim_pos) < 0) {
		return smtp_str_getdelimfd_throw_error(gdfd);
	    }
	    return STRING_GETDELIMFD_NEXT;
	} else if (bytes_read == 0) {
	    if (smtp_str_getdelimfd_set_line_and_buf(gdfd, blob_left(gdfd->blob)) < 0) {
		return smtp_str_getdelimfd_throw_error(gdfd);
	    }
	    return STRING_GETDELIMFD_DONE;
	}
	bytes_read = (*gdfd->getdelimfd_read)(gdfd);
	if (bytes_read < 0) {
	    return smtp_str_getdelimfd_throw_error(gdfd);
	}
	blob_seek(gdfd->blob, -bytes_read, SEEK_CUR);
    }
}

/**
 * Search for all substrings in a string and replace each instance with a
 * replacement string.
 *
 * @param[in] search  Substring to search for in @p s.
 * @param[in] replace Replace each instance of the search string with this.
 * @param[in] s       Null-terminated string to search and replace.
 * @retval char* A dynamically allocated string with the replaced instances
 *               as described above. The caller must free the allocated
 *               memory when finished.
 * @retval NULL  Memory allocation failure.
 */
SMTP_LINKAGE char *
smtp_str_replace(const char *const search,
                 const char *const replace,
                 const char *const s) {
    size_t	search_len = strlen(search);
    size_t	replace_len = strlen(replace);
    size_t	slen = strlen(s);
    size_t	s_idx = 0;
    size_t	snew_idx = 0;
    char	* snew = NULL;

    if ((snew = ZSTRDUP(s)) == NULL)
	return NULL;
    if ((search_len < 1) || (slen < search_len))
	return snew;
    while (slen - s_idx >= search_len) {
	if (strncmp(&s[s_idx], search, search_len) == 0) {
	    char	* stmp;

	    if ((stmp = ZALLOC(strlen(snew) + replace_len + 1)) == NULL) {
		ZFREE(snew);
		return NULL;
	    }
	    s_idx += search_len;
	    memcpy(stmp, snew, snew_idx);
	    memcpy(&stmp[snew_idx], replace, replace_len);
	    snew_idx += replace_len;
	    memcpy(&stmp[snew_idx], &s[s_idx], slen - s_idx);
	    ZFREE(snew);
	    snew = stmp;
	} else {
	    snew_idx ++;
	    s_idx ++;
	}
    }
    return snew;
}

/**
 * Get the offset of the next whitespace block to process folding.
 *
 * If a string does not have whitespace before @p maxlen, then the index
 * will get returned past @p maxlen. Also returns the index of NULL character
 * if that fits within the next block. The caller must check for the NULL
 * index to indicate the last block. It will skip past any leading whitespace
 * even if that means going over maxlen.
 *
 * Examples:
 * @ref smtp_fold_whitespace_get_offset ("Subject: Test WS", 1/2/8/9/10/13) -> 8
 * @ref smtp_fold_whitespace_get_offset ("Subject: Test WS", 14/15) -> 13
 * @ref smtp_fold_whitespace_get_offset ("Subject: Test WS", 17/18) -> 16
 *
 * @param[in] s      String to get offset from.
 * @param[in] maxlen Number of bytes for each line in the string (soft limit).
 * @return Index in @p s.
 */
static size_t
smtp_fold_whitespace_get_offset(const char *const s,
                                unsigned int maxlen){
    size_t	i = 0, offset_i = 0;

    while((s[i] == ' ') || (s[i] == '\t')) {
	i ++;
    }

    while (s[i]) {
	if ((s[i] == ' ') || (s[i] == '\t')) {
	    do {
		i ++;
	    } while ((s[i] == ' ') || (s[i] == '\t'));
	    i --;
	    if ((i < maxlen) || !offset_i) {
		offset_i = i;
	    } else {
		break;
	    }
	}
	i ++;
    }
    if (!offset_i || (i < maxlen)) {
	offset_i = i;
    }
    return offset_i;
}

/**
 * Email header lines should have no more than 78 characters and must
 * not be more than 998 characters.
 */
#define SMTP_LINE_MAX 78

/**
 * Fold a line at whitespace characters.
 *
 * This function tries to keep the total number of characters per line under
 * @p maxlen, but does not guarantee this. For really long text with no
 * whitespace, the line will still extend beyond @p maxlen and possibly
 * beyond the RFC limit as defined in @ref SMTP_LINE_MAX. This is by design
 * and intended to keep the algorithm simpler to implement. Users sending
 * long headers with no space characters should not assume that will work,
 * but modern email systems may correctly process those headers anyways.
 *
 * Lines get folded by adding a [CR][LF] and then two space characters on the
 * beginning of the next line. For example, this Subject line:
 *
 * Subject: Email[WS][WS]Header
 *
 * Would get folded like this (assuming a small @p maxlen):
 *
 * Subject: Email[WS][CR][LF]
 * [WS][WS]Header
 *
 * @param[in] s      String to fold.
 * @param[in] maxlen Number of bytes for each line in the string (soft limit).
 *                   The minimum value of this parameter is 3 and it will get
 *                   forced to 3 if the provided value is less.
 * @retval char* Pointer to an allocated string with the contents split into
 *               separate lines. The caller must free this memory when done.
 * @retval NULL  Memory allocation failed.
 */
static char * smtp_fold_whitespace(const char *const s,
                     unsigned int maxlen) {
    const char	* SMTP_LINE_FOLD_STR = "\r\n ";
    size_t	end_slen = strlen(SMTP_LINE_FOLD_STR), s_i = 0, buf_i = 0, bufsz = 0;
    char	* buf = NULL;
    size_t	ws_offset;
    char	* buf_new;

    if (maxlen < 3) {
	maxlen = 3;
    }
    while (TRUE) {
	ws_offset = smtp_fold_whitespace_get_offset(&s[s_i], maxlen - 2);

	/* bufsz += ws_offset + end_slen + 1 */
	bufsz += ws_offset;
	bufsz += end_slen;
	bufsz ++;
	if ((buf_new = ZREALLOC(buf, bufsz)) == NULL) {
	    ZFREE(buf);
	    return NULL;
	}
	buf = buf_new;
	memcpy(&buf[buf_i], &s[s_i], ws_offset);
	buf[buf_i + ws_offset] = '\0';
	if (s[s_i + ws_offset] == '\0') {
	    break;
	}
	buf_i += ws_offset;
	strcat(&buf[buf_i], SMTP_LINE_FOLD_STR);
	buf_i += end_slen;
	/*                 WS */
	s_i += ws_offset + 1;
    }
    return buf;
}

/**
 * Parse a server response line into the @ref smtp_command data structure.
 *
 * @param[in]  line Server response string.
 * @param[out] cmd  Structure containing the server response data broken up
 *                  into its separate components.
 * @return See @ref smtp_result_code.
 */
static int
smtp_parse_cmd_line(const char * line,
                    struct smtp_command * cmd) {
    char		* ep;
    char		code_str[4];
    unsigned long int	ulcode;
    size_t		line_len = (SEMPTY(line) ? 0 : strlen(line));

    if (line_len < 5) {
	cmd->code = SMTP_INTERNAL_ERROR;
	cmd->more = 0;
	cmd->text = line;
	return cmd->code;
    }
    cmd->text = &line[4];
    memcpy(code_str, line, 3);
    code_str[3] = '\0';
    ulcode = strtoul(code_str, &ep, 10);
    if (*ep != '\0' || ulcode > SMTP_BEGIN_MAIL) {
	cmd->code = SMTP_INTERNAL_ERROR;
    } else {
	cmd->code = (enum smtp_result_code)ulcode;
    }
    if (line[3] == '-') {
	cmd->more = 1;
    } else {
	cmd->more = 0;
    }
    return cmd->code;
}

/**
 * Read a server response line.
 *
 * @param[in] smtp SMTP client context.
 * @return See @ref str_getdelim_retcode.
 */
static enum str_getdelim_retcode
smtp_getline(struct smtp *const smtp) {
  enum str_getdelim_retcode rc;

  errno = 0;
  rc = smtp_str_getdelimfd(&smtp->gdfd);
  if(errno == ENOMEM){
    smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    return rc;
  }
  else if(rc == STRING_GETDELIMFD_ERROR){
    smtp_status_code_set(smtp, SMTP_STATUS_RECV);
    return STRING_GETDELIMFD_ERROR;
  }

  if(smtp->gdfd.line_len > 0) {
    /* Remove the carriage-return character ('\r'). */
    smtp->gdfd.line[smtp->gdfd.line_len - 1] = '\0';
    smtp_puts_dbg(smtp, "Server", smtp->gdfd.line);
  }
  return rc;
}

/**
 * Loop through all of the server response lines until the last line, and
 * then return the status code from the last response line.
 *
 * @param[in] smtp SMTP client context.
 * @return See @ref smtp_result_code.
 */
static int
smtp_read_and_parse_code(struct smtp *const smtp) {
    struct smtp_command		cmd;
    enum str_getdelim_retcode	rc;

    do {
	rc = smtp_getline(smtp);
	if(rc == STRING_GETDELIMFD_ERROR) {
	    return SMTP_INTERNAL_ERROR;
	}
	smtp_parse_cmd_line(smtp->gdfd.line, &cmd);
    } while (rc != STRING_GETDELIMFD_DONE && cmd.more);
    return cmd.code;
}

static int
smtp_read_and_parse_ehlo_code(struct smtp * smtp) {
    struct smtp_command		cmd;
    enum str_getdelim_retcode	rc;

    do {
	rc = smtp_getline(smtp);
	if(rc == STRING_GETDELIMFD_ERROR) {
	    return SMTP_INTERNAL_ERROR;
	}
	if (smtp_parse_cmd_line(smtp->gdfd.line, &cmd) != SMTP_INTERNAL_ERROR) {
	    if (strstr(smtp->gdfd.line, "AUTH ") != NULL) {
		if (strstr(smtp->gdfd.line, "PLAIN") != NULL) {
		    smtp->auth_method = SMTP_AUTH_PLAIN;
		} else if (strstr(smtp->gdfd.line, "LOGIN") != NULL) {
		    smtp->auth_method = SMTP_AUTH_LOGIN;
		} else if (strstr(smtp->gdfd.line, "CRAM-MD5") != NULL) {
		    smtp->auth_method = SMTP_AUTH_CRAM_MD5;
		}
	    }
	    if (strstr(smtp->gdfd.line, "STARTTLS") != NULL) {
		smtp->connection_security = SMTP_SECURITY_STARTTLS;
	    }
	}
    } while (rc != STRING_GETDELIMFD_DONE && cmd.more);
    return cmd.code;
}


static BOOL smtp_tls_init(struct smtp * smtp, SOCK_T fd) {
    struct pstream_ops	* ops = NULL;

    if ((ops = pstream_init_ops_ssl(&smtp->ssl_ctx, fd)) == NULL) {
	log_err("Could not create SSL context for connection");
	return FALSE;
    }
    if ((smtp->tls = pstream_init(NULL, fd, ops, smtp->timeout_sec)) == NULL)
	return FALSE;
    return TRUE;
}

/**
 * Connect to the server using a standard TCP socket.
 *
 * This function handles the server name lookup to get an IP address
 * for the server, and then to connect to that IP using a normal TCP
 * connection.
 *
 * @param[in] smtp   SMTP client context.
 * @param[in] server Mail server name or IP address.
 * @param[in] port   Mail server port number.
 * @retval  0 Successfully connected to server.
 * @retval -1 Failed to connect to server.
 */
static SOCK_T smtp_connect(struct smtp * smtp,
             const char * server,
             uint16_t port, BOOL tls) {
    SOCK_T	fd = tcp_connect(server, port);

    if (fd < 0)
	return fd;

    if (tls) {
	if (smtp_tls_init(smtp, fd))
	    return fd;
    } else {
	if ((smtp->plain = pstream_init(NULL, fd, NULL, smtp->timeout_sec)) != NULL)
	    return fd;
    }
out:
    sock_close(fd);
    return -1;
}

/**
 * Send the EHLO command and parse through the responses.
 *
 * Ignores all of the server extensions that get returned. If a server
 * doesn't support an extension we need, then we should receive an error
 * later on when we try to use that extension.
 *
 * @param[in] smtp SMTP client context.
 * @return See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_ehlo(struct smtp * smtp) {
    if(smtp_puts(smtp, "EHLO smtp\r\n") == SMTP_STATUS_OK) {
	smtp_read_and_parse_ehlo_code(smtp);
    }
    return smtp->status_code;
}

/**
 * Authenticate using the PLAIN method.
 *
 *   1. Set the text to the following format: "\0<user>\0<password>",
 *      or as shown in the format string: "\0%s\0%s", email, password.
 *   2. Base64 encode the text from (1).
 *   3. Send the constructed auth text from (2) to the server:
 *      "AUTH PLAIN <b64><CR><NL>".
 *
 * @param[in] smtp SMTP client context.
 * @param[in] user SMTP account user name.
 * @param[in] pass SMTP account password.
 * @retval  0 Successfully authenticated.
 * @retval -1 Failed to authenticate.
 */
static int
smtp_auth_plain(struct smtp *const smtp,
                const char *const user,
                const char *const pass) {
    size_t	user_len = SEMPTY(user) ? 0 : strlen(user);
    size_t	pass_len = SEMPTY(pass) ? 0 : strlen(pass);
    size_t	login_len = 2 + user_len + pass_len;
    size_t	login_b64_len;
    char	* login_str, * login_b64, * login_send, * concat;

    /* login_len = 1 + user_len + 1 + pass_len */
    if ((login_str = ZALLOC(login_len)) == NULL) {
	return -1;
    }
    if (!SEMPTY(user))
	memcpy(&login_str[1], user, user_len);
    if (!SEMPTY(pass))
	memcpy(&login_str[user_len + 2], pass, pass_len);

    /* (2) */
    login_b64 = b64_encode(login_str, login_len);
    ZFREE(login_str);
    if (login_b64 == NULL) {
	return -1;
    }

    /* (3) */
    login_b64_len = strlen(login_b64);
    login_b64_len += 14;
    if ((login_send = ZALLOC(login_b64_len)) == NULL) {
	ZFREE(login_b64);
	return -1;
    }

    concat = stpncpy(login_send, "AUTH PLAIN ", login_b64_len);
    concat = stpncpy(concat, login_b64, login_b64_len - 11);
    stpncpy(concat, "\r\n", 2);

    ZFREE(login_b64);
    smtp_puts(smtp, login_send);
    ZFREE(login_send);
    if(smtp->status_code != SMTP_STATUS_OK) {
	return -1;
    }
    if(smtp_read_and_parse_code(smtp) != SMTP_AUTH_SUCCESS){
	return -1;
    }
    return 0;
}

/**
 * Authenticate using the LOGIN method.
 *
 *   1. Base64 encode the user name.
 *   2. Send string from (1) as part of the login:
 *      "AUTH LOGIN <b64_username><CR><NL>".
 *   3. Base64 encode the password and send that by itself on a separate
 *      line: "<b64_password><CR><NL>".
 *
 * @param[in] smtp SMTP client context.
 * @param[in] user SMTP account user name.
 * @param[in] pass SMTP account password.
 * @retval  0 Successfully authenticated.
 * @retval -1 Failed to authenticate.
 */
static int
smtp_auth_login(struct smtp *const smtp,
                const char *const user,
                const char *const pass){
    char	* b64_user, * b64_pass, * login_str, * concat;
    size_t b64_user_len;

    /* (1) */
    if (SEMPTY(user) || ((b64_user = b64_encode(user, 0)) == NULL)) {
	return -1;
    }

    /* (2) */
    b64_user_len = strlen(b64_user);
    b64_user_len += 14;
    if ((login_str = ZALLOC(b64_user_len)) == NULL) {
	ZFREE(b64_user);
	return -1;
    }
    concat = stpncpy(login_str, "AUTH LOGIN ", b64_user_len);
    concat = stpncpy(concat, b64_user, b64_user_len - 11);
    stpncpy(concat, "\r\n", 2);

    ZFREE(b64_user);
    smtp_puts(smtp, login_str);
    ZFREE(login_str);
    if (smtp->status_code != SMTP_STATUS_OK) {
	return -1;
    }

    if (smtp_read_and_parse_code(smtp) != SMTP_AUTH_CONTINUE) {
	return -1;
    }

    /* (3) */
    if (SEMPTY(pass) || ((b64_pass = b64_encode(pass, 0)) == NULL)) {
	return -1;
    }
    smtp_puts_terminate(smtp, b64_pass);
    ZFREE(b64_pass);
    if (smtp->status_code != SMTP_STATUS_OK) {
	return -1;
    }

    if (smtp_read_and_parse_code(smtp) != SMTP_AUTH_SUCCESS) {
	return -1;
    }
    return 0;
}

/**
 * Authenticate using the CRAM-MD5 method.
 *
 *   1. Send "AUTH CRAM-MD5<CR><NL>" to the server.
 *   2. Decode the base64 challenge response from the server.
 *   3. Do an MD5 HMAC on (2) using the account password as the key.
 *   4. Convert the binary data in (3) to lowercase hex characters.
 *   5. Construct the string: "<user> <(4)>".
 *   6. Encode (5) into base64 format.
 *   7. Send the final string from (6) to the server and check the response.
 *
 * @param[in] smtp SMTP client context.
 * @param[in] user SMTP account user name.
 * @param[in] pass SMTP account password.
 * @retval  0 Successfully authenticated.
 * @retval -1 Failed to authenticate.
 */
static int
smtp_auth_cram_md5(struct smtp *const smtp,
                   const char *const user,
                   const char *const pass){
    struct smtp_command	cmd;
    char		* challenge_decoded;
    size_t		challenge_decoded_len;
    const char		* key;
    int			key_len;
    unsigned char	hmac[EVP_MAX_MD_SIZE];
    unsigned int	hmac_len;
    unsigned char	* hmac_ret;
    char		* challenge_hex;
    size_t		user_len;
    size_t		challenge_hex_len;
    char		* auth_concat;
    char		* concat;
    size_t		auth_concat_len;
    char		* b64_auth;

    /* (1) */
    if (smtp_puts(smtp, "AUTH CRAM-MD5\r\n") != SMTP_STATUS_OK) {
	return -1;
    }
    if (smtp_getline(smtp) == STRING_GETDELIMFD_ERROR) {
	return -1;
    }
    if (smtp_parse_cmd_line(smtp->gdfd.line, &cmd) != SMTP_AUTH_CONTINUE) {
	return -1;
    }

  /* (2) */
    challenge_decoded = b64_decode(cmd.text);
    challenge_decoded_len = strlen(challenge_decoded);
    if (challenge_decoded_len == SIZE_MAX) {
	return -1;
    }

    /* (3) */
    key = pass;
    key_len = (int)strlen(pass);/*********************cast*/

    hmac_ret = HMAC_MD5(key,
                  key_len,
                  (const unsigned char *)challenge_decoded,
                  challenge_decoded_len,
                  hmac,
                  &hmac_len);
    ZFREE(challenge_decoded);
    if (hmac_ret == NULL) {
	return -1;
    }

    /* (4) */
    challenge_hex = hexify(NULL, 0, (const char *)hmac, hmac_len);
    if (challenge_hex == NULL) {
	return -1;
    }

    /* (5) */
    user_len = strlen(user);
    challenge_hex_len = strlen(challenge_hex);
    auth_concat_len = user_len + challenge_hex_len + 2;
    if ((auth_concat = ZALLOC(auth_concat_len)) == NULL) {
	ZFREE(challenge_hex);
	return -1;
    }
    concat = stpncpy(auth_concat, user, auth_concat_len);
    concat = stpncpy(concat , " ", 2);
    stpncpy(concat, challenge_hex, auth_concat_len - user_len - 2);
    ZFREE(challenge_hex);

    /* (6) */
    b64_auth = b64_encode(auth_concat, 0);
    ZFREE(auth_concat);
    if (b64_auth == NULL) {
	return -1;
    }

    /* (7) */
    smtp_puts_terminate(smtp, b64_auth);
    ZFREE(b64_auth);
    if (smtp->status_code != SMTP_STATUS_OK) {
	return -1;
    }

    if (smtp_read_and_parse_code(smtp) != SMTP_AUTH_SUCCESS) {
	return -1;
    }
    return 0;
}

/**
 * Perform a handshake with the SMTP server which includes optionally
 * setting up TLS and sending the EHLO greeting.
 *
 * At this point, the client has already connected to the SMTP server
 * through its socket connection. In this function, the client will:
 *   1. Optionally convert the connection to TLS (SMTP_SECURITY_TLS).
 *   2. Read the initial server greeting.
 *   3. Send an EHLO to the server.
 *   4. Optionally initiate STARTTLS and resend the EHLO
 *      (SMTP_SECURITY_STARTTLS).
 *
 * @param[in] smtp                SMTP client context.
 * @param[in] server              Server name or IP address.
 * @param[in] connection_security See @ref smtp_connection_security.
 * @return See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_initiate_handshake(struct smtp *const smtp,
                        const char *const server, uint16_t port, 
                        enum smtp_connection_security connection_security) {
    SOCK_T	fd;

    smtp_set_read_timeout(smtp, 60 * 5);
    if (port == 465)
	connection_security = SMTP_SECURITY_TLS;
    if((fd = smtp_connect(smtp, server, port, (BOOL)connection_security == SMTP_SECURITY_TLS)) < 0) {
	return smtp_status_code_set(smtp, SMTP_STATUS_CONNECT);
    }
    /* (2) */
    /* Get initial 220 message - 5 minute timeout. */
    if(smtp_getline(smtp) == STRING_GETDELIMFD_ERROR) {
	return smtp->status_code;
    }

    /* (3) */
    if(smtp_ehlo(smtp) != SMTP_STATUS_OK) {
	return smtp->status_code;
    }

    if (smtp->connection_security != 0)
	connection_security = smtp->connection_security;
    /* (4) */
    if (connection_security == SMTP_SECURITY_STARTTLS) {
	if(smtp_puts(smtp, "STARTTLS\r\n") != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
	if(smtp_read_and_parse_code(smtp) != SMTP_READY) {
	    return smtp_status_code_set(smtp, SMTP_STATUS_HANDSHAKE);
	}
	if(smtp_tls_init(smtp, fd) < 0) {
	    return smtp_status_code_set(smtp, SMTP_STATUS_HANDSHAKE);
	}
	if (smtp_ehlo(smtp) != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
    }

    return smtp->status_code;
}

/**
   Maximum size of an RFC 2822 date string.

   @verbatim
   Thu, 21 May 1998 05:33:29 -0700
   12345678901234567890123456789012
           10        20        30 32 (bytes)
   @endverbatim

   Add more bytes to the 32 maximum size to silence compiler warning on the
   computed UTF offset.
 */
#define SMTP_DATE_MAX_SZ (32 + 15)

/**
 * Determine if the header key has already been defined in this context.
 *
 * @param[in] smtp SMTP client context.
 * @param[in] key  Header key value to search for.
 * @retval 1 If the header already exists in this context.
 * @retval 0 If the header does not exist in this context.
 */
static int
smtp_header_exists(const struct smtp *const smtp,
                   const char *const key){
  if (dict_find_node_near(smtp->header_list, key, TRUE) != NULL) {
    return 1;
  }
  return 0;
}

/**
 * Minimum length of buffer required to hold the MIME boundary test:
 * mimeXXXXXXXXXX
 * 123456789012345
 * 1       10   15 bytes
 */
#define SMTP_MIME_BOUNDARY_LEN 15

/**
 * Generate the MIME boundary text field and store it in a user-supplied
 * buffer.
 *
 * For example:
 * mimeXXXXXXXXXX
 * where each X gets set to a pseudo-random uppercase ASCII character.
 *
 * This uses a simple pseudo-random number generator since we only care
 * about preventing accidental boundary collisions.
 *
 * @param[out] boundary Buffer that has at least SMTP_MIME_BOUNDARY_LEN bytes.
 */
static void
smtp_gen_mime_boundary(char *const boundary){
    unsigned int	seed = (unsigned int)time(NULL);

    srand(seed);
    strcpy(boundary, "mime");
    for (size_t i = 4; i < SMTP_MIME_BOUNDARY_LEN - 1; i++) {
	/* Modulo bias okay since we only need to prevent accidental collision. */
	boundary[i] = rand() % 26 + 'A';
    }
    boundary[SMTP_MIME_BOUNDARY_LEN - 1] = '\0';
}

/**
 * Print the MIME header and the MIME section containing the email body.
 *
 * @param[in] smtp     SMTP client context.
 * @param[in] boundary MIME boundary text.
 * @param[in] body_dd  Email body with double dots added at the
 *                     beginning of each line.
 * @return See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_print_mime_header_and_body(struct smtp *const smtp,
                                const char *const boundary,
                                const char *const body_dd){
    char	* concat;

    concat = os_sprintf("MIME-Version: 1.0\r\n"
			"Content-Type: multipart/mixed; boundary=%s\r\n"
			"\r\n"
			"Multipart MIME message.\r\n"
			"--%s\r\n"
			"Content-Type: text/plain; charset=\"UTF-8\"\r\n"
			"\r\n%s\r\n\r\n"
			, boundary, boundary, body_dd);
    if (concat == NULL) {
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    }
    smtp_puts(smtp, concat);
    ZFREE(concat);
    return smtp->status_code;
}

/**
 * Prints double hyphen on both sides of the MIME boundary which indicates
 * the end of the MIME sections.
 *
 * @param[in] smtp     SMTP client context.
 * @param[in] boundary MIME boundary text.
 * @return See @ref smtp_status_code and @ref smtp_puts.
 */
static enum smtp_status_code
smtp_print_mime_end(struct smtp *const smtp,
                    const char *const boundary){
    char	mime_end[2 + SMTP_MIME_BOUNDARY_LEN + 4 + 1];

    snprintf(mime_end, sizeof(mime_end), "--%s--\r\n", boundary);
    return smtp_puts(smtp, mime_end);
}

/**
 * Send the main email body to the SMTP server.
 *
 * This includes the MIME sections for the email body and attachments.
 *
 * @param[in] smtp     SMTP client context.
 * @param[in] body_dd  Email body with double dots added at the
 *                     beginning of each line.
 * @return See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_print_mime_email(struct smtp *const smtp,
                      const char *const body_dd){
    char	boundary[SMTP_MIME_BOUNDARY_LEN];
    size_t	i;

    smtp_gen_mime_boundary(boundary);

    if (smtp_print_mime_header_and_body(smtp, boundary, body_dd) != SMTP_STATUS_OK) {
	return smtp->status_code;
    }

    FOREACH_IN_DICT(smtp->attachment_list, i) {
	val_t	* attachment = V_CHILD(smtp->attachment_list, i);

	if (smtp_print_mime_attachment(smtp,
                                  boundary,
                                  attachment) != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
    }
    return smtp_print_mime_end(smtp, boundary);
}

/**
 * Print the email data provided by the user without MIME formatting.
 *
 * @param[in,out] smtp     SMTP client context.
 * @param[in]     body_dd  Email body with double dots added at the
 *                         beginning of each line.
 * @return                 See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_print_nomime_email(struct smtp *const smtp,
                        const char *const body_dd){
    return smtp_puts_terminate(smtp, body_dd);
}

/**
 * Send the email body to the mail server.
 *
 * @param[in,out] smtp SMTP client context.
 * @param[in]     body Email body text.
 * @return             See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_print_email(struct smtp *const smtp,
                 const char *const body) {
    char	* body_double_dot;

  /*
   * Insert an extra dot for each line that begins with a dot. This will
   * prevent data in the body parameter from prematurely ending the DATA
   * segment.
   */
  if((body_double_dot = smtp_str_replace("\n.", "\n..", body)) == NULL){
    return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
  }

  if(smtp_header_exists(smtp, "Content-Type")){
    smtp_print_nomime_email(smtp, body_double_dot);
  }
  else{
    smtp_print_mime_email(smtp, body_double_dot);
  }
  free(body_double_dot);
  return smtp->status_code;
}

/**
 * Convert a header into an RFC 5322 formatted string and send it to the
 * SMTP server.
 *
 * This will adding proper line wrapping and indentation for long
 * header lines.
 *
 * @param[in] smtp   SMTP client context.
 * @param[in] header See @ref smtp_header.
 * @return See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_print_header(struct smtp *const smtp, val_t * header) {
    char	* concat, * valptr,  * header_fmt;

    if (header == NULL) {
	return smtp->status_code;
    }
    valptr = val_as_str_simple(header);
    if (valptr == NULL) {
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    }
    concat = os_sprintf("%s: %s", header->key, valptr);
    ZFREE(valptr);
    if (concat == NULL) {
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    }
    header_fmt = smtp_fold_whitespace(concat, SMTP_LINE_MAX);
    ZFREE(concat);
    if (header_fmt == NULL) {
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    }
    smtp_puts_terminate(smtp, header_fmt);
    ZFREE(header_fmt);
    return smtp->status_code;
}

/**
 * Send envelope MAIL FROM or RCPT TO header address.
 *
 * Examples:
 * MAIL FROM:<mail\@example.com>
 * RCPT TO:<mail\@example.com>
 *
 * @param[in] smtp    SMTP client context.
 * @param[in] header  Either "MAIL FROM" or "RCPT TO".
 * @param[in] address See @ref smtp_address -> email field.
 * @return See @ref smtp_status_code.
 */
static enum smtp_status_code
smtp_mail_envelope_header(struct smtp *const smtp,
                          const char *const header,
                          val_t *const address){
    const char	* const SMTPUTF8 = " SMTPUTF8", * smtputf8_opt = "";
    char	* email, * envelope_address;
    int		i;

    FOREACH_IN_DICT(address, i) {
	val_t	* child = V_CHILD(address, i);

    email = val_as_str_simple(child);
//    if (SEMPTY(email)) {
//	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
//    }
    if (utf8_str_has(email)) {
	smtputf8_opt = SMTPUTF8;
    }
    envelope_address = os_sprintf("%s:%s <%s>%s\r\n", header, (SEMPTY(email) ? "" : email), child->key, smtputf8_opt);
    ZFREE(email);
    if (envelope_address == NULL) {
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    }
    smtp_puts(smtp, envelope_address);
    ZFREE(envelope_address);
    if (smtp->status_code != SMTP_STATUS_OK) {
	return smtp->status_code;
    }
    smtp_read_and_parse_code(smtp);
    if (smtp->status_code != SMTP_STATUS_OK) {
	break;
    }
    }
    return smtp->status_code;
}

/**
 * Special flag value for the SMTP context used to determine if the initial
 * memory allocation failed to create the context.
 */
#define SMTP_FLAG_INVALID_MEMORY (enum smtp_flag)(0xFFFFFFFF)

PLEX enum smtp_status_code
smtp_open(const char * server,
          int port,
          enum smtp_connection_security connection_security,
          enum smtp_flag flags,
          const char * cafile, 
          struct smtp **smtp) {
    struct smtp *snew;

    if((snew = ZALLOC(sizeof(*snew))) == NULL){
	return SMTP_STATUS_NOMEM;
    }
    *smtp = snew;

    snew->flags                = flags;
    snew->gdfd.delim           = '\n';
    snew->gdfd.getdelimfd_read = smtp_str_getdelimfd_read;
    snew->gdfd.data       = snew;
    snew->gdfd.blob       = blob_new();
    if (!ssl_init_app_ctx(&((*smtp)->ssl_ctx), cafile, NULL, NULL, TRUE, (BOOL)(flags & SMTP_NO_CERT_VERIFY)))
	return smtp_status_code_set(*smtp, SMTP_STATUS_SSL);

#ifndef __WINDOWS__
    signal(SIGPIPE, SIG_IGN);
#endif /* !(SMTP_IS_WINDOWS) */

    if (smtp_initiate_handshake(snew, server, port, 
                             connection_security) != SMTP_STATUS_OK)
	return smtp_status_code_set(*smtp, SMTP_STATUS_HANDSHAKE);
    return snew->status_code;
}

PLEX enum smtp_status_code
smtp_auth(struct smtp *const smtp,
          enum smtp_authentication_method auth_method,
          const char *const user,
          const char *const pass) {
    int auth_rc;

    if (smtp->status_code != SMTP_STATUS_OK)
	return smtp->status_code;

    if (smtp->auth_method != 0)
	auth_method = smtp->auth_method;
    if (auth_method == SMTP_AUTH_PLAIN) {
	auth_rc = smtp_auth_plain(smtp, user, pass);
    } else if (auth_method == SMTP_AUTH_LOGIN) {
	auth_rc = smtp_auth_login(smtp, user, pass);
    } else if (auth_method == SMTP_AUTH_CRAM_MD5) {
	auth_rc = smtp_auth_cram_md5(smtp, user, pass);
    } else if (auth_method == SMTP_AUTH_NONE) {
	auth_rc = 0;
    } else {
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
    }
    if (auth_rc < 0) {
	return smtp_status_code_set(smtp, SMTP_STATUS_AUTH);
    }
    return smtp->status_code;
}

PLEX enum smtp_status_code smtp_mail(struct smtp * smtp, const char *body) {
    int		i;
    val_t	* address;

    if(smtp->status_code != SMTP_STATUS_OK) {
	return smtp->status_code;
    }

    smtp_set_read_timeout(smtp, 60 * 5);
    address = smtp->address_list[SMTP_ADDRESS_FROM];
    if (address != NULL) {
	if(smtp_mail_envelope_header(smtp,
                                   "MAIL FROM",
                                   address) != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
    } else {
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
    }

    for (i = SMTP_ADDRESS_FROM + 1; i < SMTP_ADDRESS_LAST; i++) {
	address = smtp->address_list[i];
	if (smtp_mail_envelope_header(smtp,
                                   "RCPT TO",
                                   address) != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
    }

    /* DATA timeout 2 minutes. */
    smtp_set_read_timeout(smtp, 60 * 2);

    if (smtp_puts(smtp, "DATA\r\n") != SMTP_STATUS_OK)
	return smtp->status_code;

    /* 354 response to DATA must get returned before we can send the message. */
    if (smtp_read_and_parse_code(smtp) != SMTP_BEGIN_MAIL) {
	return smtp_status_code_set(smtp, SMTP_STATUS_SERVER_RESPONSE);
    }

    if (!smtp_header_exists(smtp, "Date")) {
	char	date[SMTP_DATE_MAX_SZ];

	if(time_to_rfc2822str(date, sizeof(date), time(NULL)) == NULL) {
	    return smtp_status_code_set(smtp, SMTP_STATUS_DATE);
	}
	if (smtp_header_add(smtp, "Date", date) != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
    }

    /* DATA block timeout 3 minutes. */
    smtp_set_read_timeout(smtp, 60 * 3);

    if (smtp_append_address_to_header(smtp,
                                   SMTP_ADDRESS_FROM,
                                   "From") != SMTP_STATUS_OK ||
     smtp_append_address_to_header(smtp,
                                   SMTP_ADDRESS_TO,
                                   "To") != SMTP_STATUS_OK ||
     smtp_append_address_to_header(smtp,
                                   SMTP_ADDRESS_CC,
                                   "Cc") != SMTP_STATUS_OK) {
	return smtp->status_code;
    }

    FOREACH_IN_DICT(smtp->header_list, i) {
	if (smtp_print_header(smtp, V_CHILD(smtp->header_list, i)) != SMTP_STATUS_OK) {
	    return smtp->status_code;
	}
    }

    if (smtp_print_email(smtp, body)) {
	return smtp->status_code;
    }

    /* End of DATA segment. */
    if (smtp_puts(smtp, ".\r\n") != SMTP_STATUS_OK) {
	return smtp->status_code;
    }

    /* DATA termination timeout 250 return code - 10 minutes. */
    smtp_set_read_timeout(smtp, 60 * 10);
    if (smtp_read_and_parse_code(smtp) != SMTP_DONE) {
	return smtp_status_code_set(smtp, SMTP_STATUS_SERVER_RESPONSE);
    }

    return smtp->status_code;
}

PLEX enum smtp_status_code smtp_close(struct smtp * smtp){
    enum smtp_status_code	status_code = smtp->status_code;

    if(smtp->flags == SMTP_FLAG_INVALID_MEMORY) {
	return status_code;
    }

    if ((smtp->plain != NULL) || (smtp->tls != NULL)) {
	/*
	 * Do not error out if this fails because we still need to free the
	 * SMTP client resources.
	 */
	smtp->status_code = SMTP_STATUS_OK;
	smtp_puts(smtp, "QUIT\r\n");

	if (smtp->tls != NULL) {
	    pstream_close(smtp->tls);
	    smtp->tls = NULL;
	}
	if (smtp->plain != NULL) {
	    pstream_close(smtp->plain);
	    smtp->plain = NULL;
	}
	if(smtp->status_code == SMTP_STATUS_OK) {
	    smtp_status_code_set(smtp, SMTP_STATUS_CLOSE);
	}
    }

    smtp_header_clear_all(smtp);
    smtp_address_clear_all(smtp);
    smtp_attachment_clear_all(smtp);
    if(status_code == SMTP_STATUS_OK) {
	status_code = smtp->status_code;
    }
    ZFREE(smtp);

    return status_code;
}

PLEX struct uri_t * smtp_uri_parse(const char * uric, SMTP_CONNECTION_SECURITY * security_flag) {
    struct uri_t		* uri = NULL;

    if (!SEMPTY(uric)) {
	if ((uri = uri_parse(uric)) != NULL) {
	    if (uri->uri_port <= 0) {
		if (strcmp("smtps", uri->uri_scheme) == 0) {
		    uri->uri_port = 465;
		    if (security_flag != NULL)
			*security_flag = SMTP_SECURITY_TLS;
		} else if (strcmp("smtp", uri->uri_scheme) == 0) {
		    uri->uri_port = 25;
		} else if (strcmp("submission", uri->uri_scheme) == 0) {
		    uri->uri_port = 587;
		}
	    }
	}
    }
    return uri;
}

PLEX BOOL smtp_send_blob(const char * url, const char * fromaddr, const char * from,  const char * toaddr, const char * to, const char * subj, const blob_t * template) {
    BOOL			result = FALSE;
    val_t			* root = NULL;
    blob_t			* envelope = NULL;
    struct smtp			* smtp;
    struct uri_t		* uri;
    SMTP_CONNECTION_SECURITY	security_flag = SMTP_SECURITY_STARTTLS;

    if (SEMPTY(url) || SEMPTY(fromaddr) || SEMPTY(from) || SEMPTY(toaddr) || SEMPTY(to) || SEMPTY(subj) || (template == NULL))
	return result;
    if ((root = val_new(NULL)) != NULL) {
	val_t * v = val_new("from");

	if (v == NULL)
	    return result;
	val_from_str(v, from);
	val_dict_add(root, v);
	if ((v = val_new("to")) == NULL)
	    goto ssb_end;
	val_from_str(v, to);
	val_dict_add(root, v);
	if ((v = val_new("subject")) == NULL)
	    goto ssb_end;
	val_from_str(v, subj);
	val_dict_add(root, v);
	if ((envelope = blob_xlat(envelope, (const char *)blob_data(template), blob_left(template), NULL, NULL, root)) == NULL)
	    goto ssb_end;
    } else return result;

    if ((uri = smtp_uri_parse(url, &security_flag)) == NULL)
	goto ssb_end;
    if (uri->uri_port == 465)
	security_flag = SMTP_SECURITY_TLS;
    smtp_open(uri->uri_host,
            uri->uri_port,
            security_flag,
            SMTP_NO_CERT_VERIFY|SMTP_DEBUG,
            NULL,
            &smtp);

    smtp_auth(smtp,
            (SEMPTY(uri->uri_password) ? SMTP_AUTH_NONE : SMTP_AUTH_PLAIN),
            uri->uri_username,
            uri->uri_password);

    smtp_header_add(smtp, "Subject", subj);
    blob_rewind(envelope);
    smtp_mail(smtp, (const char *)blob_data(envelope));
    result = (BOOL)(smtp_close(smtp) == SMTP_STATUS_OK);
    uri_free(uri);
ssb_end:
    if (root != NULL)
	val_free(root);
    if (envelope != NULL)
	blob_free(envelope);
    return result;
}

PLEX BOOL smtp_file(const char * url, const char * fromaddr, const char * from,  const char * toaddr, const char * to, const char * subj, const char * filename, BOOL del) {
    BOOL	res = FALSE;
    blob_t	* template = fs_read_to_blob(NULL, filename);

    if (template != NULL) {
	blob_rewind(template);
	res = smtp_send_blob(url, fromaddr, from,  toaddr, to, subj, template);
	blob_free(template);
    }
    if (del)
	unlink(filename);
    return res;
}
