#include "internal.h"

ssize_t smtp_str_getdelimfd_read(struct str_getdelimfd * p) {
    struct smtp	* smtp = p->data;

    if (p->blob == NULL) {
	if ((p->blob = blob_new()) == NULL) {
	    return -1;
	}
    }
    return pstream_read_blob((smtp->tls != NULL ? smtp->tls : smtp->plain), p->blob);
}

/**
 * Set the timeout for the next socket read operation.
 *
 * @param[in] smtp    SMTP client context.
 * @param[in] seconds Timeout in seconds.
 */
void smtp_set_read_timeout(struct smtp * smtp, long seconds) {
    if (smtp->tls != NULL)
	smtp->tls->timeout = seconds * 1000;
    if (smtp->plain != NULL)
	smtp->plain->timeout = seconds * 1000;
}

/**
 * This function gets called by the @ref smtp_str_getdelimfd interface when it
 * needs to read in more data.
 *
 * It reads using either the plain socket connection if encryption not
 * enabled, or it reads using OpenSSL if it has an active TLS connection.
 *
 * @param[in]  gdfd  See @ref str_getdelimfd.
 * @param[out] buf   Pointer to buffer for storing bytes read.
 * @param[in]  count Maximum number of bytes to try reading.
 * @retval >=0 Number of bytes read.
 * @retval -1  Failed to read from the socket.
 */
ssize_t smtp_read(const struct smtp * smtp,
                         void *buf,
                         size_t count) {
    return pstream_read((smtp->tls ? smtp->tls : smtp->plain), buf, count);
}

/**
 * Prints communication between the client and server to stderr only if
 * the debug flag has been set.
 *
 * @param[in] smtp   SMTP client context.
 * @param[in] prefix Print this prefix before the main debug line text.
 * @param[in] str    Debug text to print out.
 */
void smtp_puts_dbg(struct smtp * smtp,  const char * prefix,  const char * str) {
    char	* sdup;
    size_t	i;

    if (smtp->flags & SMTP_DEBUG) {
	if ((sdup = ZSTRDUP(str)) == NULL)
	     return;

	/* Remove carriage return and newline when printing to stderr. */
	for (i = 0; sdup[i]; i++) {
	    if ((sdup[i] == '\r') || (sdup[i] == '\n'))
		sdup[i] = ' ';
	}

	if (fprintf(stderr, "[smtp %s]: %s\n", prefix, sdup) < 0) {
	    /* Do not care if this fails. */
	}
	ZFREE(sdup);
    }
}

/**
 * Send data to the SMTP server.
 *
 * Writes a buffer of length len into either the unencrypted TCP socket or
 * the TLS encrypted socket, depending on the current underlying mode of
 * the socket.
 *
 * @param[in] smtp SMTP client context.
 * @param[in] buf Data to send to the SMTP server.
 * @param[in] len Number of bytes in buf.
 * @return See @ref smtp_status_code.
 */
SMTP_STATUS_CODE smtp_write(struct smtp * smtp, const char * buf, size_t len) {

    smtp_puts_dbg(smtp, "Client", buf);

    if (pstream_write((smtp->tls ? smtp->tls : smtp->plain), buf, len) != (ssize_t)len)
	return smtp_status_code_set(smtp, SMTP_STATUS_SEND);
    return smtp->status_code;
}

/**
 * Send a null-terminated string to the SMTP server.
 *
 * @param[in] smtp SMTP client context.
 * @param[in] s    Null-terminated string to send to the SMTP server.
 * @return See @ref smtp_status_code and @ref smtp_write.
 */
SMTP_STATUS_CODE smtp_puts(struct smtp * smtp, const char * s) {
    return smtp_write(smtp, s, strlen(s));
}

/**
 * Same as @ref smtp_puts except this function also appends the line
 * terminating carriage return and newline bytes at the end of the string.
 *
 * @param[in] smtp SMTP client context.
 * @param[in] s    Null-terminated string to send to the SMTP server.
 * @return See @ref smtp_status_code and @ref smtp_puts.
 */
SMTP_STATUS_CODE smtp_puts_terminate(struct smtp * smtp, const char * s) {
    SMTP_STATUS_CODE	rc;

    if ((rc = smtp_puts(smtp, s)) == SMTP_STATUS_OK)
	rc = smtp_puts(smtp, "\r\n");
    return rc;
}

static inline SMTP_STATUS_CODE __smtp_printf_p(struct smtp * smtp, const char * fmt, va_list lst) {
	char	buffer[BUFSIZ];
	ssize_t	res = 0;

	if ((res = vsnprintf(buffer, sizeof(buffer), fmt, lst)) > 0)
	    return smtp_write(smtp, (const void *)buffer, (size_t)res);
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
}

SMTP_STATUS_CODE smtp_printf(struct smtp * smtp, const char * fmt, ...) {
	va_list			ap;
	SMTP_STATUS_CODE	res;

	if ((smtp == NULL) || SEMPTY(fmt))
	    return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
	va_start(ap, fmt);
	res = __smtp_printf_p(smtp, fmt, ap);
	va_end(ap);
	return res;
}
