#include "internal.h"

/**
 * Validate characters in the email header key.
 *
 * Must consist only of printable US-ASCII characters except colon.
 *
 * @param[in] key Header key to validate.
 * @retval TRUE Successful validation.
 * @retval FALSE Failed to validate.
 */
static BOOL smtp_header_key_validate(const char * key) {
    unsigned char	uc;
    size_t		i, keylen = strlen(key);

    if (keylen < 1)
	return FALSE;

    for (i = 0; i < keylen; i++) {
	uc = (unsigned char)key[i];
	if ((uc <= ' ') || (uc > 126) || (uc == ':'))
	    return FALSE;
    }
    return TRUE;
}

/**
 * Validate characters in the email header contents.
 *
 * Must consist only of printable character, space, or horizontal tab.
 *
 * @param[in] value Header value to validate.
 * @retval TRUE Successful validation.
 * @retval FALSE Failed to validate.
 */
static BOOL smtp_header_value_validate(const char * value) {
    size_t		i;
    unsigned char	uc;

    for(i = 0; value[i]; i++) {
	uc = (unsigned char)value[i];
	if (((uc < ' ') || (uc > 126)) && (uc != '\t') && (uc < 0x80)) /* Allow UTF-8 byte sequence. */
	    return FALSE;
    }
    return TRUE;
}

PLEX SMTP_STATUS_CODE smtp_header_add(struct smtp * smtp,
                const char * key,
                const char * value) {
    val_t	* header_list, * new_header;

    if (smtp->status_code != SMTP_STATUS_OK)
	return smtp->status_code;

    if (!smtp_header_key_validate(key))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);

    if (!SEMPTY(value) && !smtp_header_value_validate(value))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);

    header_list = smtp->header_list;
    if (header_list == NULL) {
	if ((header_list = val_new(NULL)) == NULL)
	    return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
	smtp->header_list = header_list;
    }
    if ((new_header = val_new(key)) == NULL)
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);

    val_from_str(new_header, value);
    val_dict_add(header_list, new_header);
    return smtp->status_code;
}

PLEX void smtp_header_clear_all(struct smtp *const smtp){
    if (smtp->header_list != NULL) {
	val_free(smtp->header_list);
	smtp->header_list = NULL;
    }
}

/**
 * Take a FROM, TO, and CC address and add it into the email header list.
 *
 * The following example shows what the final header might look like when
 * the client sends an email to two CC addresses:
 * Cc: mail1\@example.com, mail2\@example.com
 *
 * @param[in] smtp         SMTP client context.
 * @param[in] address_type See @ref smtp_address_type.
 * @param[in] key          Header key value, for example, To From Cc.
 * @return See @ref smtp_status_code.
 */
SMTP_STATUS_CODE smtp_append_address_to_header(struct smtp * smtp, SMTP_ADDRESS_TYPE address_type, const char * key) {
    val_t	* address_list;
    char	header_value[SMTP_LINE_MAX], * ptr = header_value;
    size_t	num_address_in_header, remain, i;
    val_t	* val;
    const char	* cptr;

    if (smtp->status_code != SMTP_STATUS_OK)
	return smtp->status_code;

    if ((address_type < 0) || (address_type >= SMTP_ADDRESS_LAST))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);

    address_list = smtp->address_list[address_type];
    if (address_list == NULL)
	return smtp->status_code;

    header_value[0] = 0;
    num_address_in_header = 0;
    remain = sizeof(header_value);

    FOREACH_IN_DICT(address_list, i) {
	val = V_CHILD(address_list, i);
	cptr = val_as_str_ptr(val);

      /*
       *                   ', "'      NAME     '" <'      EMAIL     >  \0
       * header_value_sz +=  3  +  name_slen  +  3  +  email_slen + 1 + 1
       */
	snprintf(ptr, remain, "%s%s%s%s<%s>", ((num_address_in_header > 0) ? ", " : ""), (SEMPTY(cptr) ? "" : "\""), (SEMPTY(cptr) ? "" : cptr), (SEMPTY(cptr) ? "" : "\" "), val->key);
	num_address_in_header ++;
	remain = strlen(header_value);
	ptr = header_value + remain;
	remain = sizeof(header_value) - remain;
    }

    if (!SEMPTY(header_value))
	smtp_header_add(smtp, key, header_value);
    return smtp->status_code;
}
