/**
 * @file
 * @brief POSIX mailx utility.
 * @author James Humphrey (mail@somnisoft.com)
 * @version 1.00
 *
 * Implementation of POSIX mailx utility in send mode.
 *
 * mailx [-s subject] [[-S option]...] [[-a attachment]...] address...
 *
 * This software has been placed into the public domain using CC0.
 */

/**
 * Required on some POSIX systems to include some standard functions.
 */
#define _POSIX_C_SOURCE 200809L

#include <plib/smtp.h>
#include <plib/fs.h>

/**
 * Stores the to and from email addresses.
 */
struct mailx_address{
  /**
   * See @ref smtp_address_type.
   */
  enum smtp_address_type address_type;

  /**
   * Email address.
   */
  char email[1000];
};

/**
 * The attachment name and path stored for each attachment to send to the
 * recipient.
 */
struct mailx_attachment{
  /**
   * File name for this attachment to display to the recipient.
   */
  char name[1000];

  /**
   * Local file path pointing to the attachment to send.
   */
  char path[1000];
};

/**
 * The mailx context structure containing the parameters for setting up the
 * SMTP connection and sending the email.
 */
struct mailx{
  /**
   * SMTP client context.
   */
  struct smtp *smtp;

  /**
   * Email subject line.
   */
  const char *subject;

  /**
   * Email body text.
   */
  char *body;

  /**
   * SMTP server name or IP address.
   */
  char *server;

  /**
   * SMTP server port number.
   */
  int port;

  /**
   * SMTP account user name used for authenticating.
   */
  char *user;

  /**
   * SMTP account password used for authenticating.
   */
  char *pass;

  /**
   * From email address or name.
   */
  char *from;

  /**
   * Determine if using a TLS encrypted connection or plain socket.
   */
  enum smtp_connection_security connection_security;

  /**
   * SMTP user account authentication method.
   */
  enum smtp_authentication_method auth_method;

  /**
   * Miscellaneous control flags for smtp-lib.
   *
   * See @ref smtp_flag for more details.
   */
  enum smtp_flag smtp_flags;

    val_t	* attachments;
};

/**
 * Read the entire contents of a file stream and store the data into a
 * dynamically allocated buffer.
 *
 * @param[in]  stream     File stream already opened by the caller.
 * @param[out] bytes_read Number of bytes stored in the return buffer.
 * @retval char* A dynamically allocated buffer which contains the entire
 *               contents of @p stream. The caller must free this memory
 *               when done.
 * @retval NULL Memory allocation or file read error.
 */
static char *
smtp_ffile_get_contents(FILE *stream,
                        size_t *bytes_read){
  char *read_buf;
  size_t bufsz;
  char *new_buf;
  const size_t BUFSZ_INCREMENT = 512;

  read_buf = NULL;
  bufsz = 0;

  if(bytes_read){
    *bytes_read = 0;
  }

  do{
    size_t bytes_read_loop;
    if((new_buf = realloc(read_buf, bufsz + BUFSZ_INCREMENT)) == NULL){
      free(read_buf);
      return NULL;
    }
    read_buf = new_buf;
    bufsz += BUFSZ_INCREMENT;

    bytes_read_loop = fread(&read_buf[bufsz - BUFSZ_INCREMENT],
                            sizeof(char),
                            BUFSZ_INCREMENT,
                            stream);
    if(bytes_read){
      *bytes_read += bytes_read_loop;
    }
    if(ferror(stream)){
      free(read_buf);
      return NULL;
    }
  } while(!feof(stream));

  return read_buf;
}

/**
 * Append this email to the list of email addresses to send to.
 *
 * @param[in] mailx        Append the email address into this mailx context.
 * @param[in] address_type See @ref smtp_address_type.
 * @param[in] email        Email address to send to.
 */
static void
mailx_address_append(struct mailx *const mailx,
                     enum smtp_address_type address_type,
                     const char *const email){
    smtp_address_add(mailx->smtp, address_type, email, NULL);
}

/**
 * Send the email using the configuration options in the @p mailx context.
 *
 * @param[in] mailx Email context.
 */
static void
mailx_send(struct mailx *const mailx){
  int rc;

  rc = smtp_auth(mailx->smtp,
            mailx->auth_method,
            mailx->user,
            mailx->pass);

  if(rc != SMTP_STATUS_OK){
    log_err("%s", smtp_status_code_errstr(rc));
    goto quit;
  }

  rc = smtp_header_add(mailx->smtp, "Subject", mailx->subject);

  if(rc != SMTP_STATUS_OK){
    log_err("%s", smtp_status_code_errstr(rc));
    goto quit;

  }
  rc = smtp_mail(mailx->smtp, mailx->body);

  if(rc != SMTP_STATUS_OK){
    log_err("%s", smtp_status_code_errstr(rc));
    goto quit;
  }

quit:
  rc = smtp_close(mailx->smtp);

  if(rc != SMTP_STATUS_OK){
    log_errx(1, "%s", smtp_status_code_errstr(rc));
  }
}

/**
 * Attach a file to the @p mailx context.
 *
 * @param[in] mailx    Store the attachment details into this mailx context.
 * @param[in] filename File name to display to the recipient.
 * @param[in] path     Local path of file to attach.
 */
static void
mailx_append_attachment(struct mailx *const mailx,
                        const char *const filename,
                        const char *const path){

    val_t	* new_attachment;
    if (filename == NULL || path == NULL) {
	log_errx(1, "must provide attachment with valid name:path");
    }

    if (mailx->attachments == NULL) {
	if ((mailx->attachments = val_new(NULL)) == NULL) {
	    log_errx(1, "Could not allocate memory for attachment variable");
	}
    }

    if (!smtp_attachment_validate_name(filename)) {
	log_errx(1, "Attachment has invalid filename (%s)", filename);
    }
    if (!fs_exists(path)) {
	log_errx(1, "Attachment file does not exist at provided locaion (%s)", path);
    }
    if ((new_attachment = val_new(filename)) == NULL) {
	log_errx(1, "Could not allocate memory for attachment variable");
    }

    val_file(new_attachment,  path, NULL);
    val_dict_add(mailx->attachments, new_attachment);
}

/**
 * Parse the file name and path and attach it to the @p mailx context.
 *
 * @param[in] mailx      Store the attachment details into this mailx context.
 * @param[in] attach_arg String with format: 'filename:filepath'.
 */
static void
mailx_append_attachment_arg(struct mailx *const mailx,
                            const char *const attach_arg){
  char *attach_arg_dup;
  char *filename;
  char *filepath;

  if((attach_arg_dup = strdup(attach_arg)) == NULL){
    log_errx(1, "strdup: %s", attach_arg);
  }

  filename = strtok(attach_arg_dup, ":");
  filepath = strtok(NULL, ":");

  mailx_append_attachment(mailx, filename, filepath);

  free(attach_arg_dup);
}

/**
 * Parses the -S option which contains a key/value pair separated by an '='
 * character.
 *
 * @param[in] mailx  Store the results of the option parsing into the relevant
 *                   field in this mailx context.
 * @param[in] option String containing key/value option to parse.
 */
static void
mailx_parse_smtp_option(struct mailx *const mailx,
                        const char *const option){
  char *optdup;
  char *opt_key;
  char *opt_value;
  int rc;

  rc = 0;

  if((optdup = strdup(option)) == NULL){
    log_errx(1, "strdup: option: %s", option);
  }

  if((opt_key = strtok(optdup, "=")) == NULL){
    log_errx(1, "strtok: %s", optdup);
  }

  opt_value = strtok(NULL, "=");

  if(strcmp(opt_key, "smtp-security") == 0){
    if(strcmp(opt_value, "none") == 0){
      mailx->connection_security = SMTP_SECURITY_NONE;
    }
    else if(strcmp(opt_value, "tls") == 0){
      mailx->connection_security = SMTP_SECURITY_TLS;
    }
    else if(strcmp(opt_value, "starttls") == 0){
      mailx->connection_security = SMTP_SECURITY_STARTTLS;
    }
    else{
      rc = -1;
    }
  }
  else if(strcmp(opt_key, "smtp-auth") == 0){
    if(strcmp(opt_value, "none") == 0){
      mailx->auth_method = SMTP_AUTH_NONE;
    }
    else if(strcmp(opt_value, "plain") == 0){
      mailx->auth_method = SMTP_AUTH_PLAIN;
    }
    else if(strcmp(opt_value, "login") == 0){
      mailx->auth_method = SMTP_AUTH_LOGIN;
    }
    else if(strcmp(opt_value, "cram-md5") == 0){
      mailx->auth_method = SMTP_AUTH_CRAM_MD5;
    }
    else{
      rc = -1;
    }
  }
  else if(strcmp(opt_key, "smtp-flag") == 0){
    if(strcmp(opt_value, "debug") == 0){
      mailx->smtp_flags |= SMTP_DEBUG;
    }
    else if(strcmp(opt_value, "no-cert-verify") == 0){
      mailx->smtp_flags |= SMTP_NO_CERT_VERIFY;
    }
    else{
      rc = -1;
    }
  }
  else if(strcmp(opt_key, "smtp-server") == 0){
    if((mailx->server = strdup(opt_value)) == NULL){
      log_errx(1, "strdup");
    }
  }
  else if(strcmp(opt_key, "smtp-port") == 0){
    if((mailx->port = atoi(opt_value)) <= 0){
      log_errx(1, "port");
    }
  }
  else if(strcmp(opt_key, "smtp-user") == 0){
    if((mailx->user = strdup(opt_value)) == NULL){
      log_errx(1, "strdup");
    }
  }
  else if(strcmp(opt_key, "smtp-pass") == 0){
    if((mailx->pass = strdup(opt_value)) == NULL){
      log_errx(1, "strdup");
    }
  }
  else if(strcmp(opt_key, "smtp-from") == 0){
    if((mailx->from = strdup(opt_value)) == NULL){
      log_errx(1, "strdup");
    }
  }
  else{
    rc = -1;
  }

  free(optdup);

  if(rc < 0){
    log_errx(1, "invalid argument: %s", option);
  }
}

/**
 * Initialize and set the default options in the mailx context.
 *
 * See description of -S argument in main for more details.
 *
 * @param[in] mailx The mailx content to initialize.
 */
static void
mailx_init_default_values(struct mailx *const mailx){
  memset(mailx, 0, sizeof(*mailx));
  mailx->subject = "";
  mailx->connection_security = SMTP_SECURITY_NONE;
  mailx->auth_method = SMTP_AUTH_NONE;
}

/**
 * Frees the allocated memory associated with the mailx context.
 *
 * @param[in] mailx The mailx context to free.
 */
static void
mailx_free(const struct mailx *const mailx){
  free(mailx->body);
  free(mailx->server);
  free(mailx->user);
  free(mailx->pass);
  free(mailx->from);
}

/**
 * Main program entry point for the mailx utility.
 *
 * This program supports the following options:
 *   - -a 'name:path' - Attach a file with name to display to recipient and
 *                      file path pointing to file location on local storage.
 *   - -s subject     - Email subject line.
 *   - -S key=value   - A key/value pair to set various configuration options,
 *                      controlling the behavior of the SMTP client connection.
 *
 * The following list contains possible options for the -S argument:
 *   - smtp-security - none, tls, starttls
 *   - smtp-auth     - none, plain, login, cram-md5
 *   - smtp-flag     - debug, no-cert-verify
 *   - smtp-server   - server name or IP address
 *   - smtp-port     - server port number
 *   - smtp-user     - server authentication user name
 *   - smtp-pass     - server authentication user password
 *   - smtp-from     - from email account
 *
 * The following list shows the default option for -S argument if not provided:
 *   - smtp-security - none
 *   - smtp-auth     - none
 *   - smtp-flag     - none
 *   - smtp-server   - localhost
 *   - smtp-port     - 25
 *   - smtp-user     - none
 *   - smtp-pass     - none
 *   - smtp-from     - none
 *
 * @param[in] argc Number of arguments in @p argv.
 * @param[in] argv String array containing the program name and any optional
 *                 parameters described above.
 * @retval 0 Email has been sent.
 * @retval 1 An error occurred while sending email. Although unlikely, an email
 *           can still get sent even after returning with this error code.
 */
int main(int argc, char *argv[]){
  int rc;
  int i;
  struct mailx mailx;

  mailx_init_default_values(&mailx);

  while((rc = getopt(argc, argv, "a:s:S:")) != -1){
    switch(rc){
    case 'a':
      mailx_append_attachment_arg(&mailx, optarg);
      break;
    case 's':
      mailx.subject = optarg;
      break;
    case 'S':
      mailx_parse_smtp_option(&mailx, optarg);
      break;
    default:
      return 1;
    }
  }
  argc -= optind;
  argv += optind;

  if(argc < 1){
    log_errx(1, "must provide at least one email destination address");
  }

  if(mailx.from == NULL){
    log_errx(1, "must provide a FROM address");
  }

  if(mailx.server == NULL){
    if((mailx.server = strdup("localhost")) == NULL){
      log_errx(1, "strdup");
    }
  }

  if(mailx.port == 0){
    mailx.port = 25;
  }

  puts("Reading email body from stdin");
  if((mailx.body = smtp_ffile_get_contents(stdin, NULL)) == NULL){
    log_errx(1, "failed to read email body from stdin");
  }

  if (smtp_open(mailx.server,
            mailx.port,
            mailx.connection_security,
            mailx.smtp_flags,
            NULL,
            &mailx.smtp) != 0) {
  mailx_free(&mailx);
    return -1;
    }

  mailx_address_append(&mailx, SMTP_ADDRESS_FROM, mailx.from);
  for(i = 0; i < argc; i++){
    mailx_address_append(&mailx, SMTP_ADDRESS_TO, argv[i]);
  }

    mailx.smtp->attachment_list = mailx.attachments;
    mailx.attachments = NULL;

  mailx_send(&mailx);
  mailx_free(&mailx);
  return 0;
}

