// session.c -- Session management for HTTP/HTTPS connections
// Copyright (C) 2008-2010 Markus Gutschke <markus@shellinabox.com>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//
// In addition to these license terms, the author grants the following
// additional rights:
//
// If you modify this program, or any covered work, by linking or
// combining it with the OpenSSL project's OpenSSL library (or a
// modified version of that library), containing parts covered by the
// terms of the OpenSSL or SSLeay licenses, the author
// grants you additional permission to convey the resulting work.
// Corresponding Source for a non-source form of such a combination
// shall include the source code for the parts of OpenSSL used as well
// as that of the covered work.
//
// You may at your option choose to remove this additional permission from
// the work, or from any part of it.
//
// It is possible to build this program in a way that it loads OpenSSL
// libraries at run-time. If doing so, the following notices are required
// by the OpenSSL and SSLeay licenses:
//
// This product includes software developed by the OpenSSL Project
// for use in the OpenSSL Toolkit. (http://www.openssl.org/)
//
// This product includes cryptographic software written by Eric Young
// (eay@cryptsoft.com)
//
//
// The most up-to-date version of this program is always available from
// http://shellinabox.com

#include "launcher.h"
#include <plib/crypt.h>
#include <plib/exec.h>

static int addService(struct shell_in_a_box_data * data, struct Service * value) {

    if ((data->services = ZREALLOC(data->services, ++data->numServices * sizeof(struct Service *))) == NULL)
	return -1;
    data->services[data->numServices - 1] = value;
    data->services[data->numServices - 1]->id = data->numServices - 1;
    return 1;
}

struct shell_in_a_box_data * initShellInABoxData(struct shell_in_a_box_data * data) {
    if ((data == NULL) && ((data = ZALLOC(sizeof(struct shell_in_a_box_data))) != NULL)) {
	struct Service * service = newService(
#ifdef HAVE_BIN_LOGIN
                                    geteuid() ? ":SSH" : ":LOGIN"
#else
                                    ":SSH"
#endif
                                    );


	os_mutex_init(&data->mutex);
	data->launcher = -1;
	data->restricted = -1;

	if (service != NULL) {
	    // If the user did not register any services, provide the default service
	    addService(data, service);
	    forkLauncher(data);
	}
    }
    return data;
}

void initSession(struct Session *session, char *sessionKey, const char *peerName, struct shell_in_a_box_data * data) {
    session->sessionKey = sessionKey;
    if ((session->peerName = ZSTRDUP(peerName)) == NULL)
	return;
    session->pty = -1;
    session->ptyFirstRead = 1;
    session->data = data;
    os_mutex_init(&session->mutex);
    session->last_time = time(NULL);
}

struct Session *newSession(char *sessionKey, const char *peerName, struct shell_in_a_box_data * data) {
    struct Session	* session;

    if ((session = ZALLOC(sizeof(struct Session))) != NULL)
	initSession(session, sessionKey, peerName, data);
    return session;
}

void destroySession(struct Session *session) {
    if (session != NULL) {
	os_mutex_lock(&session->mutex);
	terminateChild(session->data, session);
	session->peerName = ZFREE(session->peerName);
	session->sessionKey = ZFREE(session->sessionKey);
	if (session->pty >= 0)
	    close(session->pty);
	os_mutex_unlock(&session->mutex);
    }
}

void deleteSession(struct Session *session) {
    destroySession(session);
    ZFREE(session);
}

void abandonSession(struct shell_in_a_box_data * data, struct Session *session) {
    if ((data != NULL) && (data->sessions != NULL)) {
	os_mutex_lock(&data->mutex);
	delete_from_hash_map(data->sessions, session->sessionKey, NULL);
	os_mutex_unlock(&data->mutex);
    }
}

void finishSession(struct shell_in_a_box_data * data, struct Session *session) {
    if ((data != NULL) && (data->sessions != NULL)) {
	os_mutex_lock(&data->mutex);
	delete_from_hash_map(data->sessions, session->sessionKey, NULL);
	os_mutex_unlock(&data->mutex);
    }
}

void finishAllSessions(struct shell_in_a_box_data * data) {
    if ((data != NULL) && (data->sessions != NULL)) {
	os_mutex_lock(&data->mutex);
	data->sessions = delete_hash_map(data->sessions);
	os_mutex_unlock(&data->mutex);
    }
}

static int checkOrphanedSession(void * t, const char * key, void ** value) {
    int			timeout = (int)t;
    struct Session	* session = *value;
    time_t		last_access;

    os_mutex_lock(&session->mutex);
    last_access = time(NULL) - session->last_time;
    os_mutex_unlock(&session->mutex);
    if (last_access >= timeout)
	return 0;
    return 1;
}

void finishOrphanedSessions(struct shell_in_a_box_data * data, int timeout) {
    if ((data != NULL) && (data->sessions != NULL)) {
	iterateOverSessions(data, checkOrphanedSession, (void *)timeout);
    }
}

static void destroySessionHashEntry(void * arg, const char * key, void * value) {
    deleteSession((struct Session *)value);
}

static char *newSessionKey(struct shell_in_a_box_data * data) {
    char	* sessionKey = NULL;

    while (TRUE) {
	if ((sessionKey = generate_v7_uuid(NULL, 0)) == NULL) {
	    log_err("[server] Failed to allocate sessionKey");
	    break;
	}
	if (get_from_hash_map(data->sessions, sessionKey) == NULL)
	    break;
	sessionKey = ZFREE(sessionKey);
    }
    return sessionKey;
}

struct Session *findSession(struct shell_in_a_box_data * data, char *sessionKey, const char *peerName, BOOL *sessionIsNew) {
    struct Session	* session= NULL;

    *sessionIsNew = TRUE;
    if ((data == NULL) && ((data = initShellInABoxData(data)) == NULL))
	return NULL;
    if (data->sessions == NULL)
	data->sessions = new_hash_map(destroySessionHashEntry, NULL);

    os_mutex_lock(&data->mutex);
    if (!SEMPTY(sessionKey))
	session = (struct Session *)get_from_hash_map(data->sessions, sessionKey);
    if (session != NULL) {
	*sessionIsNew = FALSE;
	if ((session->peerName != NULL) && (strcmp(session->peerName, peerName) != 0)) {
	    log_err("Attempt to steal session %s-%s", session->peerName, peerName);
	    os_mutex_unlock(&data->mutex);
	    return NULL;
	}
	if ((session->pty < 0) && (session->pid <= 0)) {
	    log_info("Attempt to use closed child app %d-%d", session->pty, session->pid);
	    os_mutex_unlock(&data->mutex);
	    return NULL;
	}
    } else if (!SEMPTY(sessionKey)) {
	*sessionIsNew = FALSE;
	log_debug("[server] Failed to find session: %s", sessionKey);
    } else {
	// First contact. Create session, now.
	if ((sessionKey = newSessionKey(data)) != NULL) {
	    if ((session = newSession(sessionKey, peerName, data)) != NULL) {
		add_to_hash_map(data->sessions, sessionKey, (void *)session, NULL);
		log_debug("[server] Creating a new session: %s", sessionKey);
	    }
	}
    }
    os_mutex_unlock(&data->mutex);
    if (session != NULL)
	os_mutex_lock(&session->mutex);
    return session;
}

void iterateOverSessions(struct shell_in_a_box_data * data, int (*fnc)(void *, const char *, void **), void *arg){
    os_mutex_lock(&data->mutex);
    iterate_over_hash_map(data->sessions, fnc, arg);
    os_mutex_unlock(&data->mutex);
}

int numSessions(struct shell_in_a_box_data * data) {
    return get_hash_map_size(data->sessions);
}
