// service.c -- Service descriptions
// Copyright (C) 2008-2010 Markus Gutschke <markus@shellinabox.com>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//
// In addition to these license terms, the author grants the following
// additional rights:
//
// If you modify this program, or any covered work, by linking or
// combining it with the OpenSSL project's OpenSSL library (or a
// modified version of that library), containing parts covered by the
// terms of the OpenSSL or SSLeay licenses, the author
// grants you additional permission to convey the resulting work.
// Corresponding Source for a non-source form of such a combination
// shall include the source code for the parts of OpenSSL used as well
// as that of the covered work.
//
// You may at your option choose to remove this additional permission from
// the work, or from any part of it.
//
// It is possible to build this program in a way that it loads OpenSSL
// libraries at run-time. If doing so, the following notices are required
// by the OpenSSL and SSLeay licenses:
//
// This product includes software developed by the OpenSSL Project
// for use in the OpenSSL Toolkit. (http://www.openssl.org/)
//
// This product includes cryptographic software written by Eric Young
// (eay@cryptsoft.com)
//
//
// The most up-to-date version of this program is always available from
// http://shellinabox.com

#include "launcher.h"

void initService(struct Service *service, const char *arg) {
    // The first part of the argument is the path where the service should
    // be mounted. Remove any trailing slashes and make sure there is exactly
    // one leading slash before copying it into service->path.
    char	* desc, * ptr;

    if ((desc = ZSTRDUP(arg)) == NULL)
	return;
    while (*arg == '/')
	arg++;
    if ((ptr = strchr(arg, ':')) == NULL) {
error:
	log_err("[config] Syntax error in service description \"%s\"!", desc);
    }
    service->id = -1;
    if ((service->path = ZALLOC(ptr - arg + 2)) == NULL)
	return;
    ((char *)service->path)[0] = '/';
    memcpy((char *)service->path + 1, arg, ptr - arg);
    ((char *)service->path)[ptr - arg + 1]    = '\000';
    while (service->path[1] && strrchr(service->path, '\000')[-1] == '/') {
	strrchr(service->path, '\000')[-1] = '\000';
    }
    arg = ptr + 1;

#ifdef HAVE_BIN_LOGIN
    // The next part of the argument is either the word 'LOGIN' or the
    // application definition.
    if (!strcmp(arg, "LOGIN")) {
	if (geteuid()) {
	    log_err("[config] Must be \"root\" to invoke LOGIN service!");
	}
	service->useLogin = TRUE;
	if ((service->user = ZSTRDUP("root")) == NULL)
	    return;
	if ((service->group = ZSTRDUP("root")) == NULL)
	    return;
	if ((service->cwd = ZSTRDUP("/tmp")) == NULL)
	    return;
#ifdef HAVE_BIN_NULOGIN
	if ((service->cmdline = strdup("/usr/bin/nu-login")) == NULL)
#else
	if ((service->cmdline = strdup("/bin/login")) == NULL)
#endif
	    return;
    } else
#endif
    if (!strcmp(arg, "SSH") || !strncmp(arg, "SSH:", 4)) {
	char	* host,* sshPort;

	service->authUser = 2;
	service->uid = -1;
	service->gid = -1;
	if ((service->cwd = ZSTRDUP("/")) == NULL)
	    return;
	if ((host = ZSTRDUP("localhost")) == NULL)
	    return;
	if ((sshPort = ZSTRDUP("22")) == NULL)
	    return;
	if ((ptr = strchr(arg, ':')) != NULL) {
	    ptr ++;
	    if (*ptr) {
		char	* tmp = strchr(ptr, ':');

		if (tmp == NULL) {
		    // If the second ":" is not found, keep as host whatever is after first ":".
		    ZFREE(host);
		    if ((host = ZSTRDUP(ptr)) == NULL)
			return;
		} else {
		    // If we find a second ":", keep as a host whatever is in between first ":"
		    // and second ":" and as sshPort whatever is after second ":".
		    int	size = (tmp - ptr + 1);

		    ZFREE(host);
		    ZFREE(sshPort);
		    if ((host = ZALLOC(size)) == NULL)
			return;
		    memcpy(host, ptr, size - 1);
		    if ((sshPort = ZSTRDUP(tmp + 1)) == NULL)
			return;
		}
	    }
	}

	// Don't allow manipulation of the SSH command line through "creative" use
	// of the host name.
	for (char *h = host; *h; h++) {
	    char	ch = *h;

	    if (!((ch >= '0' && ch <= '9') ||
		(ch >= 'A' && ch <= 'Z') ||
		(ch >= 'a' && ch <= 'z') ||
		ch == '-' || ch == '.')) {
		log_err("[config] Invalid hostname \"%s\" in service definition!", host);
		return;
	    }
	}

	// Don't allow manipulation of the SSH command line through "creative" use
	// of the port.
	for (char *h = sshPort; *h; h++) {
	    char	ch = *h;

	    if (!(ch >= '0' && ch <= '9')) {
		log_err("[config] Invalid port \"%s\" in service definition!", sshPort);
		return;
	    }
	}

	os_asprintf(&service->cmdline, "ssh -a -e none -i /dev/null -x -oChallengeResponseAuthentication=no "
          "-oCheckHostIP=no -oClearAllForwardings=yes -oCompression=no "
          "-oControlMaster=no -oGSSAPIAuthentication=no "
          "-oHostbasedAuthentication=no -oIdentitiesOnly=yes "
          "-oKbdInteractiveAuthentication=yes -oPasswordAuthentication=yes "
          "-oPreferredAuthentications=keyboard-interactive,password "
          "-oPubkeyAuthentication=no -oRhostsRSAAuthentication=no "
          "-oRSAAuthentication=no -oStrictHostKeyChecking=no -oTunnel=no "
          "-oUserKnownHostsFile=/dev/null -oVerifyHostKeyDNS=no "
	// beewoolie-2012.03.30: while it would be nice to disable this
	//          feature, we cannot be sure that it is available on the
	//          target server.  Removing it for the sake of Centos.
	//          "-oVisualHostKey=no"
	    " -oLogLevel=FATAL -p%s %%s@%s",sshPort,  host);
	ZFREE(host);
	ZFREE(sshPort);
    } else {
	// The user definition is either the word 'AUTH' or a valid user and
	// group id.
	if ((ptr = strchr(arg, ':')) == NULL) {
	    goto error;
	}
	*ptr = '\000';
	if (!strcmp(arg, "AUTH")) {
	    service->authUser = 1;
	    service->uid = -1;
	    service->gid = -1;
	} else {
	    // Numeric or symbolic user id
//	    service->uid = parseUserArg(arg, &service->user);
	    *ptr = ':';
	    arg = ptr + 1;

	    // Numeric or symbolic group id
	    if ((ptr = strchr(arg, ':')) == NULL) {
		goto error;
	    }
	    *ptr = '\000';
//	    service->gid = parseGroupArg(arg, &service->group);
	}
	*ptr = ':';
	arg = ptr + 1;

	// The next part of the argument is the starting working directory
	if ((ptr = strchr(arg, ':')) == NULL) {
	    goto error;
	}
	*ptr = '\000';
	if (!strcmp(arg, "HOME")) {
	    service->useHomeDir = TRUE;
	} else {
	    if (*arg != '/') {
		log_err("[config] Working directories must have absolute paths!");
		return;
	    }
	    if ((service->cwd = ZSTRDUP(arg)) == NULL)
		return;
	}
	*ptr = ':';
	arg = ptr + 1;

	// The final argument is the command line
	if (!*arg)
	    goto error;
	if (!strcmp(arg, "SHELL")) {
	    service->useDefaultShell = TRUE;
	} else {
	    if ((service->cmdline = strdup(arg)) == NULL)
		return;
	}
    }
    ZFREE(desc);
}

struct Service *newService(const char *arg) {
    struct Service	* service;

    if ((service = ZALLOC(sizeof(struct Service))) != NULL)
	initService(service, arg);
    return service;
}

void destroyService(struct Service *service) {
    if (service != NULL) {
	ZFREE(service->path);
	ZFREE(service->user);
	ZFREE(service->group);
	ZFREE(service->cwd);
	free(service->cmdline);
    }
}

void deleteService(struct Service *service) {
    destroyService(service);
    ZFREE(service);
}
