#include <pty.h>
#include <utmp.h>
#include <pwd.h>
#include <sys/wait.h>
#ifdef HAVE_SYS_TTYDEFAULTS_H
#include <sys/ttydefaults.h>
#endif
#include "session.h"
#include "launcher.h"
#include <plib/exec.h>

int launchChild(struct shell_in_a_box_data * data, int service, struct Session *session, const char *url) {
    char			* u;
    pid_t			pid;
    char			cmsg_buf[CMSG_SPACE(sizeof(int))];
    struct iovec		iov = { 0 };
    struct msghdr		msg = { 0 };
    struct LaunchRequest	* request;
    ssize_t			len, bytes;
    char			* peerName, * realIP = NULL;
    struct cmsghdr		* cmsg;

    if (data->launcher < 0) {
	errno = EINVAL;
	log_errno("Launcher is not open");
	return -1;
    }

    if ((u = ZSTRDUP(url)) == NULL) {
	log_err("Could not allocate URL");
	return -1;
    }
    for (int i; u[i = strcspn(u, "\\\"'`${};() \r\n\t\v\f")]; ) {
	static const char hex[] = "0123456789ABCDEF";

	if ((u = ZREALLOC(u, strlen(u) + 4)) == NULL) {
	    log_err("Could not allocate subpart of URL");
	    return -1;
	}
	memmove(u + i + 3, u + i + 1, strlen(u + i));
	u[i + 2] = hex[ u[i]       & 0xF];
	u[i + 1] = hex[(u[i] >> 4) & 0xF];
	u[i] = '%';
    }
    len = sizeof(struct LaunchRequest) + strlen(u) + 1;
    if ((request = ZALLOC(len)) == NULL) {
	ZFREE(u);
	log_err("Could not allocate Request struct");
	return -1;
    }
    request->service = service;
    request->terminate = -1;
    request->width = session->width;
    request->height = session->height;
    peerName = session->peerName;
    strncat(request->peerName, peerName, sizeof(request->peerName) - 1);
    if (realIP && *realIP)
	strncat(request->realIP, realIP, sizeof(request->realIP) - 1);
    request->urlLength = strlen(u);
    memcpy(&request->url, u, request->urlLength);
    ZFREE(u);
    if (!sock_watch_single(data->launcher, FALSE, 100)) {
	ZFREE(request);
	return -1;
    }
    if (sock_write(data->launcher, request, len, 100) != len) {
	ZFREE(request);
	return -1;
    }
    ZFREE(request);
    iov.iov_base = &pid;
    iov.iov_len = sizeof(pid);
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    msg.msg_control = &cmsg_buf;
    msg.msg_controllen = sizeof(cmsg_buf);
    if (!sock_watch_single(data->launcher, TRUE, 1000)) {
	return -1;
    }
    bytes = recvmsg(data->launcher, &msg, 0);
    if ((bytes < 0) || (bytes != sizeof(pid)) || (session->pid == pid)) {
	log_err("Wrong bytes or PID. (FD %d, B %d, PID %d. SPID %d)", data->launcher, bytes, pid, session->pid);
	return -1;
    }
    cmsg = CMSG_FIRSTHDR(&msg);
    if ((cmsg == NULL) || (cmsg->cmsg_level != SOL_SOCKET) || (cmsg->cmsg_type  != SCM_RIGHTS)) {
	log_err("Wrong message struct");
	return -1;
    }
    memcpy(&session->pty, CMSG_DATA(cmsg), sizeof(int));
    return pid;
}

int terminateChild(struct shell_in_a_box_data * data, struct Session *session) {
    struct LaunchRequest	* request;
    ssize_t			len = sizeof(struct LaunchRequest);
    int				res = -1;

    if (data->launcher < 0) {
	errno = EINVAL;
	return res;
    }
    if ((session == NULL) || (session->pid <= 0)) {
	log_debug("[server] Child pid for termination not valid!");
	return res;
    }
    os_mutex_lock(&session->mutex);
    // Send terminate request to launcher process
    if ((request = ZALLOC(len)) == NULL) {
	log_err("[server] Could not allocate memory!");
	goto err_e;
    }
    request->terminate = session->pid;
    if (!sock_watch_single(data->launcher, FALSE, 100)) {
	log_err("[server] Child %d termination request failed! Could not wait", request->terminate);
	goto err;
    }
    if (sock_write(data->launcher, request, len, 100) != len) {
	log_err("[server] Child %d termination request failed!", request->terminate);
	goto err;
    }
    session->pid = 0;
    session->cleanup = 0;
    res = 0;
err:
    ZFREE(request);
err_e:
    os_mutex_unlock(&session->mutex);
    return res;
}

static int forkPty(int *pty, struct Service *service,
                   const char *peerName, const char *realIP) {
    int		slave;
    pid_t	pid;
#ifdef HAVE_OPENPTY
    char	* ptyPath = NULL;
    size_t	length = 32;
    char	* path = NULL;

    if (openpty(pty, &slave, NULL, NULL, NULL) < 0) {
	*pty = -1;
	return -1;
    }
    // Recover name of PTY in a Hurd compatible way.  PATH_MAX doesn't
    // exist, so we need to use ttyname_r to fetch the name of the
    // pseudo-tty and find the buffer length that is sufficient. After
    // finding an adequate buffer size for the ptyPath, we allocate it
    // on the stack and release the freestore copy.  In this was we know
    // that the memory won't leak.  Note that ptsname_r is not always
    // available but we're already checking for this so it will be
    // defined in any case.
    while (path == NULL) {
	if ((path = ZALLOC(length)) == NULL) {
	    *pty = -1;
	    return -1;
	}    
	*path = 0;
	if (ptsname_r (*pty, path, length)) {
	    if (errno == ERANGE)
		path = ZFREE(path);
	    else
		break;          // Every other error means no name for us
	}
	length <<= 1;
    }
    ptyPath = ZSTRDUP(path);
    ZFREE(path);
#else
    char	ptyPath[PATH_MAX];

    if ((*pty = posix_openpt(O_RDWR|O_NOCTTY)) < 0 ||
	grantpt(*pty) < 0 ||
	unlockpt(*pty) < 0 ||
	ptsname_r(*pty, ptyPath, sizeof(ptyPath)) < 0 ||
	(slave = open(ptyPath, O_RDWR|O_NOCTTY)) < 0) {
	char	fname[40] = "/dev/ptyXX";

	if (*pty >= 0)
	    close(*pty);

	// Try old-style pty handling
	for (const char *ptr1 = "pqrstuvwxyzabcde"; *ptr1; ptr1++) {
	    fname[8] = *ptr1;
	    for (const char *ptr2 = "0123456789abcdef"; *ptr2; ptr2++) {
		fname[9] = *ptr2;
		if ((*pty = open(fname, O_RDWR, 0)) < 0) {
		    if (errno == ENOENT)
			continue;
		}
		grantpt(*pty);
		unlockpt(*pty);
		if (ptsname_r(*pty, ptyPath, sizeof(ptyPath)) < 0) {
		    strcpy(ptyPath, fname);
		    ptyPath[5] = 't';
		}
		if ((slave = open(ptyPath, O_RDWR|O_NOCTTY)) >= 0) {
		    log_debug("[server] Opened old-style pty: %s", ptyPath);
		    goto success;
		}
		close(*pty);
	    }
	}
	*pty = -1;
	return -1;
    }
success:
#endif
    // Now, fork off the child process
    if ((pid = fork()) < 0) {
	close(slave);
	close(*pty);
	*pty = -1;
	return -1;
    } else if (pid == 0) {
	pid  = getpid();
	snprintf((char *)&service->pid[0], sizeof(service->pid), "%d", pid);
	service->pty = slave;

	proc_close_all_fds((int []){ slave }, 1);

#ifdef HAVE_LOGIN_TTY
	login_tty(slave);
#else
	// Become the session/process-group leader
	setsid();
	setpgid(0, 0);

	// Redirect standard I/O to the pty
	dup2(slave, STDIN_FILENO);
	dup2(slave, STDOUT_FILENO);
	dup2(slave, STDERR_FILENO);
	if (slave > STDERR_FILENO)
	    close(slave);
#endif
	*pty = 0;

	// Force the pty to be our control terminal
	close(open(ptyPath, O_RDWR));

	return 0;
    } else {
	snprintf((char *)&service->pid[0], sizeof(service->pid), "%d", pid);
	service->pty = *pty;
	if (fcntl(*pty, F_SETFL, O_NONBLOCK | O_RDWR) != 0)
	    return -1;
	close(slave);
	return pid;
    }
}

static struct passwd *getPWEnt(uid_t uid) {
    struct passwd pwbuf, *pw;
    char *buf;
    struct passwd *passwd;
#ifdef _SC_GETPW_R_SIZE_MAX
    int	len = sysconf(_SC_GETPW_R_SIZE_MAX);

    if (len <= 0)
	len = 4096;
#else
    int	len = 4096;
#endif
    if ((buf = ZALLOC(len)) == NULL) {
	log_err("Could not allocate buffer");
	return NULL;
    }

    if((getpwuid_r(uid, &pwbuf, buf, len, &pw) != 0) && pw) {
	log_err("Could not get user information");
	return NULL;
    }
    if (!pw->pw_name  ) pw->pw_name   = "";
    if (!pw->pw_passwd) pw->pw_passwd = "";
    if (!pw->pw_gecos ) pw->pw_gecos  = "";
    if (!pw->pw_dir   ) pw->pw_dir    = "";
    if (!pw->pw_shell ) pw->pw_shell  = "";

    if ((passwd = ZALLOC(sizeof(struct passwd) +
	strlen(pw->pw_name) +
	strlen(pw->pw_passwd) +
	strlen(pw->pw_gecos) +
	strlen(pw->pw_dir) +
	strlen(pw->pw_shell) + 5)) == NULL) {
	log_err("Could not allocate paswd struct");
	return NULL;
    }

    passwd->pw_uid = pw->pw_uid;
    passwd->pw_gid = pw->pw_gid;
    strncat(passwd->pw_shell = strrchr(
    strncat(passwd->pw_dir = strrchr(
    strncat(passwd->pw_gecos          = strrchr(
    strncat(passwd->pw_passwd         = strrchr(
    strncat(passwd->pw_name           = (char *)(passwd + 1),
         pw->pw_name,   strlen(pw->pw_name)),   '\000') + 1,
         pw->pw_passwd, strlen(pw->pw_passwd)), '\000') + 1,
         pw->pw_gecos,  strlen(pw->pw_gecos)),  '\000') + 1,
         pw->pw_dir,    strlen(pw->pw_dir)),    '\000') + 1,
         pw->pw_shell,  strlen(pw->pw_shell));
    ZFREE(buf);
    return passwd;
}

static void destroyVariableHashEntry(void *arg, const char *key,
                                     void *value) {
    ZFREE(key);
    ZFREE(value);
}

static void execService(int width, int height,
                        struct Service *service, const char *peerName,
                        const char *realIP, char **environment,
                        const char *url) {

    // Create a hash table with all the variables that we can expand. This
    // includes all environment variables being passed to the child.
    struct hash_map	* vars;
    char	* key, * value;
    char	* cmdline;
    enum { ENV, ARGS }	state = ENV;
    enum { NONE, SINGLE, DOUBLE }	quote = NONE;
    int	argc = 0;
    char	** argv;
    char	* cmd;
    char	* slash;

    if ((vars = new_hash_map(destroyVariableHashEntry, NULL)) == NULL)
	return;
    for (char **e = environment; *e; e++) {
	char *ptr = strchr(*e, '=');

	if (!ptr) {
	    if ((key = strdup(*e)) == NULL)
		return;
	    if ((value = strdup("")) == NULL)
		return;
	} else {
	    if ((key = ZALLOC(ptr - *e + 1)) == NULL)
		return;
	    memcpy(key, *e, ptr - *e);
	    key[ptr - *e] = '\000';
	    if ((value = ZSTRDUP(ptr + 1)) == NULL)
		return;
	}
	// All of our variables are lower-case
	for (ptr = key; *ptr; ptr++) {
	    if (*ptr >= 'A' && *ptr <= 'Z') {
		*ptr += 'a' - 'A';
	    }
	}
	add_to_hash_map(vars, key, value, NULL);
    }
    if ((key = ZSTRDUP("gid")) == NULL)
	return;
    os_asprintf(&value, "%d", service->gid);
    add_to_hash_map(vars, key, value, NULL);
    if (((key = ZSTRDUP("group")) == NULL) || ((value = ZSTRDUP(service->group)) == NULL))
	return;
    add_to_hash_map(vars, key, value, NULL);
    if (((key = ZSTRDUP("peer")) == NULL) || ((value = ZSTRDUP(peerName)) == NULL))
	return;
    add_to_hash_map(vars, key, value, NULL);
    if (((key = ZSTRDUP("realip")) == NULL) || ((value = ZSTRDUP(realIP)) == NULL))
	return;
    add_to_hash_map(vars, key, value, NULL);
    if ((key = ZSTRDUP("uid")) == NULL)
	return;
    os_asprintf(&value, "%d", service->uid);
    add_to_hash_map(vars, key, value, NULL);
    if ((key = ZSTRDUP("url")) == NULL)
	return;
    add_to_hash_map(vars, key, strdup(url), NULL);

    if (((cmdline = ZSTRDUP(service->cmdline)) == NULL) || ((argv = ZALLOC(sizeof(char *))) == NULL))
	return;
    key = NULL;
    value = NULL;
    for (char *ptr = cmdline; ; ptr++) {
	char	ch;

	if (!key && *ptr && *ptr != ' ') {
	    key = ptr;
	}
	switch (*ptr) {
	    case '\'':
		if (quote == SINGLE || quote == NONE) {
		    memmove(ptr, ptr + 1, strlen(ptr));
		    ptr--;
		    quote = quote == SINGLE ? NONE : SINGLE;
		} else {
		    if (quote == DOUBLE)
			return;
		}
	    break;
	    case '\"':
		if (quote == DOUBLE || quote == NONE) {
		    memmove(ptr, ptr + 1, strlen(ptr));
		    ptr--;
		    quote = quote == DOUBLE ? NONE : DOUBLE;
		} else {
		    if (quote == SINGLE)
			return;
		}
	    break;
	    case '$':
		if ((quote == NONE || quote == DOUBLE) && ptr[1] == '{') {
		    // Always treat environment variables as if they were quoted. There

		    // is no good reason for us to try to look for spaces within
		    // expanded environment variables. This just leads to subtle bugs.
		    char	* end = ptr + 2;
		    const char	* repl;
		    int		replLen, incr;

		    while (*end && *end != '}')
			end ++;
		    ch = *end;
		    *end = '\000';
		    repl = get_from_hash_map(vars, ptr + 2);
		    replLen = repl ? strlen(repl) : 0;
		    *end = ch;
		    if (ch)
			end++;
		    incr = replLen - (end - ptr);
		    if (incr > 0) {
			char	* oldCmdline = cmdline;

			if ((cmdline = ZREALLOC(cmdline,
                                        (end - cmdline) + strlen(end) +
                                        incr + 1)) == NULL)
			    return;

			ptr += cmdline - oldCmdline;
			end += cmdline - oldCmdline;
			if (key)
			    key += cmdline - oldCmdline;
			if (value)
			    value += cmdline - oldCmdline;
		    }
		    memmove(ptr + replLen, end, strlen(end) + 1);
		    if (repl != NULL)
			memcpy(ptr, repl, replLen);
		    ptr += replLen - 1;
		}
	    break;
	    case '\\':
		if (!ptr[1]) {
		    *ptr-- = '\000';
		} else {
		    memmove(ptr, ptr + 1, strlen(ptr));
		}
	    break;
	    case '=':
		// This is the seperator between keys and values of any environment
		// variable that we are asked to set.
		if (state == ENV && quote == NONE && !value) {
		    *ptr = '\000';
		    value = ptr + 1;
		}
	    break;
	    case ' ':
		// If this space character is not quoted, this is the start of a new
		// command line argument.
		if (quote != NONE)
		    break;
		// Fall thru
	    case '\000':;
		ch = *ptr;
		if (key) {
		    *ptr = '\000';
		    if (state == ENV && value) {
			// Override an existing environment variable.
			int	numEnvVars = 0;
			int	len = strlen(key);

			for (char **e = environment; *e; e++, numEnvVars++) {
			    if (!strncmp(*e, key, len) && (*e)[len] == '=') {
				int	s_size = len + strlen(value) + 1;

				if ((*e = ZREALLOC(*e, s_size + 1)) == NULL)
				    return;
				(*e)[len + 1] = '\000';
				strncat(*e, value, s_size);
				numEnvVars = -1;
				break;
			    }
			}
			// Add a new environment variable
			if (numEnvVars >= 0) {
			    if ((environment = ZREALLOC(environment,
                                        (numEnvVars + 2)*sizeof(char *))) == NULL)
				return;
			    value[1] = '=';
			    environment[numEnvVars++] = ZSTRDUP(key);
			    environment[numEnvVars] = NULL;
			}
		    } else {
			// Add entry to argv.
			state = ARGS;
			argv[argc++] = ZSTRDUP(key);
			if ((argv = ZREALLOC(argv, (argc + 1)*sizeof(char *))) == NULL)
			    return;
		    }
		}
		key = NULL;
		value = NULL;
		if (!ch)
		    goto done;
	    break;
	    default:
	    break;
	}
    }
done:
    ZFREE(cmdline);
    argv[argc] = NULL;
    delete_hash_map(vars);
    if (argc <= 0)
	return;

    environ = environment;
    if ((cmd = ZSTRDUP(argv[0])) == NULL)
	return;
    slash = strrchr(argv[0], '/');
    if (slash != NULL)
	memmove(argv[0], slash + 1, strlen(slash));
    if (service->useDefaultShell) {
	int	len = strlen(argv[0]);

	if ((argv[0] = ZREALLOC(argv[0], len + 2)) == NULL)
	    return;
	memmove(argv[0] + 1, argv[0], len);
	argv[0][0] = '-';
	argv[0][len + 1] = '\000';
    }
    if (execvp(cmd, argv) < 0) {
	ZFREE(argv);
	ZFREE(cmd);
    }
}

void setWindowSize(int pty, int width, int height) {
    if (width > 0 && height > 0) {
#ifdef TIOCSSIZE
    {
	struct ttysize	win;

	ioctl(pty, TIOCGSIZE, &win);
	win.ts_lines = height;
	win.ts_cols = width;
	ioctl(pty, TIOCSSIZE, &win);
    }
#endif
#ifdef TIOCGWINSZ
    {
	struct winsize	win;

	ioctl(pty, TIOCGWINSZ, &win);
	win.ws_row = height;
	win.ws_col = width;
	ioctl(pty, TIOCSWINSZ, &win);
    }
#endif
    }
}

static void childProcess(struct Service *service, int width, int height,
                         const char *peerName, const char *realIP,
                         const char *url) {
    static const char	* legalEnv[] = { "TZ", "HZ", NULL };
    char		** environment;
    int			numEnvVars = 1;
    struct termios	tt = { 0 };

    // Set initial window size
    setWindowSize(0, width, height);

    // Set up environment variables
    if ((environment = ZALLOC(2*sizeof(char *))) == NULL) {
	_exit(1);
    }
    if ((environment[0] = strdup("TERM=xterm")) == NULL) {
	_exit(1);
    }
    if (width > 0 && height > 0) {
	numEnvVars += 2;
	if ((environment = ZREALLOC(environment, (numEnvVars + 1)*sizeof(char *))) == NULL) {
	    _exit(1);
	}
	os_asprintf(&environment[numEnvVars-2], "COLUMNS=%d", width);
	os_asprintf(&environment[numEnvVars-1], "LINES=%d", height);
    }
    for (int i = 0; legalEnv[i]; i++) {
	char	* value = getenv(legalEnv[i]);

	if (value != NULL) {
	    numEnvVars++;
	    if ((environment = ZREALLOC(environment, (numEnvVars + 1)*sizeof(char *))) == NULL) {
		_exit(1);
	    }
	    os_asprintf(&environment[numEnvVars-1], "%s=%s", legalEnv[i], value);
	}
    }

    // Add useful environment variables that can be used in custom client scripts
    // or programs.
    numEnvVars += 3;
    if ((environment = ZREALLOC(environment, (numEnvVars + 1)*sizeof(char *))) == NULL) {
	_exit(1);
    }
    os_asprintf(&environment[numEnvVars-3], "SHELLINABOX_URL=%s", url);
    os_asprintf(&environment[numEnvVars-2], "SHELLINABOX_PEERNAME=%s", peerName);
    os_asprintf(&environment[numEnvVars-1], "SHELLINABOX_REALIP=%s", realIP);
    environment[numEnvVars] = NULL;

    // Set initial terminal settings
    tcgetattr(0, &tt);
    tt.c_iflag = TTYDEF_IFLAG & ~ISTRIP;
    tt.c_oflag = TTYDEF_OFLAG;
    tt.c_lflag = TTYDEF_LFLAG;
    tt.c_cflag = (TTYDEF_CFLAG & ~(CS7|PARENB|HUPCL)) | CS8;
    tt.c_cc[VERASE] = '\x7F';
    cfsetispeed(&tt, B38400);
    cfsetospeed(&tt, B38400);
    tcsetattr(0, TCSAFLUSH, &tt);

    // Change user and group ids
    (void)setresgid(service->gid, service->gid, service->gid);
    (void)setresuid(service->uid, service->uid, service->uid);

    // Change working directory
    if (service->useHomeDir) {
	struct passwd *pw = getPWEnt(getuid());

	if (service->cwd != NULL) {
	    _exit(1);
	}
	if ((service->cwd = ZSTRDUP(pw->pw_dir)) == NULL) {
	    _exit(1);
	}
	ZFREE(pw);
    }
    if (service->cwd == NULL) {
	_exit(1);
    }
    if (!*service->cwd || *service->cwd != '/' || chdir(service->cwd)) {
	if ((service->cwd  = ZREALLOC((char *)service->cwd, 2)) == NULL) {
	    _exit(1);
	}
	*(char *)service->cwd = '\000';
	strncat((char *)service->cwd, "/", 1);
	puts("No directory, logging in with HOME=/");
	if (chdir("/") != 0)
	    _exit(1);
	for (int i = 0; environment[i]; i++) {
	    if (!strncmp(environment[i], "HOME=", 5)) {
		free(environment[i]);
    		if ((environment[i] = strdup("HOME=/")) == NULL) {
		    _exit(1);
		}
		break;
	    }
	}
    }

    // Reset the sigaction for HUP to the default, so the child does not inherit the action of SIG_IGN from us.
    // We need the child to be able to get HUP's because we send HUP if the session times out/closes.
    signal(SIGHUP, SIG_DFL);
    // Launch user provied service
    execService(width, height, service, peerName, realIP, environment, url);
    _exit(1);
}

static void sigChildHandler(int sig, siginfo_t *info,
                            void *unused) {
    do { } while (0);
}

static void launcherDaemon(struct shell_in_a_box_data * data, int fd) {
    struct sigaction sa;
    struct LaunchRequest request;

    memset(&sa, 0, sizeof(sa));
    sa.sa_flags = SA_NOCLDSTOP | SA_SIGINFO;
    sa.sa_sigaction = sigChildHandler;

    if (sigaction(SIGCHLD, &sa, NULL) != 0) {
	log_errno("[server] Could not manipulate with signal block!");
	return;
    }

    // pututxline() can cause spurious SIGHUP signals. Better ignore those.
    signal(SIGHUP, SIG_IGN);

    while (TRUE) {
	ssize_t	len;
	int	status;
	pid_t	pid;
	char	* url;
	int	pty;
	char	cmsg_buf[CMSG_SPACE(sizeof(int))]; // = { 0 }; // Valid initializer makes OSX mad.
	struct iovec	iov = { 0 };
	struct msghdr	msg = { 0 };
	struct cmsghdr	* cmsg;

	if (!sock_watch_single(fd, TRUE, 1000)) {
	    continue;
	}
	errno = 0;
	len = sock_read(fd, &request, sizeof(request), 100);
//	if ((size_t)len != sizeof(request) && errno != EINTR) {
	if (len < 0) {
//	    if (len)
//		log_errno("[server] Failed to read launch request!");
//	    break;
log_errno("Could not read request from FD %d-%ld", fd, len);
break;
	}

	// Check whether our read operation got interrupted, because a child
	// has died.
	while ((pid = waitpid(-1, &status, WNOHANG)) > 0) {
//	    log_debug("[server] Child %d exited with exit code %d.", pid, WEXITSTATUS(status));
	    log_info("[server] Child %d exited with exit code %d.", pid, WEXITSTATUS(status));
	}
	if (len != sizeof(request))
	    continue;

	// Check if we received terminate request from parent process and
	// try to terminate child, if child is still running
	if (request.terminate > 0) {
	    errno = 0;
	    pid = waitpid(request.terminate, &status, WNOHANG);
	    if (pid == 0 && errno == 0) {
		if (kill(request.terminate, SIGHUP) == 0) {
		    log_debug("[server] Terminating child %d! [HUP]", request.terminate);
		} else {
		    log_errno("[server] Terminating child %d failed!", request.terminate);
		}
	    }
	    continue;
	}

	if ((url = ZALLOC(request.urlLength + 1)) == NULL) {
	    log_errno("[server] Could not allocate URL!");
	    break;
	}
readURL:
//	if (!sock_watch_single(fd, TRUE, 100))
//	    continue;
	len = sock_read(fd, url, request.urlLength + 1, 100);
	if (len != request.urlLength + 1 && errno != EINTR) {
	    log_errno("[server] Failed to read URL!");
	    ZFREE(url);
	    break;
	}
	while ((pid = waitpid(-1, &status, WNOHANG)) > 0) {
	    log_debug("[server] Child %d exited with exit code %d.", pid, WEXITSTATUS(status));
	}
	if (len != request.urlLength + 1)
	    goto readURL;

	if ((request.service < 0) || (request.service >= data->numServices)) {
	    log_errno("[server] Wrong SERVICE (%d-%d)!", request.service, data->numServices);
//	    break;
continue;
	}

	// Sanitize peer name and real IP, so that we do not pass any unexpected
	// characters to our child process.
	request.peerName[sizeof(request.peerName)-1] = '\000';
	for (char *s = request.peerName; *s; s++) {
	    if (!((*s >= '0' && *s <= '9') ||
		(*s >= 'A' && *s <= 'Z') ||
		(*s >= 'a' && *s <= 'z') ||
		*s == '.' || *s == '-')) {
		    *s = '-';
	    }
	}

	request.realIP[sizeof(request.realIP)-1] = '\000';
	for (char *s = request.realIP; *s; s++) {
	    if (!((*s >= '0' && *s <= '9') ||
		(*s >= 'A' && *s <= 'Z') ||
		(*s >= 'a' && *s <= 'z') ||
		*s == '.' || *s == '-')) {
		    *s  = '-';
	    }
	}

	// Fork and exec the child process.
	if ((pid = forkPty(&pty,
		data->services[request.service],
		request.peerName,
		request.realIP)) == 0) {
	    childProcess(data->services[request.service], request.width, request.height,
		    request.peerName, request.realIP, url);
	    ZFREE(url);
	    _exit(1);
	} else {
	    // Remember the utmp entry so that we can clean up when the child
	    // terminates.
	    ZFREE(url);
	    if (pid > 0) {
		log_debug("[server] Child %d launched", pid);
	    } else {
		int fds[2];

		if (os_pipe(fds) == 0) {
		    (void)sock_write(fds[1], "forkpty() failed\r\n", 18, 100);
		    close(fds[1]);
		    pty = fds[0];
		    pid = 0;
		}
	    }

	    // Send file handle and process id back to parent

	    memset (cmsg_buf, 0, sizeof (cmsg_buf)); // Quiet complaint from valgrind
	    iov.iov_base            = &pid;
	    iov.iov_len             = sizeof(pid);
	    msg.msg_iov             = &iov;
	    msg.msg_iovlen          = 1;
	    msg.msg_control         = &cmsg_buf;
	    msg.msg_controllen      = sizeof(cmsg_buf);
	    if ((cmsg = CMSG_FIRSTHDR(&msg)) == NULL) {
		log_err("[Server] CMSG is NULL");
		break;
	    }
	    cmsg->cmsg_level        = SOL_SOCKET;
	    cmsg->cmsg_type         = SCM_RIGHTS;
	    cmsg->cmsg_len          = CMSG_LEN(sizeof(int));
	    memcpy(CMSG_DATA(cmsg), &pty, sizeof(int));
	    if (sendmsg(fd, &msg, 0) != sizeof(pid)) {
		log_errno("[Server] Could not write data back");
		break;
	    }
	    close(pty);
	}
    }
    _exit(0);
}

#ifndef HAVE_GETRESUID
int getresuid(uid_t *ruid, uid_t *euid, uid_t *suid) {
  *ruid = getuid();
  *euid = geteuid();
  *suid = -1;
  return 0;
}
#endif

int forkLauncher(struct shell_in_a_box_data * data) {
    int		pair[2];
    uid_t	tmp;
    pid_t	pid;

    if (data == NULL) {
	log_err("[server] Data struct is NULL!");
	return -1;
    }
    if (data->pid > 0)
	return data->launcher;
    if (socketpair(AF_UNIX, SOCK_STREAM, 0, pair) != 0) {
	log_err("[server] Could not create socket pair!");
	return -1;
    }
    pid = fork();
    switch (pid) {
	case 0:;
	    // If our real-uid is not "root", then we should not allow anybody to
	    // login unauthenticated users as anyone other than their own.
	    if (getresuid(&data->restricted, &tmp, &tmp) != 0) {
		log_errno("[server] Wrong uid/gid!");
		_exit(1);
	    }
	    // Temporarily drop most permissions. We still retain the ability to
	    // switch back to root, which is necessary for launching "login".
//	    lowerPrivileges();
	    proc_close_all_fds((int []){ pair[1], 2 }, 2);
	    if (fcntl(pair[1], F_SETFL, O_NONBLOCK | O_RDWR) != 0) {
		log_errno("[server] Child. Could not make socket nonblocking");
		_exit(1);
	    }
	    launcherDaemon(data, pair[1]);
	    log_err("[server] Launcher exit() failed!");
	break;
	case -1:
	    log_err("[server] Launcher fork() failed!");
	break;
	default:
	    close(pair[1]);
	    if (fcntl(pair[0], F_SETFL, O_NONBLOCK | O_RDWR) != 0) {
		log_errno("[server. Parent] Could not make socket nonblocking");
		os_kill((int)pid, SIGKILL);
		os_sleep(0, 100, 0);
		os_kill((int)pid, SIGTERM);
		return -1;
	    }
	    data->pid = (int)pid;
	    data->launcher = pair[0];
	    return data->launcher;
	break;
    }
    return -1;
}

void terminateLauncher(struct shell_in_a_box_data * data) {
    if ((data != NULL) && (data->launcher >= 0)) {
	close(data->launcher);
	data->launcher = -1;
	os_kill((int)data->pid, SIGKILL);
	os_sleep(0, 100, 0);
	os_kill((int)data->pid, SIGTERM);
	data->pid = 0;
    }
}
