/*
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 2 of the License, or
 *	(at your option) any later version.
 *	HTTP digest auth functions: originally imported from libmicrohttpd
 *
 *	Copyright (C) 2012-2014 PIVA SOFTWARE (www.pivasoftware.com)
 *		Author: Oussama Ghorbel <oussama.ghorbel@pivasoftware.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <plib/crypt.h>
#include "../httpint.h"
#include "digestauth.h"

/* Server part */

/**
 * Calculate the server nonce so that it mitigates replay attacks
 * The current format of the nonce is ...
 * H(timestamp ":" method ":" random ":" uri ":" realm) + Hex(timestamp)
 *
 * @param nonce_time The amount of time in seconds for a nonce to be invalid
 * @param method HTTP method
 * @param rnd A pointer to a character array for the random seed
 * @param rnd_size The size of the random seed array
 * @param uri HTTP URI (in MHD, without the arguments ("?k=v")
 * @param realm A string of characters that describes the realm of auth.
 * @param nonce A pointer to a character array for the nonce to put in
 */
static void calculate_nonce(uint32_t nonce_time, const char *method,
		const char *rnd, unsigned int rnd_size, const char *uri,
		const char *realm, char *nonce, size_t noncelen) {
	P_MD5_CTX md5;
	unsigned char timestamp[4];
	unsigned char tmpnonce[MD5_DIGEST_SIZE];
	char timestamphex[sizeof(timestamp) * 2 + 1];

	memset(nonce, 0, noncelen);
	P_MD5_Init(&md5);
	timestamp[0] = (nonce_time & 0xff000000) >> 0x18;
	timestamp[1] = (nonce_time & 0x00ff0000) >> 0x10;
	timestamp[2] = (nonce_time & 0x0000ff00) >> 0x08;
	timestamp[3] = (nonce_time & 0x000000ff);
	P_MD5_Update(&md5, timestamp, 4);
	P_MD5_Update(&md5, (const unsigned char *)":", 1);
	P_MD5_Update(&md5, (const unsigned char *)method, strlen(method));
	P_MD5_Update(&md5, (const unsigned char *)":", 1);
	if (rnd_size > 0)
		P_MD5_Update(&md5, (const unsigned char *)rnd, rnd_size);
	P_MD5_Update(&md5, (const unsigned char *)":", 1);
	P_MD5_Update(&md5, (const unsigned char *)uri, strlen(uri));
	P_MD5_Update(&md5, (const unsigned char *)":", 1);
	P_MD5_Update(&md5, (const unsigned char *)realm, strlen(realm));
	P_MD5_Final(tmpnonce, &md5);
	hexify(nonce, noncelen, (const char *)tmpnonce, sizeof(tmpnonce));
	hexify(timestamphex, sizeof(timestamphex), (const char *)timestamp, 4);
	strncat(nonce, timestamphex, 8);
}

static char * __prepare_uri(HTTP_REQUEST * http, char * buf, size_t len) {
	char	uri[1024], * ptr;

	uri_format(uri, sizeof(uri), http->uri);
	if ((ptr = strstr(uri, "://")) != NULL) {
		char	* sptr;

		ptr += 3;
		sptr = ptr;
		if ((ptr = strstr(sptr, "/")) == NULL)
			ptr = sptr;
		else
			ptr ++;
	} else
		ptr = uri;
	strncpy(buf, ptr, len);
	return buf;
}

/**
 * make response to request authentication from the client
 *
 * @param fp
 * @param http_method
 * @param opaque string to user for opaque value
 * @return HTTP_ERROR_CODES
 */

HTTP_ERROR_CODES http_digest_auth_fail_response(HTTP_REQUEST * http, const char * opaque) {
	char	header[1024], uri[1024];
	char	nonce[HASH_MD5_HEX_LEN + 9];
	int	signal_stale = 0;
	val_t	* val;
	struct http_server	* srv = http->app->localcfg;;
	const char	* priv_key, * realm = (SEMPTY(srv->auth_data.realm) ? srv->domain : srv->auth_data.realm);

	if ((val = val_new(__http_authen_str)) == NULL)
	    return HTTP_ERR_OUT_OF_MEMORY;

	priv_key = http_get_digest_nonce_priv_key();
	__prepare_uri(http, uri, sizeof(uri));
	/* Generating the server nonce */
	calculate_nonce((uint32_t)(os_time_msec() / 1000), http_get_method_str(http),
			priv_key, strlen(priv_key), uri, realm, nonce, sizeof(nonce));
	/* Building the authentication header value */
	snprintf(header, sizeof(header),
		"Digest realm=\"%s\",qop=\"auth\",nonce=\"%s\",opaque=\"%s\"%s",
		realm, nonce, (SEMPTY(opaque) ? OPAQUE : opaque), signal_stale ? ",stale=\"true\"" : "");
	if (!val_from_str(val, header)) {
	    val_free(val);
	    return HTTP_ERR_OUT_OF_MEMORY;
	}
	http_set_resp_header(http, val);
	return HTTP_ERR_NOT_AUTHORIZED;
}

/**
 * Authenticates the authorization header sent by the client
 *
 * @param http_method
 * @param url
 * @param header: pointer to the position just after the string "Authorization: Digest "
 * @param realm The realm presented to the client
 * @param username The username needs to be authenticated
 * @param password The password used in the authentication
 * @param nonce_timeout The amount of time for a nonce to be
 *			invalid in seconds
 * @return TRUE if authenticated, FALSE if not,
 *			FALSE if nonce is invalid
 */
int http_digest_auth_check(HTTP_REQUEST * http, const char * header, unsigned int nonce_timeout) {
	size_t len;
	char *end;
	char nonce[HTTP_MAX_NONCE_LENGTH];
	char cnonce[HTTP_MAX_NONCE_LENGTH];
	char qop[15]; /* auth,auth-int */
	char nc[20];
	char response[HTTP_MAX_AUTH_RESPONSE_LENGTH];
	const char *hentity = NULL; /* "auth-int" is not supported */
	char ha1[HASH_MD5_HEX_LEN + 1];
	char respexp[HASH_MD5_HEX_LEN + 1];
	char noncehashexp[HASH_MD5_HEX_LEN + 9];
	uint32_t nonce_time;
	uint32_t t;
	size_t left; /* number of characters left in 'header' for 'uri' */
	unsigned long int nci;
	const char	* priv_key = http_get_digest_nonce_priv_key();
	struct http_server	* srv = http->app->localcfg;
	const char	* realm = (SEMPTY(srv->auth_data.realm) ? srv->domain : srv->auth_data.realm), * username = srv->auth_data.username, * password = srv->auth_data.password;

	log_debug("header: %s", header);

	left = strlen(header);

	{
		char un[HTTP_MAX_USERNAME_LENGTH];

		len = http_lookup_sub_value(un, sizeof(un), header, "username");
		if ((0 == len) || (0 != strcmp(username, un))) {
			log_err("Authentication failed: Invalid username (%s : %s).", username, un);
			return -1;
		}
		left -= strlen("username") + len;
	}

	{
		char r[HTTP_MAX_REALM_LENGTH];

		len = http_lookup_sub_value(r, sizeof(r), header, "realm");
		if ((0 == len) || (0 != strcmp(realm, r))) {
			log_err("Authentication failed, no realm");
			return -1;
		}
		left -= strlen("realm") + len;
	}

	if (0 == (len = http_lookup_sub_value(nonce, sizeof(nonce), header, "nonce"))) {
		log_err("Authentication failed, no nonce");
		return -1;
	}
	left -= strlen("nonce") + len;

	{
		char uri[left], muri[left];

		if (0 == http_lookup_sub_value(uri, sizeof(uri), header, "uri")) {
			log_err("Authentication failed, no uri");
			return -1;
		}

		/* 8 = 4 hexadecimal numbers for the timestamp */
		nonce_time = strtoul(nonce + len - 8, (char **) NULL, 16);
		t = (uint32_t)(os_time_msec() / 1000);
		/*
		 * First level vetting for the nonce validity if the timestamp
		 * attached to the nonce exceeds `nonce_timeout' then the nonce is
		 * invalid.
		 */
		if ((t > nonce_time + nonce_timeout)
				|| (nonce_time + nonce_timeout < nonce_time))
			return 0;

		__prepare_uri(http, muri, sizeof(muri));

		if (0 != strncmp(uri, muri, strlen(muri)))
		{
			log_err("Authentication failed: URI does not match (%s : %s).", uri, muri);
			return -1;
		}

		calculate_nonce(nonce_time, http_get_method_str(http), priv_key,
				strlen(priv_key), uri, realm, noncehashexp, sizeof(noncehashexp));

		/*
		 * Second level vetting for the nonce validity
		 * if the timestamp attached to the nonce is valid
		 * and possibly fabricated (in case of an attack)
		 * the attacker must also know the random seed to be
		 * able to generate a "sane" nonce, which if he does
		 * not, the nonce fabrication process going to be
		 * very hard to achieve.
		 */

		if (0 != strcmp(nonce, noncehashexp)) {
			log_info("Invalid nonce (%s : %s). May be method has been changed", nonce, noncehashexp);
			return 0;
		}

		if ((0 == http_lookup_sub_value(cnonce, sizeof(cnonce), header, "cnonce"))
				|| (0 == http_lookup_sub_value(qop, sizeof(qop), header, "qop"))
				|| ((0 != strcmp(qop, "auth")) && (0 != strcmp(qop, "")))
				|| (0 == http_lookup_sub_value(nc, sizeof(nc), header, "nc"))
				|| (0
						== http_lookup_sub_value(response, sizeof(response), header,
								"response"))) {
			log_err("Authentication failed, invalid format");
			return -1;
		}
		nci = strtoul(nc, &end, 16);
		if (('\0' != *end) || ((LONG_MAX == nci) && (ERANGE == errno))) {
			log_err("Authentication failed, invalid format");
			return -1; /* invalid nonce format */
		}

		if ((srv->auth_data.handler != NULL) && (srv->auth_data.data != NULL)) {
			PEER_ADDR	* remote = http->data;
			PEER_ADDR	* local = remote->parent;

			int	retcode = (*srv->auth_data.handler)(remote, local, username, password, srv->auth_data.data);

			if (retcode >= 0)
				http->app->validated_level = retcode;
			else {
				log_err("Authentication failed: Could not authenticate (%s:%s) against external validator", username, password);
				return -1;
			}
		}

		/*
		 * Checking if that combination of nonce and nc is sound
		 * and not a replay attack attempt. Also adds the nonce
		 * to the nonce-nc map if it does not exist there.
		 */

		digest_calc_ha1("md5", username, realm, password, nonce, cnonce, ha1, sizeof(ha1));
		digest_calc_response(ha1, nonce, nc, cnonce, qop, http_get_method_str(http), uri,
				hentity, respexp, sizeof(respexp));
		if (0 == strcmp(response, respexp)) {
			return 1;
		} else {
			log_err("Authentication failed: Could not authenticate (%s:%s)", response, respexp);
		}
	}
	return -1;
}
