#include <plib/httpserver.h>
#include <plib/basenet.h>
#include <plib/exec.h>
#include <sys/wait.h>
#include "cgi.h"
#include "../httpint.h"
#include <sys/wait.h>

/* This routine is used for parsed-header CGIs and for all SSL CGIs. */
static void cgi_interpose_output(HTTP_REQUEST * http, int rfd) {
	char	buf[ASCIILINESZ], headers[ASCIILINESZ], * hptr = headers, * cp, * br = buf, datetimebuf[VAL_KEY_LEN];
	ssize_t	r = sizeof(buf), r2, written;
	BOOL	close_header = FALSE, add_cl = FALSE;
	struct http_server	* srv = http->app->localcfg;
	size_t	hremain = sizeof(headers);
	HTTP_ERROR_CODES	errn;

	blob_rewind(http->body);
	blob_len(http->body) = 0;

	/* Header parsing.  The idea here is that the CGI can return special
	** headers such as "Status:" and "Location:" which change the return
	** status of the response.  Since the return status has to be the very
	** first line written out, we have to accumulate all the headers
	** and check for the special ones before writing the status.  Then
	** we write out the saved headers and proceed to echo the rest of
	** the response.
	*/

	while (r > 0) {
	    r2 = sock_read(rfd, br, r, srv->cgi_data.timeout * 1000);
	    if (r2 <= 0)
		break;
	    br += r2;
	    r -= r2;
	}
	if (r == sizeof(buf)) {
		sock_close(rfd);
		http_server_send_indication(http, HTTP_ERR_CANT_READ);
		return;
	}
	r = sizeof(buf) - r;
	buf[r] = 0;

	/* If there were no headers, bail. */
	if ((br = strstr(buf, __http_hdr_end)) != NULL) {
		char	* ptr;

		if (((ptr = strstr(buf, "HTTP/1.")) != NULL) && (ptr < br)) {
			/* Assume that we have full prepared reply. Just echo it */
			goto echo_rest;
		} else {
			int	s = 200;
			const char	* title;

			/* Figure out the status. */
			if (((cp = strstr(buf, "Location:")) != NULL) && (cp < br) && ((cp == buf) || (*(cp-1) == '\012')))
				s = 302;
			if (((cp = strstr(buf, "Status:")) != NULL) && (cp < br) && ((cp == buf) || (*(cp-1) == '\012'))) {
				cp += 7;
				cp += strspn(cp, " \t" );
				s = atoi(cp);
			}

			/* Write the status line. */
			switch (s) {
				case 200: title = "OK"; break;
				case 302: title = "Found"; break;
				case 304: title = "Not Modified"; break;
				case 400: title = "Bad Request"; break;
				case 401: title = "Unauthorized"; break;
				case 403: title = "Forbidden"; break;
				case 404: title = "Not Found"; break;
				case 408: title = "Request Timeout"; break;
				case 500: title = "Internal Error"; break;
				case 501: title = "Not Implemented"; break;
				case 503: title = "Service Temporarily Overloaded"; break;
				default: title = "Something"; break;
			}
			http->retcode = s;
			written = snprintf(hptr, hremain, "HTTP/1.%d %d %s\r\n", ((http->proto == HTTP_PROTO_V11) ? 1 : 0), s, title);
			hptr += written;
			hremain -= written;
			if (!SEMPTY(http->app->appname) && (hremain > 0)) {
			    written = snprintf(hptr, hremain, "Server: %s\r\n", http->app->appname);
			    hptr += written;
			    hremain -= written;
			}
			time_to_rfc1123str(datetimebuf, sizeof(datetimebuf), time(NULL));
			written = snprintf(hptr, hremain, "Date: %s\r\n", datetimebuf);
			hptr += written;
			hremain -= written;
			if (http->proto == HTTP_PROTO_V11) {
			    if (http->flags & HTTP_FLAG_CLOSE_CONNECTION) {
				written = snprintf(hptr, hremain, "Connection: close\r\n");
			    } else {
				written = snprintf(hptr, hremain, "Connection: keep-alive\r\nKeep-Alive: timeout=5; max=1000\r\n");
			    }
			    hptr += written;
			    hremain -= written;
			}
			if (((cp = strstr(buf,  "Content-length")) == NULL) || (cp > br)) {
				add_cl = TRUE;
			} else if (((cp = strstr(buf,  __http_content_len_str)) == NULL) || (cp > br)) {
				/* Broken output. Try to fix it by adding Content-Lenght */
				add_cl = TRUE;
			} else {
				*br = 0;
				written = snprintf(hptr, hremain, "%s%s", buf, __http_hdr_end);
				hptr += written;
				hremain -= written;
				http_write(http, headers, (size_t)(hptr - headers));
				br ++;
				br ++;
				http_write(http, br, r - (br - buf));
				/* Echo rest to the peer */
				goto echo_rest_next;
			}
		}
	} else {
		/* If we're did not find headers, write out the default status line.
		** try to add content length and type and proceed to the echo phase.
		*/
		char http_head[] = "HTTP/1.%d 200 OK\r\n";

		written = snprintf(hptr, hremain, http_head, ((http->proto == HTTP_PROTO_V11) ? 1 : 0));
		hptr += written;
		hremain -= written;
		if (!SEMPTY(http->app->appname) && (hremain > 0)) {
		    written = snprintf(hptr, hremain, "Server: %s\r\n", http->app->appname);
		    hptr += written;
		    hremain -= written;
		}
		time_to_rfc1123str(datetimebuf, sizeof(datetimebuf), time(NULL));
		written = snprintf(hptr, hremain, "Date: %s\r\n", datetimebuf);
		hptr += written;
		hremain -= written;
		if (http->proto == HTTP_PROTO_V11) {
		    if (http->flags & HTTP_FLAG_CLOSE_CONNECTION) {
			written = snprintf(hptr, hremain, "Connection: close\r\n");
		    } else {
			written = snprintf(hptr, hremain, "Connection: keep-alive\r\nKeep-Alive: timeout=5; max=1000\r\n");
		    }
		    hptr += written;
		    hremain -= written;
		}
		add_cl = TRUE;
		close_header = TRUE;
	}

	(void)blob_write(http->body, buf, (size_t)r);
	/* Echo the rest of the output. */
	while ((r = sock_read(rfd, buf, sizeof(buf), srv->cgi_data.timeout * 1000)) > 0) {
		r2 = blob_write(http->body, buf, (size_t)r);
		if (r2 != r) {
			sock_close(rfd);
			http_server_send_indication(http, HTTP_ERR_OUT_OF_MEMORY);
			return;
		}
	}
	blob_rewind(http->body);
	r2 = blob_len(http->body);
	br = (char *)blob_data(http->body);
	if (add_cl || close_header) {
		r = r2;
		if ((cp = strstr(br, __http_hdr_end)) == NULL) {
			close_header = TRUE;
		} else {
			if (strstr(br, __http_content_len_str) != NULL)
				add_cl = FALSE;
			r = r2 - strlen(__http_hdr_end) - (cp - br);
		}
		if (add_cl) {
			written = snprintf(hptr, hremain, "%s: %ld\r\n", __http_content_len_str, r);
			hptr += written;
			hremain -= written;
		}
		if (close_header) {
			written = snprintf(hptr, hremain, __http_head_end_fmt, __http_content_type_str, "text/html", web_get_charset(http->app));
			hptr += written;
			hremain -= written;
		}
	}
	http_write(http, headers, (size_t)(hptr - headers));
	http_write(http, br, r2);
	goto done;
echo_rest:
	(void)http_write(http, buf, (size_t)r);
echo_rest_next:
	while ((r = sock_read(rfd, buf, sizeof(buf), srv->cgi_data.timeout * 1000)) > 0) {
		r2 = http_write(http, buf, (size_t)r);
		if (r2 != r)
			break;
	}
done:
	sock_close(rfd);
	return;
}

/* Set up CGI argument vector.  We don't have to worry about freeing
** stuff since we're a sub-process.  This gets done after make_envp() because
** we scribble on query.
*/
static char ** make_argp(HTTP_REQUEST * http, int * argn) {
	char	** argp;
	char	* cp1, * cp2;
	size_t	len = 0;

	/* By allocating an arg slot for every character in the query, plus
	** one for the filename and one for the NULL, we are guaranteed to
	** have enough.  We could actually use strlen/2.
	*/
	*argn = 0;
	if (!SEMPTY(http->query))
		len = strlen(http->query);
	
	argp = (char**)ZALLOC((len + 2) * sizeof(char *));
	if (argp == NULL)
		return NULL;

	argp[0] = http->filename;
	*argn = 1;
	/* According to the CGI spec at http://hoohoo.ncsa.uiuc.edu/cgi/cl.html,
	** "The server should search the query information for a non-encoded =
	** character to determine if the command line is to be used, if it finds
	** one, the command line is not to be used."
	*/
	if ((len > 0) && (strchr(http->query, '=') == NULL)) {
		for (cp1 = cp2 = http->query; *cp2 != '\0'; ++cp2) {
			if (*cp2 == '+') {
				*cp2 = '\0';
				len = strlen(cp1);
				url_decode(cp1, len, cp1, len);
				argp[*argn] = cp1;
				cp1 = cp2 + 1;
				*argn += 1;
			}
		}
		if (cp2 != cp1) {
			len = strlen(cp1);
			url_decode(cp1, len, cp1, len);
			argp[*argn] = cp1;
			*argn += 1;
		}
	}

	argp[*argn] = (char*)NULL;
	return argp;
}

/* Set up CGI environment variables. Be real careful here to avoid
** letting malicious clients overrun a buffer.  We don't have
** to worry about freeing stuff since we're a sub-process.
*/

static ssize_t __format_cookie(char * ptr, size_t remain, const val_t * v, BOOL * flag) {
	ssize_t	len = 0;
	char	* cp;

	if ((v == NULL) || (ptr == NULL) || (remain < 1) || (flag == NULL))
		return len;
	cp = val_as_str_simple(v);
	if (!*flag) {
		*ptr++ = ';';
		*ptr = 0;
		remain --;
		len ++;
	} else
		*flag = FALSE;
	len += snprintf(ptr, remain, "%s=%s", val_name(v), cp);
	ZFREE(cp);
	return len;
}

static char** make_envp(HTTP_REQUEST * http) {
	char		** envp;
	int		envn = 0;
	char		* cp;
	const char	* ccp;
	char		buf[ASCIILINESZ];
	size_t		clen;
	PEER_ADDR	* peer = http->data;

	if ((envp = ZALLOC(sizeof(char *) * 50)) == NULL)
		return NULL;
	envp[envn++] = os_sprintf("PATH=%s", CGI_PATH);
	envp[envn++] = os_sprintf("LD_LIBRARY_PATH=%s", CGI_LD_LIBRARY_PATH);
	envp[envn++] = os_sprintf("GATEWAY_INTERFACE=CGI/1.1", NULL);
	envp[envn++] = os_sprintf( "SERVER_SOFTWARE=%s", http->app->appname);
	envp[envn++] = os_sprintf("REQUEST_METHOD=%s", http_get_method_str(http));
	envp[envn++] = os_sprintf( "SCRIPT_FILENAME=%s", realpath(http->filename, buf));
	envp[envn++] = os_sprintf("SERVER_PROTOCOL=HTTP/1.0", NULL);
	if (!SEMPTY(http->query))
		envp[envn++] = os_sprintf("QUERY_STRING=%s", http->query);
	cp = getenv("TZ");
	if (cp != NULL)
		envp[envn++] = os_sprintf("TZ=%s", cp);
	ccp = http_get_content_type(http);
	if (!SEMPTY(ccp))
		envp[envn++] = os_sprintf("CONTENT_TYPE=%s", ccp);
	if ((clen = http_get_content_len(http)) > 0) {
		(void) snprintf(buf, sizeof(buf), "%lu", (unsigned long) clen);
		envp[envn++] = os_sprintf("CONTENT_LENGTH=%s", buf);
	}
	cp = addr_ntoa(&peer->addr);
	envp[envn++] = os_sprintf("REMOTE_HOST=%s", cp);
	envp[envn++] = os_sprintf("REMOTE_ADDR=%s", cp);
	(void)snprintf(buf, sizeof(buf), "%d", (int)peer->port);
	envp[envn++] = os_sprintf("REMOTE_PORT=%s", buf);
	if (peer->parent != NULL) {
		envp[envn++] = os_sprintf("SERVER_ADDR=%s", addr_ntoa(&peer->parent->addr));
		(void)snprintf(buf, sizeof(buf), "%d", (int)peer->parent->port);
		envp[envn++] = os_sprintf("SERVER_PORT=%s", buf);
	}
	ccp = http_get_user_agent(http);
	if (!SEMPTY(ccp))
		envp[envn++] = os_sprintf("HTTP_USER_AGENT=%s", ccp);
	ccp = val_as_str_ptr(dict_find_node_near(http->req_headers, __http_host_str, TRUE));
	if (!SEMPTY(ccp)) {
		envp[envn++] = os_sprintf("HTTP_HOST=%s", ccp);
		envp[envn++] = os_sprintf("SERVER_NAME=%s", ccp);
	} else {
		struct addr	addr;

		if (sock_get_localaddr(http->conn->fd, &addr))
		    envp[envn++] = os_sprintf("HTTP_HOST=%s", addr_ntoa(&addr));
		else
		    envp[envn++] = os_sprintf("HTTP_HOST=%s", addr_ntoa(&peer->parent->addr));
		(void)gethostname(buf, sizeof(buf));
		envp[envn++] = os_sprintf("SERVER_NAME=%s", buf);
	}
	if (!SEMPTY(http->remoteuser))
		envp[envn++] = os_sprintf("REMOTE_USER=%s", http->remoteuser);
	if (http->flags & HTTP_FLAG_AUTH_BASIC)
		envp[envn++] = os_sprintf("AUTH_TYPE=%s", "Basic");
	else if (http->flags & HTTP_FLAG_AUTH_DIGEST)
		envp[envn++] = os_sprintf("AUTH_TYPE=%s", "Digest");
	ccp = val_as_str_ptr(dict_find_node_near(http->req_headers, "Referer", TRUE));
	if (SEMPTY(ccp))
		ccp = val_as_str_ptr(dict_find_node_near(http->req_headers, "Referrer", TRUE));
	if (!SEMPTY(ccp)) {
		envp[envn++] = os_sprintf("HTTP_REFERER=%s", ccp);
		envp[envn++] = os_sprintf("HTTP_REFERRER=%s", ccp);
	}
	if (http->req_cookies != NULL) {
		int	i;
		BOOL	first = TRUE;
		size_t	remain = sizeof(buf);
		char	* lptr = buf;

		buf[0] = 0;
		FOREACH_IN_DICT(http->req_cookies, i) {
			val_t	* v = V_CHILD(http->req_cookies, i);

			if (V_DICT_SIZE(v) > 0) {
				int	j;

				FOREACH_IN_DICT(v, j) {
					val_t	* vn = V_CHILD(v, j);

					ssize_t	len = __format_cookie(lptr, remain, vn, &first);
					lptr += len;
					remain -= len;
				}
			} else {
				ssize_t	len = __format_cookie(lptr, remain, v, &first);
				lptr += len;
				remain -= len;
			}
		}
		if (!SEMPTY(buf))
			envp[envn++] = os_sprintf("HTTP_COOKIE=%s", buf);
	}
	if (http->uri != NULL) {
		struct http_server		* srv = http->app->localcfg;

/*		envp[envn++] = os_sprintf("PATH_INFO=/%s", (SEMPTY(http->uri->uri_path) ? "" : http->uri->uri_path));*/
		(void) snprintf( buf, sizeof(buf), "%s%s", (SEMPTY(http->uri->uri_path) ? "" : http->uri->uri_path), (SEMPTY(http->uri->uri_filename) ? "" : http->uri->uri_filename));
		envp[envn++] = os_sprintf( "SCRIPT_NAME=%s", buf);
		(void) snprintf( buf, sizeof(buf), "%s%s", srv->webroot, (SEMPTY(http->uri->uri_path) ? "" : http->uri->uri_path));
		envp[envn++] = os_sprintf("PATH_TRANSLATED=%s", buf);
	}
	envp[envn] = (char*)NULL;
	return envp;
}

static HTTP_REQUEST	* __req = NULL;

static void _handle_exec_timeout(int sig) {
    log_err("CGI execution takes a lot of time. Cancel it");
    if (__req != NULL)
	http_server_send_indication(__req, HTTP_ERR_CANT_OPEN);
    exit(-1);
}

static void process_cgi_req(HTTP_REQUEST * http) {
    char	** argp, ** envp, * binary, * directory;
    int		pid, argc, p[2];
    BOOL	flag = (BOOL)((http->method == HTTP_REQ_POST) || IS_REQ_SSL_MODE(http));
    struct http_server	* srv = http->app->localcfg;

    proc_close_all_fds((int []){ http->conn->fd, STDIN_FILENO, STDOUT_FILENO, STDERR_FILENO }, 4);
    if (((argp = make_argp(http, &argc)) == NULL) || (argc == 0)) {
	log_err("Could not make ARGs");
	http_server_send_indication(http, HTTP_ERR_OUT_OF_MEMORY);
	return;
    }
    if ((envp = make_envp(http)) == NULL) {
	log_err("Could not make ENVPs");
	http_server_send_indication(http, HTTP_ERR_OUT_OF_MEMORY);
	return;
    }
    if (flag) {
	if (os_pipe(p) != 0) {
	    log_err("Could not create pipe");
	    http_server_send_indication(http, HTTP_ERR_CANT_READ);
	    return;
	}
	pid = fork();
	if (pid < 0) {
	    log_err("Could not fork");
	    goto out;
	} else if (pid == 0) {
	    BOOL	cnt = TRUE;
	    size_t	remain = http->cl, len = blob_left(http->body);

	    close(p[0]);
	    if (len > 0) {
		sock_write(p[1], blob_data(http->body), len, 100);
		remain -= len;
	    }
	    if (remain > 0) {

		blob_rewind(http->body);
		blob_len(http->body) = 0;
		while ((remain > 0) && cnt) {
		    ssize_t	nr = http_read(http, blob_base(http->body), MIN(remain, blob_size(http->body)));

		    if (nr > 0) {
			sock_write(p[1], blob_base(http->body), nr, 100);
			remain -= nr;
		    } else if (remain > 0) {
			log_err("Could not read more from HTTP connection. Remain %lu bytes. Read %ld. FD %d", remain, nr, http->conn->fd);
			cnt = FALSE;
		    }
		}
	    }
	    if (cnt) {
		http_read_garbage(http);
	    } else {
		log_err("Read input to CGI process");
		http_server_send_indication(http, HTTP_ERR_CANT_READ);
	    }
	    exit(0);
	}
	(void)close(p[1]);
	if (p[0] != STDIN_FILENO) {
	    (void)dup2(p[0], STDIN_FILENO);
	    (void)close(p[0]);
	}
    }
    if (os_pipe(p) != 0) {
	log_err("Could not create pipe");
	http_server_send_indication(http, HTTP_ERR_CANT_READ);
	return;
    }
    __req = http;
#ifdef HAVE_SIGSET
    (void)sigset(SIGALRM, _handle_exec_timeout);
#else /* HAVE_SIGSET */
    (void)signal(SIGALRM, _handle_exec_timeout);
#endif /* HAVE_SIGSET */
    (void)alarm(srv->cgi_data.timeout);

    pid = fork();
    if (pid < 0) {
	log_err("Could not fork");
	goto out;
    } else if (pid == 0) {
	close(p[1]);
	cgi_interpose_output(http, p[0]);
	exit(0);
    }

    close(p[0]);
    if (p[1] != STDOUT_FILENO)
	(void)dup2(p[1], STDOUT_FILENO);
//    if (p[1] != STDERR_FILENO)
//	(void)dup2(p[1], STDERR_FILENO);
    if ((p[1] != STDOUT_FILENO) && (p[1] != STDERR_FILENO))
	(void)close(p[1]);
#ifdef HAVE_SIGSET
    (void) sigset(SIGPIPE, SIG_IGN);
#else /* HAVE_SIGSET */
    (void) signal(SIGPIPE, SIG_IGN);
#endif /* HAVE_SIGSET */

    /* Set priority. */
    (void)nice(CGI_NICE);

    directory = ZSTRDUP(http->filename);
    binary = strrchr(directory, '/');
    if (binary == (char*) 0)
	binary = http->filename;
    else {
	*binary++ = '\0';
	if (chdir(directory) != 0) {
	    log_errno("Could not change directory to %s", directory);
	    http_server_send_indication(http, HTTP_ERR_CANT_OPEN);
	    return;
	}
    }
    /* Run the program. */
    (void)execve(binary, argp, envp);

    /* Something went wrong. */
    log_errno("Could not execute %s", http->filename);
out:
    http_server_send_indication(http, HTTP_ERR_CANT_OPEN);
    return;
}

void do_cgi(HTTP_REQUEST * http) {
    struct http_server	* srv = http->app->localcfg;
    pid_t	pid = fork();
    sigset_t	sigorig, sigblock;
    struct sigaction sigchild_action = {
	.sa_handler = SIG_DFL,
	.sa_flags = SA_NOCLDWAIT
    };
    int status;

    sigfillset(&sigorig);
    sigemptyset(&sigblock);
    if (sigprocmask(SIG_SETMASK, &sigorig, &sigblock) != 0)
	log_errno("Could not block signals before fork");
    sigaction(SIGCHLD, &sigchild_action, NULL);
    if (pid < 0) {
	log_err("Could not execute CGI prog %s", http->filename);
	http_server_send_indication(http, HTTP_ERR_CANT_OPEN);
	return;
    } else if (pid == 0) {
	process_cgi_req(http);
	exit(0);
    }
    waitpid(pid, &status, WUNTRACED
#ifdef WCONTINUED       /* Not all implementations support this */
	| WCONTINUED
#endif
    );
    if (sigprocmask(SIG_SETMASK, &sigblock, NULL) != 0)
	log_errno("Could not restore original signals after fork");
    os_sleep(0, 10, 0);
    http_free_request(http);
}

BOOL http_server_is_file_for_cgi(HTTP_REQUEST * http, const char * filename) {

    if (http->app != NULL) {
	struct http_server	* srv = http->app->localcfg;

	if (srv != NULL) {
	    if (!SEMPTY(srv->cgi_data.directory)) {
		char	* ptr;

		if (((ptr = strstr(filename, srv->cgi_data.directory)) == NULL) || (ptr <= filename))
		    return FALSE;
		ptr --;
		if ((*ptr != '.') && (*ptr != '/'))
		    return FALSE;
		if (*ptr == '/') {
		    ptr --;
		    if (*ptr != '.')
			return FALSE;
		}
	    }
	    if (!SEMPTY(srv->cgi_data.pattern)) {
		const char	* ptr = srv->cgi_data.pattern;
		char	* fptr;

		do {
		    ptr = pat_tokenize(ptr, NULL, &fptr);
		    if (fptr != NULL) {
			if (strstr(filename, fptr) != NULL) {
			    ZFREE(fptr);
			    return TRUE;
			}
			ZFREE(fptr);
		    }
		} while (ptr != NULL);
	    }
	}
    }
    return FALSE;
}

PLEX BOOL http_server_add_cgi(struct http_server * srv, const char * cgi_directory, const char * cgi_pattern, int timeout) {
    if (srv == NULL) {
	log_err("Wrong server param %p", srv);
	return FALSE;
    }
    if (!SEMPTY(cgi_pattern))
	strncpy(srv->cgi_data.pattern, cgi_pattern, sizeof(srv->cgi_data.pattern));
    else {
	log_err("Empty CGI pattern");
	return FALSE;
    }
    if (timeout > 0)
	srv->cgi_data.timeout = timeout;
    else
	srv->cgi_data.timeout = CGI_HANDLER_TIMEOUT;
    if (!SEMPTY(cgi_directory))
	strncpy(srv->cgi_data.directory, cgi_directory, sizeof(srv->cgi_data.directory));
    return TRUE;
}
