#include <plib/httpserver.h>
#include <plib/thread.h>
#include <plib/basenet.h>
#include <plib/exec.h>

static int	exit_flag = 0;

static void _term_signal(int signum) {
    fputs("Got TERM signal", stderr);
    exit_flag = 1;
}

static void _int_signal(int signum) {
    fputs("Got INT signal", stderr);
    exit_flag = 2;
}

static struct http_server	srv;

int main(int argc, char ** argv) {
	BOOL		https_only = FALSE, debug = FALSE, digest = TRUE;
	int		ch, http_port = 80, https_port = 443, timeout = 30, poolsize = 4, i = 0;
	const char	* bind_addr = NULL, * cert = NULL, * sysuser = NULL, * sysgrp = NULL, * pidfile = NULL, * cgidir = NULL, * cgipat = NULL,
	* domain = "/", * webroot = NULL, * indexes = NULL, * servername = NULL, * realm = NULL, * auser = NULL, * apass = NULL;
	HTTP_PROTO_TYPE	proto = HTTP_PROTO_V11;

	while ((ch = getopt(argc, argv, "1bsva:B:C:D:H:N:P:R:U:a:c:d:i:h:o:p:t:u:g:")) != -1) {
	    char *ep;

	    switch (ch) {
		case '1':
		    proto = 0;
		break;
		case 'b':
		    digest = FALSE;
		break;
		case 's':
		    https_only = TRUE;
		break;
		case 'v':
		    debug = TRUE;
		break;
		case 'c':
		    cgipat = optarg;
		break;
		case 'd':
		    cgidir = optarg;
		break;
		case 'i':
		    indexes = optarg;
		break;
		case 'C':
		    cert = optarg;
		break;
		case 'D':
		    domain = optarg;
		break;
		case 'p':
		    pidfile = optarg;
		break;
		case 'u':
		    sysuser = optarg;
		break;
		case 'g':
		    sysgrp = optarg;
		break;
		case 'N':
		    servername = optarg;
		break;
		case 'P':
		    apass = optarg;
		break;
		case 'R':
		    realm = optarg;
		break;
		case 'U':
		    auser = optarg;
		break;
		case 'B':
		case 'a':
		    bind_addr = optarg;
		break;
		case 'h':
		    http_port = strtoul(optarg, &ep, 0);
		    if (*ep || ep == optarg)
			exit_flag = -1;
		break;
		case 'H':
		    https_port = strtol(optarg, &ep, 0);
		    if (*ep || ep == optarg)
			exit_flag = -1;
		break;
		case 't':
		    timeout = strtol(optarg, &ep, 0);
		    if (*ep || ep == optarg)
			exit_flag = -1;
		break;
		case 'o':
		    poolsize = strtol(optarg, &ep, 0);
		    if (*ep || ep == optarg)
			exit_flag = -1;
		break;
		default:
		    exit_flag = -1;
		break;
	    }
	}

	if (argc - optind == 1)
	    webroot = argv[optind];
	if (SEMPTY(webroot))
	    exit_flag = -1;

	if (exit_flag != 0) {
	    fprintf(stderr, "Usage: %s [OPTIONS] webrootdir\nwhere OPTIONS are:\n"
"\t-1\t\t\tUse HTTP v1.0 only\n"
"\t-b\t\t\tEnforce Basic auth. Requires Realm '-R', Username '-U' and Password '-P' options to be specified\n"
"\t-s\t\t\tRun HTTPS mode only. Requires certificate to be specified by '-C' option\n"
"\t-v\t\t\tVerbose output\n"
"\t-C <certificate>\tPath to certificate file. Mandatory with '-s' option\n"
"\t-D <domainame>\t\tDomain name\n"
"\t-H <port>\t\tPort for HTTPS service. Requires '-C' option\n"
"\t-R <realm>\t\tRealm to do authentication. Implements Digest auth if '-b' is not used\n"
"\t-U <username>\t\tUsername to do authentication. Implements Digest auth if '-b' is not used\n"
"\t-P <password>\t\tPassword to do authentication. Implements Digest auth if '-b' is not used\n"
"\t-N <servername>\t\tServer name\n"
"\t-a | -B  <address>\tIPv4 or IPv6 address to bind as HTTP server\n"
"\t-c <cgipattern>\t\tThe set of extensions to execute as CGI program\n"
"\t-d <cgidir>\t\tDirectory where matched files will be executed as CGI programs, Has sense with option '-c' only\n"
"\t-g <group>\t\tGroup to run HTTP server\n"
"\t-h <port>\t\tPort for HTTP service. Has sense without '-s' option\n"
"\t-i <indexes>\t\tList of index file names\n"
"\t-o <poolsize>\t\tAmount of service threads\n"
"\t-p <pidfile>\t\tPID file name\n"
"\t-t <timeout>\t\tConnection timeout in seconds\n"
"\t-u <user>\t\tUser to run HTTP server\n", argv[0]);

	    exit(exit_flag);
	}

	log_setlevel((debug ? 15 : 7));
	if (!http_server_config(&srv, poolsize, timeout, servername, domain, webroot, indexes, WEB_LANG_EN)) {
	    log_err("Could not initialize basic HTTP server config");
	    return -1;
	}
	port_signal(SIGINT, _int_signal);
	port_signal(SIGTERM, _term_signal);
#ifdef HAVE_SIGSET
	(void) sigset(SIGPIPE, SIG_IGN);
#else /* HAVE_SIGSET */
	(void) signal(SIGPIPE, SIG_IGN);
#endif

	if (!SEMPTY(realm) && !SEMPTY(auser) && !SEMPTY(apass))
	    http_server_add_auth(&srv, (digest ? WEB_APP_AUTH_DIGEST : WEB_APP_AUTH_BASIC),  realm,  auser, apass, NULL, NULL, 0);
	if (debug)
	    http_server_add_debug(&srv, TRUE);
	if (!SEMPTY(cert))
	    http_server_add_ssl(&srv, cert, NULL);
	if (!proto)
	    http_server_add_masks(&srv, proto, 0);
	http_server_add_cgi(&srv, cgidir, cgipat, 0);
	http_server_add_runinfo(&srv, sysuser, sysgrp, pidfile);
	if (bind_addr == NULL) {
	    if (!https_only && http_server_add_listener_any(&srv, http_port, FALSE))
		i ++;
	    if ((cert != NULL) && http_server_add_listener_any(&srv, https_port, TRUE))
		i ++;
	} else {
	    struct addr	addr;

	    if (addr_pton_a(bind_addr, &addr, FALSE) == 0) {
		if (!https_only && http_server_add_listener(&srv, &addr, http_port, FALSE))
		    i ++;
		if ((cert != NULL) && http_server_add_listener(&srv, &addr, https_port, TRUE))
		    i ++;
	    } else {
		log_err("Could not convert bind address %s to any valid IPv4/IPv6 address", bind_addr);
	    }
	}
	if (i > 0) {
	    http_server_run(&srv, &exit_flag);
	} else {
	    log_err("No any listener for HTTP server");
	}
	http_server_destroy(&srv);
	return exit_flag;
}
