#include <sys/stat.h>
#include <plib/fs.h>
#include <plib/crypt.h>
#include <plib/basenet.h>
#include <plib/httpclient.h>
#include "../httpint.h"

extern char * http_digest_auth_reply(const char *, const char *, const char *, const char *, const char *);
extern HTTP_ERROR_CODES http_parse_and_filter_cookie(HTTP_REQUEST *, char *);

static void http_close_connection(HTTP_REQUEST * ctx) {
    if ((ctx == NULL) || (ctx->conn == NULL))
	return;
    pstream_close(ctx->conn);
    ctx->conn = ZFREE(ctx->conn);
}

static HTTP_ERROR_CODES http_client_parse_header(HTTP_REQUEST * ctx) {
    HTTP_ERROR_CODES	retcode;
    val_t		** hdr = &ctx->resp_headers;
    int64_t		cl;
    BOOL	eof = FALSE;
    char	* body_start, * next_line;
    char	* fields[3];
    size_t	nf = 3;

    if ((*hdr = val_new("headers")) == NULL)
	return HTTP_ERR_OUT_OF_MEMORY;
	/* We read from socket in server or client mode */

    body_start = http_feed_next_part(ctx, __http_hdr_end, &eof);
    if (body_start == NULL)
	return HTTP_ERR_BAD_REQUEST;
    body_start += 2;
    *body_start = 0;
    body_start ++;
    *body_start = 0;
    body_start ++;
    next_line = strstr((char *)blob_data(ctx->body), __http_lf);
    if (next_line == NULL)
	return HTTP_ERR_BAD_REQUEST;
    *next_line = 0;
    next_line += 2;

    if (IS_DEBUG_MODE(ctx->app))
	fprintf(stderr, "%s\n", (char *)blob_data(ctx->body));

    split((char *)blob_data(ctx->body), fields, &nf, " ");
    if (nf < 2)
	return HTTP_ERR_BAD_REQUEST;

    retcode = http_parse_protocol(ctx, fields[0]);
    if (retcode != HTTP_ERR_NONE) {
	ZFREE(fields[0]);
	ZFREE(fields[1]);
	ZFREE(fields[2]);
	return retcode;
    }

    ctx->retcode = atol(fields[1]);
    if ((ctx->retcode < 100) || (ctx->retcode >= 600) || SEMPTY(fields[2])) {
	ZFREE(fields[0]);
	ZFREE(fields[1]);
	ZFREE(fields[2]);
	return HTTP_ERR_BAD_RESPONSE;
    }

    if ((ctx->query = ZSTRDUP(fields[2])) == NULL)
	return HTTP_ERR_OUT_OF_MEMORY;

    ZFREE(fields[0]);
    ZFREE(fields[1]);
    ZFREE(fields[2]);
    blob_skip(ctx->body, next_line - (char *)blob_data(ctx->body));
    while ((next_line < body_start) && ((next_line = strstr((char *)blob_data(ctx->body), __http_lf)) != NULL)) {
	char	key[VAL_KEY_LEN], val[1024];

	*next_line = 0;
	next_line += 2;
	if (IS_DEBUG_MODE(ctx->app))
	    fprintf(stderr, "%s\n", (char *)blob_data(ctx->body));
	if (sscanf((char *)blob_data(ctx->body), "%127[^:] : %1023[^#]", key, val) == 2) {

	    if (strcmp(key, __http_set_cookie_str) == 0) {
		if ((retcode = http_parse_and_filter_cookie(ctx, val)) != HTTP_ERR_NONE)
		    return retcode;
	    } else {
		val_t	* v = val_new(key);

		if (v == NULL)
		    return HTTP_ERR_OUT_OF_MEMORY;
		val_dict_add(*hdr, v);
		val_from_str(v, val);
	    }
	} else
	    return HTTP_ERR_BAD_REQUEST;
	blob_skip(ctx->body, next_line - (char *)blob_data(ctx->body));
    }
    blob_seek(ctx->body, body_start - (char *)blob_base(ctx->body), SEEK_SET);

    if (val_as_int(dict_find_node_near(*hdr, __http_content_len_str, TRUE), &cl))
	ctx->cl = (size_t)cl;
    http_parse_lang(ctx);
    if (ctx->cl == 0) {
	const char	* cptr;

	cptr = val_as_str_ptr(dict_find_node_near(*hdr, __http_transfer_enc_str, TRUE));
	if (!SEMPTY(cptr) && (strstr(cptr, "hunked") != NULL)) {
	    if (ctx->proto == HTTP_PROTO_V11)
		ctx->encoding |= HTTP_TRANS_ENC_CHUNKED;
	    else
		return HTTP_ERR_BAD_REQUEST;
	}
    }
    http_debug_values(ctx, WEB_APP_RESPONSE_HEADER);
    http_debug_values(ctx, WEB_APP_RESPONSE_COOKIE);
    return HTTP_ERR_NONE;
}

static const char ua[] = "Mozilla/5.0 (X11; U; Linux i386; en-US) Gecko Firefox";

static ssize_t http_client_send_request(HTTP_REQUEST * ctx, char * buf, size_t remain) {
    BOOL	ua_exists = FALSE, q_printed = FALSE;
    int		len;
    const char	* cptr, * uptr;
    char	* bptr = buf;
    ssize_t	written;

    if (ctx->resp_headers != NULL) {
	val_free(ctx->resp_headers);
	ctx->resp_headers = NULL;
    }
    if ((ctx->flags & HTTP_USE_RESP_FROM_FILE_FLAG) || (buf == NULL) || (remain < 1))
	return 0;
    else if (ctx->flags & HTTP_USE_REQ_FROM_FILE_FLAG) {
	http_write_file(ctx, ctx->app->appname, FALSE);
	return 0;
    }

    if (((ctx->method & ctx->app->method_mask) != ctx->method) || ((cptr = http_get_method_str(ctx)) == NULL)) {
	log_err(http_error_strings[HTTP_ERR_UNSUPPORTED_METHOD]);
	return 0;
    }
    uptr = (SEMPTY(ctx->uri->uri_path) ? "" : ctx->uri->uri_path);
    if (*uptr == '/')
	uptr ++;
    written = snprintf(bptr, remain, "%s /%s%s", cptr, uptr, 
	(SEMPTY(ctx->uri->uri_filename) ? "" : ctx->uri->uri_filename));
    bptr += written;
    remain -= written;
    if (!SEMPTY(ctx->uri->uri_cmd)) {
	q_printed = TRUE;
	written = snprintf(bptr, remain, "?%s", ctx->uri->uri_cmd);
	bptr += written;
	remain -= written;
    }
    if (ctx->method == HTTP_REQ_GET) {
	written = http_pack_urlencoded(ctx, NULL, bptr, remain, q_printed, !q_printed);
	bptr += written;
	remain -= written;
    }

    if (ctx->proto == HTTP_PROTO_V10)
	written = snprintf(bptr, remain, " HTTP/1.%d", 0);
    else if (ctx->proto == HTTP_PROTO_V11)
	written = snprintf(bptr, remain, " HTTP/1.%d", 1);
    bptr += written;
    remain -= written;

    written = snprintf(bptr, remain, "\r\n%s: %s", __http_host_str, ctx->uri->uri_host);
    bptr += written;
    remain -= written;
    if (ctx->uri->uri_port > 0) {
	if (IS_REQ_SSL_MODE(ctx)) {
	    if (ctx->uri->uri_port != 443) {
		written = snprintf(bptr, remain, ":%5d", ctx->uri->uri_port);
		bptr += written;
		remain -= written;
	    }
	} else {
	    if (ctx->uri->uri_port != 80) {
		written = snprintf(bptr, remain, ":%5d", ctx->uri->uri_port);
		bptr += written;
		remain -= written;
	    }
	}
    }
    written = snprintf(bptr, remain, "\r\n");
    bptr += written;
    remain -= written;
    FOREACH_IN_DICT(ctx->req_headers, len) {
	char	* value;
	val_t	* val = V_CHILD(ctx->req_headers, len);

	if (strcmp(val->key, __http_content_type_str) == 0)
	    continue;
	if (strcmp(val->key, __http_content_len_str) == 0)
	    continue;
	if (strcmp(val->key, __http_ua_str) == 0)
	    ua_exists = TRUE;
	if ((value = val_as_str_simple(val)) != NULL) {
	    written = snprintf(bptr, remain, "%s: %s\r\n", val->key, value);
	    ZFREE(value);
	    bptr += written;
	    remain -= written;
	}
    }
    if (!ua_exists) {
	written = snprintf(bptr, remain, "%s: %s\r\n", __http_ua_str, ua);
	bptr += written;
	remain -= written;
    }
    written = snprintf(bptr, remain, "%s: %s", __http_accepted_lang_str, "en-US");
    bptr += written;
    remain -= written;
    if (ctx->app->lang & WEB_LANG_RU) {
	written = snprintf(bptr, remain, "%s", ",ru-RU");
	bptr += written;
	remain -= written;
    }
    if (ctx->app->lang & WEB_LANG_ES) {
	written = snprintf(bptr, remain, "%s", ",es-ES");
	bptr += written;
	remain -= written;
    }
    if (ctx->app->lang & WEB_LANG_FR) {
	written = snprintf(bptr, remain, "%s", ",fr-FR");
	bptr += written;
	remain -= written;
    }
    if (ctx->app->lang & WEB_LANG_ZH) {
	written = snprintf(bptr, remain, "%s", ",zh-ZH");
	bptr += written;
	remain -= written;
    }
    written = snprintf(bptr, remain, "\r\n");
    bptr += written;
    remain -= written;
    FOREACH_IN_DICT(ctx->ext_headers, len) {
	char	* value;
	val_t	* val = V_CHILD(ctx->ext_headers, len);

	if ((value = val_as_str_simple(val)) != NULL) {
	    written = snprintf(bptr, remain, "%s: %s\r\n", val->key, value);
	    ZFREE(value);
	    bptr += written;
	    remain -= written;
	}
    }
    if (ctx->proto == HTTP_PROTO_V11) {
	written = snprintf(bptr, remain, "Connection: keep-alive\r\n");
	bptr += written;
	remain -= written;
    }
    if ((ctx->method == HTTP_REQ_POST) || (ctx->method == HTTP_REQ_PUT)) {
    } else if (ctx->method == HTTP_REQ_PATCH) {
    } else if (ctx->method == HTTP_REQ_DELETE) {
    } else {
	written = snprintf(bptr, remain, "\r\n");
	bptr += written;
	remain -= written;
    }
    return (size_t)(bptr - buf);
}

static BOOL http_create_client_connection(HTTP_REQUEST * ctx) {
    struct pstream_ops	* ops = NULL;
    SOCK_T		fd;

    if (ctx->flags & HTTP_USE_RESP_FROM_FILE_FLAG) {
	if ((fd = open(ctx->uri->uri_filename, O_RDONLY | O_NONBLOCK | O_CLOEXEC)) < 0) {
	    log_errno("Could not open file to parse response %s", ctx->uri->uri_filename);
	    return FALSE;
	}
	return TRUE;
    }
    if ((fd = tcp_connect(ctx->uri->uri_host, ctx->uri->uri_port)) < 0) {
	log_err("Could not connect to host %s:%d", ctx->uri->uri_host, ctx->uri->uri_port);
	return FALSE;
    }

    ctx->form_headers = http_client_send_request;
    if (IS_REQ_SSL_MODE(ctx)) {
	if ((ops = pstream_init_ops_ssl(&ctx->app->app_ctx, fd)) == NULL) {
	    log_err("Could not create SSL context for connection");
	    goto out;
	}
    }
    if ((ctx->conn = pstream_init(NULL, fd, ops, ctx->app->connection_timeout)) != NULL)
	return TRUE;
out:
    sock_close(fd);
    return FALSE;
}

static BOOL http_parse_chunked(HTTP_REQUEST * ctx) {
    BOOL	eof = FALSE;

    while (!eof) {
	char	* ptr, * nptr;
	size_t	sz;

	while (!eof && ((ptr = strstr((char *)blob_data(ctx->body), __http_lf)) == NULL))
	    ptr = http_feed_next_part(ctx, __http_lf, &eof);
	if (ptr == NULL) {
	    log_err("Did not find required CRLF");
	    return FALSE;
	}
	*ptr = 0;
	nptr = ptr;
	sz = (size_t)strtoul((const char *)blob_data(ctx->body), &ptr, 16);
	if (ptr != nptr) {
	    log_err("Could not parse number");
	    return FALSE;
	}
	if (sz == 0)
	    return TRUE;
	blob_skip(ctx->body, ptr - (char *)blob_data(ctx->body) + 2);
	if (blob_left(ctx->body) > 0) {
	    size_t	to_process = MIN(blob_left(ctx->body), sz);

	    if (ctx->app->stream == NULL)
		(void)write(STDOUT_FILENO, blob_data(ctx->body), to_process);
	    else
		pstream_write_file(blob_data(ctx->body), to_process, sizeof(char), ctx->app->stream);
	    sz -= to_process;
	    blob_skip(ctx->body, to_process);
	}
	while (sz > 0) {
	    ssize_t	nread;
	    size_t	to_process = MIN(blob_size(ctx->body), sz);

	    blob_rewind(ctx->body);
	    nread = http_read(ctx, blob_base(ctx->body), to_process);
	    if ((size_t)nread != to_process) {
		log_err("Read %lu bytes but awaited %lu", (unsigned long)nread, (unsigned long)to_process);
		return FALSE;
	    }
	    ctx->body->end = to_process;
	    if (ctx->app->stream == NULL)
		(void)write(STDOUT_FILENO, blob_data(ctx->body), to_process);
	    else
		pstream_write_file(blob_data(ctx->body), to_process, sizeof(char), ctx->app->stream);
	    sz -= to_process;
	    blob_skip(ctx->body, to_process);
	}
	/* We reached end of blob or next CRLF */
	if (blob_left(ctx->body) < 2)
	    ptr = http_feed_next_part(ctx, __http_lf, &eof);
	/* Strip out CRLF before number */
	blob_skip(ctx->body, 2);
    }
    log_err("Something is going wrong when parse CHUNKED content");
    return FALSE;
}

static BOOL _http_check_scheme_ok(WEB_CONTEXT * ctx, const char * scheme) {
    ctx->http->flags &= ~(HTTP_SSL_FLAG | HTTP_USE_RESP_FROM_FILE_FLAG);
    if (IS_DEBUG_MODE(ctx) && (strcmp(scheme, "file") == 0)) {
	ctx->http->flags |= HTTP_USE_RESP_FROM_FILE_FLAG;
	return TRUE;
    }
    if (strcmp(scheme, "https") == 0) {
	if (IS_ALLOW_SSL(ctx)) {
	    ctx->http->flags |= HTTP_SSL_FLAG;
	    return TRUE;
	}
    } else if (strcmp(scheme, "http") == 0)
	return TRUE;
    return FALSE;
}

static char * _http_make_basic_auth(HTTP_REQUEST * ctx) {
    char	buf[512], * ptr;

    snprintf(buf, sizeof(buf), "%s:%s", ctx->uri->uri_username, ctx->uri->uri_password);
    if ((ptr =b64_encode(buf, strlen(buf))) == NULL)
	return ptr;
    snprintf(buf, sizeof(buf), "Basic %s", ptr);
    ZFREE(ptr);
    return ZSTRDUP(buf);
}

static char * _http_make_digest_auth(HTTP_REQUEST * ctx, const char * header) {
    char	url[1024];

    snprintf(url, sizeof(url), "%s%s", (SEMPTY(ctx->uri->uri_path) ? "/" : ctx->uri->uri_path), (SEMPTY(ctx->uri->uri_filename) ? "" : ctx->uri->uri_filename));
    return http_digest_auth_reply(http_get_method_str(ctx), url, header, ctx->uri->uri_username, ctx->uri->uri_password);
}

static BOOL _http_client_handle_auth(HTTP_REQUEST * ctx, char * ptr, BOOL ext) {
    val_t	* val;

    if ((ptr == NULL) || ((val = val_new(__http_author_str)) == NULL)) {
	log_err(http_error_strings[HTTP_ERR_OUT_OF_MEMORY]);
	return FALSE;
    }
    val_from_str(val, ptr);
    ZFREE(ptr);
    if (ext)
	http_set_ext_header(ctx, val);
    else
	http_set_req_header(ctx, val);
    return TRUE;
}

static void http_client_read_remain(HTTP_REQUEST * ctx, BOOL supressoutput) {

    if ((ctx->encoding & HTTP_TRANS_ENC_CHUNKED) == HTTP_TRANS_ENC_CHUNKED) {
	http_parse_chunked(ctx);
    } else {
	size_t	remain = 0, blen, offset = blob_data(ctx->body) - blob_base(ctx->body);

	blob_rewind(ctx->body);
	blob_delete(ctx->body, NULL, offset);
	blen = blob_len(ctx->body);
	blob_skip(ctx->body, blen);
	if (ctx->cl > 0)
	    remain = ctx->cl;
	if (remain >= blen)
	    remain -= blen;

	while (remain > 0) {
	    char	buf[BUFSIZ];
	    ssize_t	nread;

	    nread = http_read(ctx, buf, MIN(remain, sizeof(buf)));
	    if (nread <= 0)
		break;
	    remain -= nread;
	    blob_write(ctx->body, buf, (size_t)nread);
	}
	if (!supressoutput) {
	    blob_rewind(ctx->body);
	    if (ctx->app->stream == NULL) {
		write_blob_to_sock(ctx->body, STDOUT_FILENO);
	    } else {
		pstream_write_blob_to_file(ctx->app->stream, ctx->body);
	    }
	}
	blob_rewind(ctx->body);
	blob_len(ctx->body) = 0;
    }
}

static BOOL http_client_exec_request(HTTP_REQUEST * ctx) {
    BOOL	res = FALSE;
    int		attempts = 0;
    HTTP_ERROR_CODES	errn;

    if (http_create_client_connection(ctx)) {
	while (attempts < HTTP_MAX_REDIRECTS) {
	    if (http_start_output(ctx, NULL, 0)) {
		if ((ctx->method == HTTP_REQ_POST) || (ctx->method == HTTP_REQ_PUT))
		    res = http_pack_post_request(ctx, FALSE);
		blob_rewind(ctx->body);
		blob_len(ctx->body) = 0;
		errn = http_client_parse_header(ctx);
		http_client_read_remain(ctx, (errn != HTTP_ERR_NONE) || (ctx->retcode != 200));
		if (errn != HTTP_ERR_NONE) {
		    log_err(http_error_strings[errn]);
		    break;
		} else if ((ctx->retcode == 301) || (ctx->retcode == 302) || (ctx->retcode == 307) || (ctx->retcode == 308)) {
		    /* Handle moving */
		    val_t		* val = http_find_val_in_resp_header(ctx, "Location");
		    struct uri_t	* new_uri;
		    BOOL		new_conn = TRUE;;
		    const char	* location = val_as_str_ptr(val);

		    if (SEMPTY(location)) {
			log_err("Did not find new URL in the %ld response", ctx->retcode);
			break;
		    }
		    if ((new_uri = uri_parse(location)) == NULL) {
			log_err("Wrong URL %s in the redirect %ld response", location, ctx->retcode);
			break;
		    }
		    if ((val = val_new("Referer")) == NULL) {
			log_err("Could not create Referer %s in the redirect %ld response", location, ctx->retcode);
			uri_destroy(new_uri);
			break;
		    }

		    val_from_str(val, ctx->query);
		    http_set_req_header(ctx, val);
		    ZFREE(ctx->query);

		    if ((ctx->query = ZSTRDUP(location)) == NULL) {
			log_err(http_error_strings[HTTP_ERR_OUT_OF_MEMORY]);
			uri_destroy(new_uri);
			break;
		    }

		    uri_inherit_network_part(new_uri, ctx->uri);

		    if (!_http_check_scheme_ok(ctx->app, new_uri->uri_scheme)) {
			uri_destroy(new_uri);
			log_err("Scheme %s does not allowed", ctx->uri->uri_scheme);
			break;
		    }
		    if ((ctx->proto & HTTP_PROTO_V11) == HTTP_PROTO_V11) {
			new_conn = (new_uri->uri_port != ctx->uri->uri_port) ||
			(strcmp(new_uri->uri_scheme, ctx->uri->uri_scheme) != 0) ||
			(strcmp(new_uri->uri_host, ctx->uri->uri_host) != 0);
			if (!new_conn) {
			    if ((val = http_find_val_in_resp_header(ctx, "Connection")) != NULL) {
				location = val_as_str_ptr(val);

				if (pat_strcmp("?lose", location) == 0)
				    new_conn = TRUE;
			    }
			}
		    }
		    uri_destroy(ctx->uri);
		    ctx->uri = new_uri;
		    if (new_conn)
			http_close_connection(ctx);
		    if (strcmp(ctx->uri->uri_scheme, "https") == 0)
			ctx->app->mode |= WEB_APP_ALLOW_SSL;
		    else
			ctx->app->mode &= ~WEB_APP_ALLOW_SSL;
		    if (new_conn)
			http_create_client_connection(ctx);
		} else if (ctx->retcode == 401) {
		    /* Handle authorization */
		    BOOL	new_conn = TRUE;
		    int		i;
		    const char	* authb = NULL, * authd = NULL, * ptr;

		    if (SEMPTY(ctx->uri->uri_username) || SEMPTY(ctx->uri->uri_password)) {
			log_err("Site %s requires credentials to access", ctx->uri->uri_host);
			break;
		    }
		    FOREACH_IN_DICT(ctx->resp_headers, i) {
			val_t	* val = V_CHILD(ctx->resp_headers, i);

			if (authb != NULL)
			    break;
			if (strcmp(val->key, __http_authen_str) == 0) {
			    ptr = val_as_str_ptr(val);

			    if ((authb == NULL) && ((authb = strstr(ptr, "Basic")) != NULL))
				authb += 5;
			    if ((authd == NULL) && ((authd = strstr(ptr, "Digest")) != NULL))
				authd += 6;
			}
		    }
		    if (SEMPTY(authb) && SEMPTY(authd)) {
			log_err("Could not find Authenticate header for %ld response", ctx->retcode);
			break;
		    }
		    if (authb != NULL) {
			if (!_http_client_handle_auth(ctx, _http_make_basic_auth(ctx), TRUE))
			    break;
		    } else if (authd != NULL) {
			if (!_http_client_handle_auth(ctx, _http_make_digest_auth(ctx, authd), FALSE))
			    break;
		    }
		    if (attempts < (HTTP_MAX_REDIRECTS - 2))
			attempts = (HTTP_MAX_REDIRECTS - 2);
		    if ((ctx->proto & HTTP_PROTO_V11) == HTTP_PROTO_V11) {
			val_t	* val;

			new_conn = FALSE;
			if ((val = http_find_val_in_resp_header(ctx, "Connection")) != NULL) {
			    const char * cptr = val_as_str_ptr(val);

			    if (pat_strcmp("?lose", cptr) == 0)
				new_conn = TRUE;
			}
		    }
		    if (new_conn) {
			http_close_connection(ctx);
			http_create_client_connection(ctx);
		    }
		} else if ((ctx->retcode >= 200) && (ctx->retcode < 300)) {
		    res = TRUE;
		    break;
		} else {
		    break;
		}
	    } else
		log_err("Could not start output");
	    attempts ++;
	}
	http_close_connection(ctx);
    }
    ctx->elapsed = os_time_msec() - ctx->start_time;
    return res;
}

PLEX WEB_CONTEXT * http_client_request(const char * url, const val_t * hdr, const val_t * args, WEB_CLIENT_OPTIONS options, 
				pstream_t * stream, const char * cua, const char * cert, int connection_timeout) {
    WEB_CONTEXT	* ctx = web_alloc_ctx(url, WEB_APP_CLIENT, (SEMPTY(cua) ? ua : cua), cert, connection_timeout, NULL);

    if (ctx != NULL) {
	HTTP_ERROR_CODES	errn;

	if ((ctx->http = http_create_request(ctx, &errn)) != NULL) {
	    if (options & WEB_CLIENT_OPT_POST)
		ctx->http->method = HTTP_REQ_POST;
	    else if (options & WEB_CLIENT_OPT_OPTIONS)
		ctx->http->method = HTTP_REQ_OPTIONS;
	    else if (options & WEB_CLIENT_OPT_PUT)
		ctx->http->method = HTTP_REQ_PUT;
	    else if (options & WEB_CLIENT_OPT_TRACE)
		ctx->http->method = HTTP_REQ_TRACE;
	    else if (options & WEB_CLIENT_OPT_DEL)
		ctx->http->method = HTTP_REQ_DELETE;
	    else
		ctx->http->method = HTTP_REQ_GET;
	    if (options & WEB_CLIENT_OPT_HTTP10)
		ctx->http->proto = HTTP_PROTO_V10;
	    else
		ctx->http->proto = HTTP_PROTO_V11;
	    if (options & WEB_CLIENT_OPT_DEBUG)
		ctx->mode |= WEB_APP_DEBUG;
	    if ((ctx->http->uri = uri_parse(url)) != NULL) {
		if (_http_check_scheme_ok(ctx, ctx->http->uri->uri_scheme)) {
		    if (ctx->http->uri->uri_port <= 0) {
			if (ctx->http->flags & HTTP_SSL_FLAG)
			    ctx->http->uri->uri_port = 443;
			else
			    ctx->http->uri->uri_port = 80;
		    }
		    if (options & WEB_CLIENT_OPT_SSL_VERIFY) {
			ctx->app_ctx.flags |= SSL_APP_CTX_VERIFY_PEER;
			ctx->app_ctx.cn = ctx->http->uri->uri_host;
		    }
		    ctx->http->req_headers = hdr;
		    ctx->http->request = args;
		    ctx->stream = stream;
		    if (http_client_exec_request(ctx->http))
			return ctx;
		    else
			log_err("Something went wrong when deal with url %s", url);
		} else
		    log_err("Wrong scheme %s", ctx->http->uri->uri_scheme);
	    } else
		log_err("Wrong url %s", url);
	} else
	    log_err(http_error_strings[errn]);
	web_cleanup_ctx(ctx);
	ZFREE(ctx);
    } else
	log_err("Could not allocate context to deal with url %s", url);
    return NULL;
}
