#include "httpint.h"
#include <plib/httpserver.h>

const char	* __http_built_in_content_types[] = { "application/octet-stream", "text/plain", "text/html", 
"multipart/form-data", "multipart/mixed", 
"application/x-www-form-urlencoded", "application/binary", "application/json"
};

const char	__http_content_type_str[] = "Content-Type";
const char	__http_content_len_str[] = "Content-Length";
const char	__http_content_disp_str[] = "Content-Disposition";

const char	__http_ua_str[] = "User-Agent";
const char	__http_date_str[] = "Date";
const char	__http_from_str[] = "From";
const char	__http_host_str[] = "Host";
const char	__http_cookie_str[] = "Cookie";
const char	__http_set_cookie_str[] = "Set-Cookie";
const char	__http_accepted_lang_str[] = "Accepted-Language";
const char	__http_cache_ctl_str[] = "Cache-Control";
const char	__http_transfer_enc_str[] = "Transfer-Encoding";

const char	* http_error_strings[HTTP_ERR_NUM_ERRS] = {
  "No error", "Bad packet", "Unknown method", "Unsupported method", "Unknown protocol", "Unsupported protocol", "Incorrect Content Type",
  "NULL Query String", "Bad Content Length", "No memory", "No boundary found or mismatch", 
  "Could not open", "Could not read", "Bad or mailformed packet", "Bad response", "No cookies", "Not authorized", "Not found",
  "No content", "Redirect", "Internal error", "Forbidden", "Header is not readable",
};

const char	__http_bnd_str[] = "boundary=";

const char	__http_lf[] = "\r\n";
const char	__http_hdr_end[] = "\r\n\r\n";
const char	__http_bnd[] = "--";
const char	__http_authen_str[] = "WWW-Authenticate";
const char	__http_author_str[] = "Authorization";

static void __clean_resp_cookies(struct http_server_cookies * c) {
    HTTP_SERVER_COOKIE	* cookie;

    while ((cookie = TAILQ_FIRST(c)) != NULL) {
	TAILQ_REMOVE(c, cookie, entry);
	if (cookie->value != NULL)
	    val_free(cookie->value);
	if (cookie->domain != NULL)
	    ZFREE(cookie->domain);
	ZFREE(cookie);
    }
}

/*
 * set cookie "name" with "value" for the given "domain" and will expire in "expires" seconds
 * set "expires" to 0 for a session cookie, -1 to remove it
 */
static size_t http_output_one_server_cookie(char * buf, size_t len, val_t * val, const char * domain, time_t expires, BOOL httponly) {
    size_t	sz = 0;
    char	* ptr;

    if ((len == 0) || (buf == NULL))
	return sz;
    if ((ptr = val_as_str_simple(val)) == NULL)
	return sz;
    snprintf(buf, len, "%s: %s=%s; SameSite=Strict; Path=/", __http_set_cookie_str, val_name(val), ptr);
    ZFREE(ptr);
    if (expires == -1) {
	strlcat(buf, "; Expires=Sat, 01-Jan-1980 00:00:00 GMT", len);
    } else if (expires > 0) {
	char timebuf[VAL_KEY_LEN];

	strlcat(buf, "; Expires=", len);
	time_to_rfc1123str(timebuf, sizeof(timebuf), time(NULL) + expires);
	strlcat(buf, timebuf, len);
    }
    if (!SEMPTY(domain)) {
	strlcat(buf, "; Domain=", len);
	strlcat(buf, domain, len);
    }
    if (httponly)
	strlcat(buf, "; HttpOnly", len); /* some will probably see as a "gotcha" once the dude in charge for the HTML/Javascript code tell them something is not working, but trust me you probably want to activate this if you want to avoid kiddies collecting your users cookies */
    strlcat(buf, __http_lf, len);
    sz = strlen(buf);
    return sz;
}

size_t http_print_server_cookies(HTTP_REQUEST * http, char * buf, size_t len) {
    size_t	total = 0;

    if ((http != NULL) && (buf != NULL) && (len > 0)) {
	HTTP_SERVER_COOKIE	* cookie;
	char			* ptr = buf;
	size_t			remain = len;

	TAILQ_FOREACH(cookie, &http->resp_cookies, entry)  {
	    size_t	written = http_output_one_server_cookie(ptr, remain, cookie->value, cookie->domain, cookie->expires, cookie->httponly);
	    remain -= written;
	    ptr += written;
	    total += written;
	    if (remain == 0)
		break;
	}
    }
    return total;
}

PLEX void http_free_request_ex(HTTP_REQUEST * ctx, BOOL noclose) {
    if (ctx != NULL) {
	if (ctx->conn != NULL) {
	    if (noclose) {
		pstream_close_ex(ctx->conn, PSTREAM_NO_CLOSE); /* Just free everything but keep socket open */
	    } else {
		BOOL	to_free = (BOOL)(IS_REQ_CGI_MODE(ctx) || IS_CLIENT_MODE(ctx->app));
		BOOL	close_flag = (BOOL)((ctx->proto != HTTP_PROTO_V11) || (ctx->flags & HTTP_FLAG_CLOSE_CONNECTION) || to_free);

		pstream_close_ex(ctx->conn, close_flag ? PSTREAM_CLOSE : PSTREAM_NO_CLOSE);
		if (to_free) {
		    ZFREE(ctx->conn);
		    ctx->conn = NULL;
		}
	    }
	}
	if (ctx->plugin != NULL)
	    close_plugin(ctx->plugin);
	if (ctx->privdata != NULL)
	    ZFREE(ctx->privdata);
	if (ctx->query != NULL)
	    ZFREE(ctx->query);
	if (ctx->resp_content_type != NULL)
	    ZFREE(ctx->resp_content_type);
	if (ctx->filename != NULL)
	    ZFREE(ctx->filename);
	if (ctx->remoteuser != NULL)
	    ZFREE(ctx->remoteuser);
	if (ctx->ext_headers != NULL)
	    val_free(ctx->ext_headers);
	if (ctx->req_headers != NULL)
	    val_free(ctx->req_headers);
	if (ctx->resp_headers != NULL)
	    val_free(ctx->resp_headers);
	if (ctx->req_cookies != NULL)
	    val_free(ctx->req_cookies);
	__clean_resp_cookies(&ctx->resp_cookies);
	if (ctx->request != NULL)
	    val_free(ctx->request);
	if (ctx->body != NULL)
	    blob_free(ctx->body);
	if (ctx->uri != NULL)
	    uri_destroy(ctx->uri);
	if (noclose) {
	    ctx->plugin = NULL;
	    ctx->privdata = NULL;
	    ctx->query = NULL;
	    ctx->resp_content_type = NULL;
	    ctx->filename = NULL;
	    ctx->remoteuser = NULL;
	    ctx->ext_headers = NULL;
	    ctx->req_headers = NULL;
	    ctx->resp_headers = NULL;
	    ctx->req_cookies = NULL;
	    ctx->request = NULL;
	    ctx->body = NULL;
	    ctx->uri = NULL;
	} else {
	    ZFREE(ctx);
	}
    }
}

void http_parse_lang(HTTP_REQUEST * ctx) {
    const char	* lang = val_as_str_ptr(dict_find_node_near((IS_CLIENT_MODE(ctx->app) ? ctx->resp_headers : ctx->req_headers), __http_accepted_lang_str, TRUE));

    ctx->accepted_lang |= WEB_LANG_EN;
    do {
	char * part;
	
	lang = pat_tokenize(lang, ",", &part);
	if (part != NULL) {
	    const char	* ptr = part;

	    do {
		char		* l;
		
		ptr = pat_tokenize(ptr, "-", &l);
		if (l != NULL) {
		    if ( strcmp("ru", l) == 0)
			ctx->accepted_lang |= WEB_LANG_RU;
		    else if ( strcmp("es", l) == 0)
			ctx->accepted_lang |= WEB_LANG_ES;
		    ZFREE(l);
		}
	    } while (ptr != NULL);
	    ZFREE(part);
	}
    } while (lang != NULL);
}

PLEX const char * http_get_method_str(HTTP_REQUEST * ctx) {
    const char	* ptr = NULL;

    switch (ctx->method) {
	case HTTP_REQ_HEAD:
	    ptr = "HEAD";
	break;
	case HTTP_REQ_DELETE:
	    ptr = "DELETE";
	break;
	case HTTP_REQ_GET:
	    ptr = "GET";
	break;
	case HTTP_REQ_OPTIONS:
	    ptr = "OPTIONS";
	break;
	case HTTP_REQ_POST:
	    ptr = "POST";
	break;
	case HTTP_REQ_PUT:
	    ptr = "PUT";
	break;
	case HTTP_REQ_TRACE:
	    ptr = "TRACE";
	break;
	default:
	break;
    }
    return ptr;
}

HTTP_ERROR_CODES http_parse_protocol(HTTP_REQUEST * ctx, const char * ptr) {
	if (ptr != NULL) {
	    if (strcmp(ptr, "HTTP/1.0") == 0)
		ctx->proto = HTTP_PROTO_V10;
	    else if (strcmp(ptr, "HTTP/1.1") == 0)
		ctx->proto = HTTP_PROTO_V11;
	    else
		return HTTP_ERR_UNKNOWN_PROTOCOL;
	}
	if ((ctx->proto & ctx->app->proto_mask) != ctx->proto) {
	    if (IS_CLIENT_MODE(ctx->app))
		return HTTP_ERR_UNSUPPORTED_PROTOCOL;
	    else
		ctx->proto = HTTP_PROTO_V10;
	}
	return HTTP_ERR_NONE;
}

HTTP_REQUEST * http_create_request(WEB_CONTEXT * app, HTTP_ERROR_CODES * errn) {
    HTTP_REQUEST	* req = (HTTP_REQUEST *)ZALLOC(sizeof(*req));

    *errn = HTTP_ERR_OUT_OF_MEMORY;
    if (req != NULL) {
	if ((req->body = blob_new()) != NULL) {
	    req->app = app;
	    req->start_time = os_time_msec();
	    *errn = HTTP_ERR_NONE;
	    TAILQ_INIT(&req->resp_cookies);
	} else {
	    req = ZFREE(req);
	}
    }
    return req;
}

static inline ssize_t http_print_body_ap(HTTP_REQUEST * ctx, const char *fmt, va_list ap) {
	char	buffer[ASCIILINESZ];
	ssize_t	res;

	res = vsnprintf(buffer, sizeof(buffer), fmt, ap);
	if (res > 0)
	    blob_write(ctx->body, buffer, (size_t)res);
	return res;
}

PLEX ssize_t http_print_body(HTTP_REQUEST * ctx, const char * fmt,...) {
    ssize_t	res;
    va_list	args;

    va_start(args, fmt);
    res = http_print_body_ap(ctx, fmt, args);
    va_end(args);
    return res;
}

PLEX BOOL http_set_v(HTTP_REQUEST * ctx, val_t * val, WEB_APP_PLACES place) {
    BOOL	res = FALSE;

    if ((ctx != NULL) && (val != NULL)) {
	val_t		** ptr;
	const char	* name;

	switch (place) {
	    case WEB_APP_EXT_HEADER:
		name = "ext_headers";
		ptr = &ctx->ext_headers;
	    break;
	    case WEB_APP_REQUEST_HEADER:
		name = "headers";
		ptr = &ctx->req_headers;
	    break;
	    case WEB_APP_REQUEST_COOKIE:
		name = "cookies";
		ptr = &ctx->req_cookies;
	    break;
	    case WEB_APP_RESPONSE_HEADER:
		name = "headers";
		ptr = &ctx->resp_headers;
	    break;
	    default:
		return FALSE;
	    break;
	}
	if (*ptr == NULL) {
	    if ((*ptr = val_new(name)) == NULL) {
		log_err(http_error_strings[HTTP_ERR_OUT_OF_MEMORY]);
		return res;
	    }
	} else {
	    val_t	* e = http_find_val_in(ctx, val->key, place);

	    if (e != NULL)
		val_free(e);
	}
	res = val_dict_add(*ptr, val);
    }
    return res;
}

PLEX val_t * http_find_val_in(HTTP_REQUEST * ctx, const char * key, WEB_APP_PLACES place) {
    if (ctx != NULL) {
	val_t		* ptr = NULL;

	switch (place) {
	    case WEB_APP_EXT_HEADER:
		ptr = ctx->ext_headers;
	    break;
	    case WEB_APP_REQUEST_HEADER:
		ptr = ctx->req_headers;
	    break;
	    case WEB_APP_REQUEST_COOKIE:
		ptr = ctx->req_cookies;
	    break;
	    case WEB_APP_RESPONSE_HEADER:
		ptr = ctx->resp_headers;
	    break;
	    default:
	    break;
	}
	if (ptr != NULL)
	    return dict_find_node_near(ptr, key, TRUE);
    }
    return NULL;
}

PLEX const char * http_get_content_type(HTTP_REQUEST * ctx) {
    return val_as_str_ptr(dict_find_node_near((IS_CLIENT_MODE(ctx->app) ? ctx->resp_headers : ctx->req_headers), __http_content_type_str, TRUE));
}

PLEX size_t http_get_content_len(HTTP_REQUEST * ctx) {
    size_t	cl = 0;
    val_t	* val = dict_find_node_near((IS_CLIENT_MODE(ctx->app) ? ctx->resp_headers : ctx->req_headers), __http_content_len_str, TRUE);

    if (val != NULL) {
	int64_t	ival = -1;

	if (val_as_int(val, &ival) && (ival > 0))
	    cl = (size_t)ival;
    }
    return cl;
}

PLEX const char * http_get_remote_addr(HTTP_REQUEST * ctx) {
    return val_as_str_ptr(dict_find_node_near((IS_CLIENT_MODE(ctx->app) ? ctx->resp_headers : ctx->req_headers), __http_from_str, TRUE));
}

PLEX const char * http_get_user_agent(HTTP_REQUEST * ctx) {
    return val_as_str_ptr(dict_find_node_near((IS_CLIENT_MODE(ctx->app) ? ctx->resp_headers : ctx->req_headers), __http_ua_str, TRUE));
}

void http_debug_values(HTTP_REQUEST * ctx, WEB_APP_PLACES place) {
    if ((ctx != NULL) && IS_DEBUG_MODE(ctx->app) && !IS_REQ_CGI_MODE(ctx)) {
	val_t		* ptr = NULL;
	const char	* cptr = NULL;

	switch (place) {
	    case WEB_APP_REQUEST_HEADER:
		cptr = "Request Headers";
		ptr = ctx->req_headers;
	    break;
	    case WEB_APP_REQUEST_COOKIE:
		cptr = "Request Cookies";
		ptr = ctx->req_cookies;
	    break;
	    case WEB_APP_RESPONSE_HEADER:
		cptr = "Response Headers";
		ptr = ctx->resp_headers;
	    break;
	    case WEB_APP_RESPONSE_COOKIE:
		do {
		    char	buf[ASCIILINESZ];
		    size_t	nbytes = http_print_server_cookies(ctx, buf, sizeof(buf));

		    if (nbytes > 0)
			write(STDERR_FILENO, buf, nbytes);
		} while (0);
	    break;
	    default:
		cptr = "Request";
		ptr = ctx->request;
	    break;
	}
	if (ptr != NULL) {
	    fprintf(stderr, "\n==> %s <==\n", cptr);
	    blob_t * blob = val_2_blob(ptr, NULL, VAL_C_TYPE_JSON, 0);
	    if (blob != NULL) {
		blob_rewind(blob);
		write_blob_to_sock(blob, STDERR_FILENO);
		blob_free(blob);
	    }
	    fprintf(stderr, "\n\n");
	}
    }
}

PLEX void web_cleanup_ctx(WEB_CONTEXT * ctx) {

    if (ctx->http != NULL) {
	http_free_request(ctx->http);
	ctx->http = NULL;
    }
    if (ctx->url != NULL)
	ctx->url = ZFREE(ctx->url);
    ssl_clear_app_ctx(&ctx->app_ctx);
}

WEB_CONTEXT * web_init_ctx_default(void) {
    WEB_CONTEXT * web = NULL;

    if ((web = (WEB_CONTEXT *)ZALLOC(sizeof(WEB_CONTEXT))) != NULL) {
	web->method_mask = HTTP_REQ_GET | HTTP_REQ_POST;
	web->proto_mask = HTTP_PROTO_V10 | HTTP_PROTO_V11;
	web->lang = WEB_LANG_EN;
	web->mode = WEB_APP_ALLOW_SSL;
	web->connection_timeout = 30;
    }
    return web;
}

WEB_CONTEXT * web_alloc_ctx(const char * url, WEB_APP_MODE mode, const char * app, const char * cert, int connection_timeout, const char * ciphers) {
    WEB_CONTEXT	* res = NULL;
    char	* ptr;

    if (SEMPTY(url)) {
	log_err("Empty url");
	return res;
    }
    if ((ptr = ZSTRDUP(url)) == NULL) {
	log_err("Not enough memory");
	return res;
    }
    if ((res = web_init_ctx_default()) == NULL) {
	ZFREE(ptr);
	log_err("Could not allocate WEB_CONTEXT structure");
	return res;
    }
    res->url = ptr;
    res->mode |= mode;
    res->appname = app;
    if (connection_timeout > 0)
	res->connection_timeout = connection_timeout * 1000;
    if (!IS_CLIENT_MODE(res) && SEMPTY(cert))
	res->mode &= ~WEB_APP_ALLOW_SSL;
    if (IS_ALLOW_SSL(res)) {
	if (!ssl_init_app_ctx(&res->app_ctx, cert, NULL, ciphers, IS_CLIENT_MODE(res), IS_VERIFY_PEER(res)))
	    res->mode &= ~WEB_APP_ALLOW_SSL;
    }
    return res;
}

void * web_free_ctx(WEB_CONTEXT * ctx) {
    if (ctx != NULL) {
	if (ctx->url != NULL)
	    ZFREE(ctx->url);
	ssl_clear_app_ctx(&ctx->app_ctx);
	ZFREE(ctx);
    }
    return NULL;
}

static const char	__http_def_charset[] = "UTF-8";

PLEX const char * web_get_charset(WEB_CONTEXT * ctx) {
    if ((ctx != NULL) && (!SEMPTY(ctx->charset)))
	return ctx->charset;
    return __http_def_charset;
}

PLEX BOOL http_set_something(HTTP_REQUEST * ctx, WEB_APP_PLACES place, const char * key, const char * fmt, ...) {

    if (ctx != NULL) {
	va_list	args;
	val_t	* val;

	va_start(args, fmt);
	val = val_from_fmt_ap(key, fmt, args);
	va_end(args);
	if (val == NULL)
	    return FALSE;
	return http_set_v(ctx, val, place);
    }
    return FALSE;
}

PLEX ssize_t http_pack_urlencoded(HTTP_REQUEST * ctx, blob_t * blob, char * buf, size_t remain, BOOL need_amp, BOOL first_q) {
	int	len;
	ssize_t	nwritten = 0, written;
	char	* ptr = buf;

	if (blob != NULL) {
	    blob_rewind(blob);
	    blob_len(blob) = 0;
	}

	FOREACH_IN_DICT(ctx->request, len) {
	    char	* value;
	    val_t	* val = V_CHILD(ctx->request, len);

	    if (V_IS_ARR(val) || V_IS_OBJ(val) || V_IS_FILE(val) || V_IS_BLOB(val))
		continue;
	    if (need_amp) {
		if (blob != NULL)
		    blob_write(blob, "&", sizeof(char));
		else {
		    written = snprintf(ptr, remain, "&");
		    ptr += written;
		    nwritten += written;
		    remain -= written;
		}
	    } else if (first_q) {
		if (blob != NULL)
		    blob_write(blob, "?", sizeof(char));
		else {
		    written = snprintf(ptr, remain, "?");
		    ptr += written;
		    nwritten += written;
		    remain -= written;
		}
	    }
	    need_amp = TRUE;
	    if (blob != NULL)
		blob_printf(blob, "%s=", val_name(val));
	    else {
		written = snprintf(ptr, remain, "%s=", val_name(val));
		ptr += written;
		nwritten += written;
		remain -= written;
	    }
	    if ((value = val_as_str_simple(val)) != NULL) {
		char	encoded[1024], * tptr;

		if ((tptr = url_encode(encoded, sizeof(encoded), value, strlen(value))) != NULL) {
		    if (blob != NULL)
			blob_printf(blob, "%s", tptr);
		    else {
			written = snprintf(ptr, remain, "%s", tptr);
			ptr += written;
			nwritten += written;
			remain -= written;
		    }
		}
		ZFREE(value);
	    }
	}
	return nwritten;
}

static BOOL http_pack_multipart_values(HTTP_REQUEST * ctx, val_t * node, uint64_t bnd) {
    int	i;

    FOREACH_IN_DICT(node, i) {
	val_t	* val = V_CHILD(node, i);

	http_printf(ctx, "\r\n--PCoreHTTPBnd%" PRIu64 "\r\n%s: form-data; name=\"", bnd, __http_content_disp_str);
	if (V_IS_BLOB(val)) {
	    http_printf(ctx, "%s\"\r\n%s: %s\r\n\r\n", val->key, __http_content_type_str, 
		__http_built_in_content_types[HTTP_TYPE_APPLICATION_BINARY]);
	    if (pstream_write_blob(ctx->conn, val->val.b) != (ssize_t)blob_left(val->val.b))
		return FALSE;
	} else if (V_IS_ARR(val)) {
	    if (!http_pack_multipart_values(ctx, val, bnd))
		return FALSE;
	} else if (V_IS_OBJ(val)) {
	    blob_t	* blob = val_2_blob(val, NULL, VAL_C_TYPE_JSON, -1);

	    if (blob != NULL) {
		BOOL	bdone;

		http_printf(ctx, "%s\"\r\n%s: %s\r\n\r\n", val->key, __http_content_type_str, 
		    __http_built_in_content_types[HTTP_TYPE_APPLICATION_JSON]);
		bdone = (BOOL)(pstream_write_blob(ctx->conn, blob) == (ssize_t)blob_left(blob));
		blob_free(blob);
		if (!bdone)
		    return FALSE;
	    }
	} else if (V_IS_FILE(val)) {
	    http_printf(ctx, "%s\"; filename=\"%s\"\r\n\r\n", val->key, 
		val->val.file.filename);
	    http_write_file(ctx, val->val.file.filename, TRUE);
	} else {
	    char	* value = val_as_str_simple(val);

	    if (value != NULL) {
		http_printf(ctx, "%s\"\r\n\r\n%s", val->key, value);
		ZFREE(value);
	    }
	}
    }
    return TRUE;
}

PLEX BOOL http_pack_post_request(HTTP_REQUEST * ctx, BOOL multipart) {
    BOOL	res = FALSE;

    if (!multipart) {
	int	len;

	FOREACH_IN_DICT(ctx->request, len) {
	    val_t	* val = V_CHILD(ctx->request, len);

	    if (V_IS_ARR(val) || V_IS_OBJ(val) || V_IS_FILE(val) || V_IS_BLOB(val) || 
		(V_IS_STRING(val) && (val->i > 64))) {
		multipart = TRUE;
		break;
	    }
	}
    }
    if (multipart) {
	uint64_t	bnd = os_random64();

	http_printf(ctx, "%s: %s; %s\"PCoreHTTPBnd%" PRIu64  "\"\r\n", 
		__http_content_type_str, __http_built_in_content_types[HTTP_TYPE_MULTIPART_FORM_DATA], 
		__http_bnd_str, bnd);
	if (!http_pack_multipart_values(ctx, ctx->request, bnd))
	    return res;
	http_printf(ctx, "\r\n--PCoreHTTPBnd%" PRIu64 "--\r\n", bnd);
	res = TRUE;
    } else {
	int	len;

	http_pack_urlencoded(ctx, ctx->body, NULL, 0, FALSE, FALSE);
	blob_rewind(ctx->body);
	len = blob_len(ctx->body);
	http_printf(ctx, "%s: %s\r\n%s: %lu\r\n\r\n", 
		__http_content_type_str, __http_built_in_content_types[HTTP_TYPE_APPLICATION_X_WWW_FORM_URLENCODED], 
		__http_content_len_str, len);
	res = (BOOL)(http_write(ctx, (const void *)blob_data(ctx->body), len) == (ssize_t)len);
    }
    return res;
}
