#include <plib/core.h>
#include "remotectrl.h"

int rmtctrl_write_msg( struct rmt_socket_t *sckHnd, uint32_t id, uint32_t extra, const char *message ){
	msg_head_t header;
	int rslt;
	
	header.id    = id;
	header.len = 0;
	if ( message != NULL)
	    header.len = strlen(message);
	header.extra = extra;
	rslt = send(sckHnd->fd,&header,sizeof(msg_head_t),0);
	if (rslt != -1 && header.len > 0) {
		sckHnd->Tx += rslt;
		rslt = send(sckHnd->fd, message, header.len, 0);
		if (rslt >= 0)
		{
			sckHnd->Tx += rslt;
			rslt += sizeof(struct msg_head_t);
		}
	}
	return rslt;
}

int rmtctrl_read_msg( struct rmt_socket_t *sckHnd, msg_head_t *head, char **message ) {
	ssize_t rslt;
	char *buffer;
	ssize_t reading = 0;
	
	rslt = sock_read(sckHnd->fd, head, sizeof(struct msg_head_t), 1000);
	if (rslt == (ssize_t)sizeof(struct msg_head_t)) {
		sckHnd->Rx += rslt;
		if ( head->len > 0) {
		    *message = (char *)malloc(head->len+1);
		    if ( *message != NULL) {
			memset(*message,'\0', head->len+1);
			reading = sock_read(sckHnd->fd, *message, head->len, 1000);
			if ( reading > 0) {
			    sckHnd->Rx += reading;
			    reading += rslt;
			}
			return reading;
		    } else {
			log_debug("Could not allocate memory for message");
			return -1;
		    }
		}
	}
	return rslt;
}

void rmtctrl_newClient(struct rmt_socket_t srv, struct rmt_socket_t *client, int *activeClients) {
	int rslt;
	int cli = (*activeClients);

	rmtctrl_accept(srv,&client[cli]);
	if (client[(*activeClients)].fd != -1) {
		(*activeClients)++;
	}
	if ((*activeClients) >= MAX_CLIENTS) {
		(*activeClients)--;
		rslt = rmtctrl_write_msg(&client[(*activeClients)],MSG_END,0, "Sorry Server is too Busy\n	Try more late\n" );
		if (rslt > 0) client[(*activeClients)].Tx += rslt;
		rmtctrl_close(&client[(*activeClients)]);
	}
}

void rmtctrl_close ( struct rmt_socket_t *client ) {
	close(client->fd); /* cierra fd_rmt_client */
	client->fd = -1;
}

void rmtctrl_accept (struct rmt_socket_t srv, struct rmt_socket_t *client ) {
	int sin_size=sizeof(struct sockaddr_in);
	int int_Send;
	struct sockaddr_in addr;
	
	if ((client->fd = accept(srv.fd,(struct sockaddr *)&client->addr,&sin_size))!=-1)  {
		client->Rx = 0;
		client->Tx = 0;
		unsigned char c = sizeof(uint32_t);
		int_Send = send(client->fd, &c, 1, 0);
		if (int_Send > 0) client->Tx += int_Send;
	}
}

//void cleanClients (int *table, int *n)
void rmtctrl_cleanClients (struct rmt_socket_t *client, int *n) {
	int i,j;

	if ((client == NULL) || ((*n) == 0))
		return;

	j=0;
	for (i=0; i<(*n); i++) {
		if (client[i].fd != -1)
		{
			client[j].fd = client[i].fd;
			client[j].addr = client[i].addr;
			client[j].Rx = client[i].Rx;
			client[j].Tx = client[i].Tx;
			j++;
		}
	}
	
	*n = j;
}

int rmtctrl_maxValue (struct rmt_socket_t *client, int n) {
	int i;
	int max;

	if ((client == NULL) || (n<1))
		return 0;
		
	max = client[0].fd;
	for (i=0; i<n; i++)
		if (client[i].fd > max)
			max = client[i].fd;

	return max;
}

struct rmt_socket_t rmtctrl_initSrv(struct in_addr rmtlisten, int rmtport){
	struct rmt_socket_t srv;
	if ((srv.fd=socket(AF_INET, SOCK_STREAM, 0)) == -1 ) {  
		log_debug("error en socket()\n");
		exit(1);
	}
	srv.addr.sin_family = AF_INET;
	srv.addr.sin_port = htons(rmtport); 
	srv.addr.sin_addr.s_addr = rmtlisten.s_addr;
	memset(&(srv.addr.sin_zero), 0, 8); 

	if(bind(srv.fd,(struct sockaddr*)&srv.addr,sizeof(struct sockaddr))==-1) {
		log_debug("error en bind() \n");
		exit(-1);
	}     

	if(listen(srv.fd,BACKLOG) == -1) {
		log_debug("error en listen()\n");
		exit(-1);
	}
	return srv;
}

int send_line( struct rmt_socket_t *client, int msg_type, int msg_extra, const char *fmt, const char *data) {
	char str[2048];

	memset(str,'\0', sizeof(str));
	snprintf(str, sizeof(str),fmt,data);
	return rmtctrl_write_msg(client,msg_type,msg_extra, str);
}

char * octets2str(uint64_t value)
{
	char *buffer;

	buffer=(char*)malloc(13*sizeof(char));
	memset(buffer,'\0',13);
	if (value/8 > 1073741824){ // gigas
		sprintf(buffer,"%.1f(GiB)",((value/8.0)/1073741824.0));
	}else if (value/8 > 1048576){ // megas
		sprintf(buffer,"%.1f(MiB)",((value/8.0)/1048576.0));
	}else if (value/8 > 1024){ // KiloBytes
		sprintf(buffer,"%.1f(KiB)",((value/8.0)/1024.0));
	}else // Bytes
		sprintf(buffer,"%.1f(B)",(value/8.0));
	return buffer;
}

int send_octets( struct rmt_socket_t *client, int msg_type, int msg_extra, const char *fmt, uint64_t value) {
	char *Buffer = octets2str(value);

	int ret = send_line( client, msg_type, msg_extra, fmt, Buffer);
	free(Buffer);
	return ret;
}

char * mac2str(uint8_t *mac) {
	char *buffer;

	buffer=(char*)malloc(18*sizeof(char));
	memset(buffer,'\0',18);
	(void) sprintf(buffer,"%.2X-%.2X-%.2X-%.2X-%.2X-%.2X",
		    mac[0], mac[1],
		    mac[2], mac[3],
		    mac[4], mac[5]);
	return buffer;
}

int send_mac( struct rmt_socket_t *client, int msg_type, int msg_extra, const char *fmt, uint8_t *value) {
	char *Buffer = mac2str(value);

	int ret = send_line( client, msg_type, msg_extra, fmt, Buffer);
	free(Buffer);
	return ret;
}

int send_number( struct rmt_socket_t *client, int msg_type, int msg_extra, const char *fmt, uint64_t value) {
	char Buffer[128];

	snprintf(Buffer,sizeof(Buffer),"%llu", (unsigned long long)value);
	return send_line( client, msg_type, msg_extra, fmt, Buffer);
}

BOOL rmtctrl_open_client( struct rmt_socket_t * srv, short port) {
    BOOL	res = FALSE;
    
    if ( srv == NULL)
	return res;
	
    memset( srv, 0, sizeof(struct rmt_socket_t));
    if ((srv->fd = socket( AF_INET, SOCK_STREAM, 0)) > 0) {
	srv->addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	srv->addr.sin_port = htons(port);
	srv->addr.sin_family = AF_INET;
	if ( connect( srv->fd, (struct sockaddr *)&srv->addr, sizeof(struct sockaddr_in)) == 0)
	    res = TRUE;
	else {
	    close( srv->fd);
	    log_debug("Could not connect to server at port - %d", port);
	}
    } else
	log_debug("Could not create socket");
    return res;
}

static blob_t * read_msg_2_blob(struct rmt_socket_t *srv) {
    msg_head_t	head;
    char	* message = NULL;
    int		n;
    BOOL	end_msg = FALSE;
    blob_t	* b = NULL;

    if ( srv == NULL)
	return b;
    while((n =  rmtctrl_read_msg(srv, &head, &message)) > 0) {
	switch (head.id) {
	    case MSG_END:
		end_msg = TRUE;
	    case MSG_START:
	    case MSG_PART:
		if ( message != NULL) {
		    if ( b == NULL)
			b = blob_new();
		    if ( b != NULL)
			blob_write( b, message, strlen(message));
		    free( message);
		    message = NULL;
		}
	}
	if ( end_msg)
	    break;
    }
    if ( message != NULL)
	free( message);
    return b;
}

blob_t * get_user_list(struct rmt_socket_t *srv) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, QRY_CONNECTED_LIST_EX, 0, NULL) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * get_user_stat_by_mac(struct rmt_socket_t *srv, const char * mac) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, QRY_STATUS_EX, EXTRA_MAC_OP, mac) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * get_user_stat_by_ip(struct rmt_socket_t *srv, const char * ip) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, QRY_STATUS_EX, EXTRA_IP_OP, ip) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * disconnect_user_by_mac(struct rmt_socket_t *srv, const char * mac) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_DISCONNECT, EXTRA_MAC_OP, mac) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * disconnect_user_by_ip(struct rmt_socket_t *srv, const char * ip) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_DISCONNECT, EXTRA_IP_OP, ip) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * disconnect_user_by_uname(struct rmt_socket_t *srv, const char * uname) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_DISCONNECT, EXTRA_USER_OP, uname) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * disconnect_user_by_transid(struct rmt_socket_t *srv, const char * transid) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_DISCONNECT, EXTRA_TRANS_OP, transid) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * disconnect_all(struct rmt_socket_t *srv) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_DISCONNECT, 0, "") > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * auth_user_by_mac(struct rmt_socket_t *srv, const char * mac) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_AUTHORIZE, EXTRA_MAC_OP, mac) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * auth_user_by_ip(struct rmt_socket_t *srv, const char * ip) {
    blob_t	* b = NULL;
    
    if ( srv != NULL) {
	if ( rmtctrl_write_msg( srv, CMD_AUTHORIZE, EXTRA_IP_OP, ip) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * set_int_info_by_uname(struct rmt_socket_t *srv, const char * uname, int64_t i) {
    blob_t	* b = NULL;
    char	buf[512];
    
    if ( srv != NULL) {
	snprintf(buf, sizeof(buf), "%s#%lld", uname, (long long int)i);
	if ( rmtctrl_write_msg( srv, CMD_SET_INFO, EXTRA_USER_OP, buf) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}

blob_t * set_int_info_by_transid(struct rmt_socket_t *srv, const char * transid, int64_t i) {
    blob_t	* b = NULL;
    char	buf[512];
    
    if ( srv != NULL) {
	snprintf(buf, sizeof(buf), "%s#%lld", transid, (long long int)i);
	if ( rmtctrl_write_msg( srv, CMD_SET_INFO, EXTRA_TRANS_OP, buf) > 0)
	    b = read_msg_2_blob( srv);
    }
    return b;
}
