#ifndef HTTP_H
#define HTTP_H

#include <plib/uri.h>
#include <plib/crypt.h>
#include <plib/basenet.h>
#include <plib/core/hashmap.h>

typedef enum {
  HTTP_REQ_UNKNOW,
  HTTP_REQ_CONNECT = 0x1,
  HTTP_REQ_DELETE = 0x2,
  HTTP_REQ_HEAD = 0x4,
  HTTP_REQ_GET = 0x8,
  HTTP_REQ_OPTIONS = 0x10,
  HTTP_REQ_PATCH = 0x20,
  HTTP_REQ_POST = 0x40,
  HTTP_REQ_PUT = 0x80,
  HTTP_REQ_TRACE = 0x100,
  HTTP_REQ_MAX = HTTP_REQ_TRACE,
} HTTP_REQUEST_TYPE;

typedef enum {
  HTTP_PROTO_V09 = 0,
  HTTP_PROTO_V10 = 0x1,
  HTTP_PROTO_V11 = 0x2,
} HTTP_PROTO_TYPE;

#define	HTTP_PROTO_09_DEFAULT_METHOD_MASK	(HTTP_REQ_GET)
#define	HTTP_PROTO_10_DEFAULT_METHOD_MASK	(HTTP_PROTO_09_DEFAULT_METHOD_MASK|HTTP_REQ_POST)
#define	HTTP_PROTO_11_DEFAULT_METHOD_MASK	(HTTP_PROTO_10_DEFAULT_METHOD_MASK|HTTP_REQ_HEAD|HTTP_REQ_OPTIONS|HTTP_REQ_TRACE)

typedef enum {
  HTTP_TYPE_APPLICATION_OCTET_STREAM,
  HTTP_TYPE_TEXT_PLAIN,
  HTTP_TYPE_TEXT_HTML,
  HTTP_TYPE_MULTIPART_FORM_DATA,
  HTTP_TYPE_MULTIPART_MIXED,
  HTTP_TYPE_APPLICATION_X_WWW_FORM_URLENCODED,
  HTTP_TYPE_APPLICATION_BINARY,
  HTTP_TYPE_APPLICATION_JSON,
} HTTP_CONTENT_TYPE;

typedef enum {
  HTTP_TRANS_ENC_UNKNOW = 0,
  HTTP_TRANS_ENC_CHUNKED = 0x1,
  HTTP_TRANS_ENC_GZIP = 0x2,
} HTTP_TRANSFER_ENCODING;

/* Possible error states: */
typedef enum {
  HTTP_ERR_NONE,
  HTTP_ERR_BAD_PACKET,
  HTTP_ERR_UNKNOWN_METHOD,
  HTTP_ERR_UNSUPPORTED_METHOD,
  HTTP_ERR_UNKNOWN_PROTOCOL,
  HTTP_ERR_UNSUPPORTED_PROTOCOL,
  HTTP_ERR_INCORRECT_TYPE,
  HTTP_ERR_NULL_QUERY_STRING,
  HTTP_ERR_BAD_CONTENT_LENGTH,
  HTTP_ERR_OUT_OF_MEMORY,
  HTTP_ERR_NO_BOUNDARY,
  HTTP_ERR_CANT_OPEN,
  HTTP_ERR_CANT_READ,
  HTTP_ERR_BAD_REQUEST,
  HTTP_ERR_BAD_RESPONSE,
  HTTP_ERR_NO_COOKIES,
  HTTP_ERR_NOT_AUTHORIZED,
  HTTP_ERR_NOT_FOUND,
  HTTP_ERR_NO_CONTENT,
  HTTP_ERR_REDIRECT,
  HTTP_ERR_INTERNAL,
  HTTP_ERR_FORBIDDEN,
  HTTP_ERR_NOT_FULL_HEADER,
  HTTP_ERR_NUM_ERRS,
} HTTP_ERROR_CODES;

typedef enum {
    HTTP_SSL_FLAG = 0x1,
    HTTP_USE_RESP_FROM_FILE_FLAG = 0x2,
    HTTP_USE_REQ_FROM_FILE_FLAG = 0x4,
    HTTP_EXEC_CGI = 0x8,
    HTTP_FLAG_AUTH_BASIC = 0x10,
    HTTP_FLAG_AUTH_DIGEST = 0x20,
    HTTP_FLAG_CLOSE_CONNECTION = 0x100,
    HTTP_FLAG_NUM_FLAGS,
} HTTP_FLAGS;


typedef enum {
    WEB_LANG_EN = 0x1,
    WEB_LANG_RU = 0x2,
    WEB_LANG_ES = 0x4,
    WEB_LANG_FR = 0x8,
    WEB_LANG_ZH = 0x10
} WEB_LANG;

#define	HTTP_MAX_QUERY_SIZE	4096
#define	HTTP_MAX_POST_SIZE	ULONG_MAX

struct web_context;

typedef struct http_server_cookie {
    val_t	* value;
    time_t	expires;
    BOOL	httponly;
    char	* domain;
    TAILQ_ENTRY(http_server_cookie)	entry;
} HTTP_SERVER_COOKIE;

TAILQ_HEAD(http_server_cookies, http_server_cookie);

typedef struct http_request {
    pstream_t		* conn;
    HTTP_FLAGS		flags;
    HTTP_PROTO_TYPE	proto;
    HTTP_REQUEST_TYPE	method;
    HTTP_CONTENT_TYPE	content_type;
    HTTP_TRANSFER_ENCODING	encoding;
    WEB_LANG		accepted_lang;
    size_t		cl;
    int			retcode;
    struct http_server_cookies	resp_cookies;
    val_t		* req_headers;
    val_t		* resp_headers;
    val_t		* req_cookies;
    val_t		* ext_headers;
    val_t		* request;
    char		* query;
    char		* remoteuser;
    char		* resp_content_type;
    char		* filename;
    const char		* validator;
    blob_t		* body;
    struct uri_t	* uri;
    uint64_t		start_time;
    uint64_t		elapsed;
    struct web_context	* app;
    const char		* replymsg;
    void		* data;
    plugin_t		* plugin;
    void		* privdata;
    ssize_t		(*form_headers)(struct http_request *, char *, size_t);
} HTTP_REQUEST;

#define	IS_REQ_CGI_MODE(c)	(((c)->conn) && ((c)->conn->fd == STDIN_FILENO))
#define	IS_REQ_SSL_MODE(c)	(((c)->flags & HTTP_SSL_FLAG) == HTTP_SSL_FLAG)

typedef enum {
    WEB_APP_CGI = 0x0,
    WEB_APP_SERVER = 0x1,
    WEB_APP_CLIENT = 0x2,
    WEB_APP_DEBUG = 0x4,
    WEB_APP_ALLOW_SSL = 0x8,
    WEB_APP_AUTH_BASIC = 0x10,
    WEB_APP_AUTH_DIGEST = 0x20,
    WEB_APP_AUTH_REQUIRED_BASIC = (WEB_APP_AUTH_BASIC | WEB_APP_SERVER),
    WEB_APP_AUTH_REQUIRED_DIGEST = (WEB_APP_AUTH_DIGEST | WEB_APP_SERVER),
    WEB_APP_VERIFY_PEER = (0x100 | WEB_APP_ALLOW_SSL | WEB_APP_CLIENT),
    WEB_APP_ALLOW_VIRTUAL = (0x200 | WEB_APP_SERVER),
} WEB_APP_MODE;

#define IS_CLIENT_MODE(c)	(((c)->mode & WEB_APP_CLIENT) == WEB_APP_CLIENT)
#define IS_SERVER_MODE(c)	(((c)->mode & WEB_APP_SERVER) == WEB_APP_SERVER)
#define IS_DEBUG_MODE(c)	((((c)->mode & WEB_APP_DEBUG) == WEB_APP_DEBUG) || (c->argc > 1))
#define IS_ALLOW_SSL(c)		(((c)->mode & WEB_APP_ALLOW_SSL) == WEB_APP_ALLOW_SSL)
#define IS_SSL_MODE(c)		(IS_ALLOW_SSL(c) && ((c)->conn) && ((c)->conn->ops != NULL))
#define IS_VERIFY_PEER(c)	(((c)->mode & WEB_APP_VERIFY_PEER) == WEB_APP_VERIFY_PEER)
#define IS_ALLOW_VIRTUAL(c)	(((c)->mode & WEB_APP_ALLOW_VIRTUAL) == WEB_APP_ALLOW_VIRTUAL)
#define IS_NOT_VALIDATED(c)	((c)->validated_level <= 0)
#define IS_BASIC_AUTH_REQUIRED(c)	(((c)->mode & WEB_APP_AUTH_REQUIRED_BASIC) == WEB_APP_AUTH_REQUIRED_BASIC)
#define IS_DIGEST_AUTH_REQUIRED(c)	(((c)->mode & WEB_APP_AUTH_REQUIRED_DIGEST) == WEB_APP_AUTH_REQUIRED_DIGEST)

typedef struct web_context {
    WEB_APP_MODE	mode;
    HTTP_REQUEST_TYPE	method_mask;
    HTTP_PROTO_TYPE	proto_mask;
    HTTP_TRANSFER_ENCODING	encoding_mask;
    HTTP_REQUEST	* http;
    WEB_LANG		lang;
    char		* url;
    const char		* appname;
    const char		* charset;
    const char		* validator;
    const char		* not_validated_view;
    const char		* validated_view;
    int			validated_level;
    int			connection_timeout;
    int			retcode;
    int			argc;
    const char		** argv;
    void		* localcfg;
    void		* privdata;
    pstream_t		* stream;
    SSL_APP_CTX		app_ctx;
} WEB_CONTEXT;

typedef enum {
    WEB_APP_REQUEST_HEADER,
    WEB_APP_REQUEST_COOKIE,
    WEB_APP_REQUEST_BODY,
    WEB_APP_RESPONSE_HEADER,
    WEB_APP_RESPONSE_COOKIE,
    WEB_APP_EXT_HEADER,
    WEB_APP_RESPONSE_BODY = WEB_APP_REQUEST_BODY,
    WEB_APP_BODY = WEB_APP_REQUEST_BODY,
}	WEB_APP_PLACES;

__BEGIN_DECLS

extern const char	* __http_built_in_content_types[];
extern const char	__http_host_str[];
extern const char	__http_lf[];
extern const char	__http_content_len_str[];
extern const char	__http_hdr_end[];
extern const char	__http_head_end_fmt[];
extern const char	__http_content_type_str[];
extern const char	__http_authen_str[];
extern const char	__http_author_str[];

static inline ssize_t http_read(HTTP_REQUEST * ctx, void * buf, size_t count) { return pstream_read(ctx->conn, buf, count); }

PLEX void http_free_request_ex(HTTP_REQUEST *, BOOL);
#define http_free_request(a)	http_free_request_ex(a, FALSE)

PLEX ssize_t http_printf(HTTP_REQUEST *, const char *, ...);
PLEX BOOL http_write_file(HTTP_REQUEST *, const char *, BOOL);

PLEX ssize_t http_print_body(HTTP_REQUEST *, const char *, ...);

PLEX BOOL http_set_v(HTTP_REQUEST *, val_t *, WEB_APP_PLACES);
#define	http_set_ext_header(c, v)	http_set_v(c, v, WEB_APP_EXT_HEADER)
#define	http_set_req_header(c, v)	http_set_v(c, v, WEB_APP_REQUEST_HEADER)
#define	http_set_req_cookie(c, v)	http_set_v(c, v, WEB_APP_REQUEST_COOKIE)
#define	http_set_resp_header(c, v)	http_set_v(c, v, WEB_APP_RESPONSE_HEADER)
#define	http_set_resp_cookie(c, v)	http_set_v(c, v, WEB_APP_RESPONSE_COOKIE)
PLEX val_t * http_find_val_in(HTTP_REQUEST *, const char *, WEB_APP_PLACES);
#define	http_find_val_in_req_header(c, n)	http_find_val_in(c, n, WEB_APP_REQUEST_HEADER)
#define	http_find_val_in_req_cookie(c, n)	http_find_val_in(c, n, WEB_APP_REQUEST_COOKIE)
#define	http_find_val_in_resp_header(c, n)	http_find_val_in(c, n, WEB_APP_RESPONSE_HEADER)
#define	http_find_val_in_resp_cookie(c, n)	http_find_val_in(c, n, WEB_APP_RESPONSE_COOKIE)

PLEX BOOL http_set_something(HTTP_REQUEST *, WEB_APP_PLACES, const char *, const char *, ...);

#define http_set_srv_cookie(h, n, fmt, ...)	http_set_something(h, WEB_APP_RESPONSE_COOKIE, n, fmt, __VA_ARGS__)
#define http_set_srv_header(h, n, fmt, ...)	http_set_something(h, WEB_APP_RESPONSE_HEADER, n, fmt, __VA_ARGS__)

PLEX const char * http_get_user_agent(HTTP_REQUEST *);
PLEX const char * http_get_remote_addr(HTTP_REQUEST *);
PLEX const char * http_get_content_type(HTTP_REQUEST *);
PLEX size_t http_get_content_len(HTTP_REQUEST *);
PLEX BOOL http_start_output(HTTP_REQUEST *, const char *, size_t);
PLEX ssize_t http_pack_urlencoded(HTTP_REQUEST *, blob_t *, char *, size_t, BOOL, BOOL);
PLEX BOOL http_pack_post_request(HTTP_REQUEST *, BOOL);
PLEX ssize_t http_write(HTTP_REQUEST *, const void *, size_t);
PLEX const char * http_get_method_str(HTTP_REQUEST *);

PLEX const char * web_get_charset(WEB_CONTEXT *);
PLEX void web_cleanup_ctx(WEB_CONTEXT *);

__END_DECLS

#endif
