static int
smtp_tls_init(struct smtp *const smtp,
              const char *const server){
  X509 *X509_cert_peer;

  /* Do not need to check the return value since this always returns 1. */
  SSL_library_init();

  SSL_load_error_strings();
  ERR_load_BIO_strings();
  OpenSSL_add_all_algorithms();

  if((smtp->tls_ctx = SSL_CTX_new(SSLv23_client_method())) == NULL){
    return -1;
  }

  /* Disable SSLv2, SSLv3, and TLSv1.0. */
  SSL_CTX_set_options(smtp->tls_ctx,
                      SSL_OP_NO_SSLv2 |
                      SSL_OP_NO_SSLv3 |
                      SSL_OP_NO_TLSv1);

  SSL_CTX_set_mode(smtp->tls_ctx, SSL_MODE_AUTO_RETRY);
  if((smtp->flags & SMTP_NO_CERT_VERIFY) == 0){
    SSL_CTX_set_verify(smtp->tls_ctx, SSL_VERIFY_PEER, NULL);
  }

  /*
   * Set the path to the user-provided CA file or use the default cert paths
   * if not provided.
   */
  if(smtp->cafile){
    if(SSL_CTX_load_verify_locations(smtp->tls_ctx, smtp->cafile, NULL) != 1){
      SSL_CTX_free(smtp->tls_ctx);
      return -1;
    }
  }
  else{
    X509_STORE_set_default_paths(SSL_CTX_get_cert_store(smtp->tls_ctx));
    if(ERR_peek_error() != 0){
      SSL_CTX_free(smtp->tls_ctx);
      return -1;
    }
  }

  if((smtp->tls = SSL_new(smtp->tls_ctx)) == NULL){
    SSL_CTX_free(smtp->tls_ctx);
    return -1;
  }

  if((smtp->tls_bio = BIO_new_socket(smtp->sock, 0)) == NULL){
    SSL_CTX_free(smtp->tls_ctx);
    SSL_free(smtp->tls);
    return -1;
  }

  SSL_set_bio(smtp->tls, smtp->tls_bio, smtp->tls_bio);
  SSL_set_connect_state(smtp->tls);
  if(SSL_connect(smtp->tls) != 1){
    SSL_CTX_free(smtp->tls_ctx);
    SSL_free(smtp->tls);
    return -1;
  }

  if(SSL_do_handshake(smtp->tls) != 1){
    SSL_CTX_free(smtp->tls_ctx);
    SSL_free(smtp->tls);
    return -1;
  }

  /* Verify matching subject in certificate. */
  if((smtp->flags & SMTP_NO_CERT_VERIFY) == 0){
    if((X509_cert_peer = SSL_get_peer_certificate(smtp->tls)) == NULL){
      SSL_CTX_free(smtp->tls_ctx);
      SSL_free(smtp->tls);
      return -1;
    }
    if(X509_check_host(X509_cert_peer, server, 0, 0, NULL) != 1){
      SSL_CTX_free(smtp->tls_ctx);
      SSL_free(smtp->tls);
      return -1;
    }
    X509_free(X509_cert_peer);
  }

  smtp->tls_on = 1;
  return 0;
}
