#include <plib/fs.h>
#include <plib/crypt.h>
#include "internal.h"

/**
 * Validate characters in the attachment file name.
 *
 * Must consist only of printable characters or the space character ( ), and
 * excluding the quote characters (') and (").
 *
 * @param[in] name Filename of the attachment shown to recipients.
 * @retval TRUE Successful validation.
 * @retval FALSE Failed to validate.
 */
static BOOL smtp_attachment_validate_name(const char *const name){
    size_t		i;
    unsigned char	uc;

    for (i = 0; name[i]; i++) {
	uc = (unsigned char)name[i];
	    if ((uc < ' ') || (uc == 127) || (uc == '\'') || (uc == '\"'))
		return FALSE;
    }
    return TRUE;
}

SMTP_STATUS_CODE smtp_attachment_add_path(struct smtp * smtp, const char * name, const char * path) {
    val_t	* new_attachment;

    if (smtp->status_code != SMTP_STATUS_OK)
	return smtp->status_code;
    if (!smtp_attachment_validate_name(name))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
    if (!fs_exists(path))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
    if ((new_attachment = val_new(name)) == NULL)
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);

    val_file(new_attachment,  path, NULL);
    val_dict_add(smtp->attachment_list, new_attachment);
    return smtp->status_code;
}

/**
 * Splits a string into smaller chunks separated by a terminating string.
 *
 * @param[in] s        The string to chunk.
 * @param[in] chunklen Number of bytes for each chunk in the string.
 * @param[in] end      Terminating string placed at the end of each chunk.
 * @retval char* Pointer to an allocated string with the contents split into
 *               separate chunks. The caller must free this memory when done.
 * @retval NULL  Memory allocation failure.
 */
static const char * smtp_chunk_split(struct smtp * smtp, const char * tail, const char * body, size_t chunklen, const char * end) {
    char	* snew, * ptr;
    const char	* sptr = tail;
    size_t	bodylen, endlen, taillen = 0, remain;

    if (chunklen < 1) {
	errno = EINVAL;
	return NULL;
    }

    endlen = strlen(end);
    if (!SEMPTY(tail)) {
	if ((bodylen = strlen(tail)) > 0) {
	    remain = bodylen;
	    taillen = bodylen;

	    do {
		size_t	to_write = MIN(remain, chunklen);

		if (smtp_write(smtp, sptr, to_write) != SMTP_STATUS_OK) {
		    smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
		    return NULL;
		}
		if (to_write == chunklen) {
		    taillen -= to_write;
		    smtp_puts(smtp, end);
		}
		sptr += to_write;
		remain -= to_write;
	    } while (remain > 0);
	}
    }
    if (!SEMPTY(body)) {
	if ((bodylen = strlen(body)) > 0) {
	    remain = bodylen;
	    sptr = body;

	    do {
		size_t	to_write = MIN(remain + taillen, chunklen - taillen);

		if (smtp_write(smtp, sptr, to_write) != SMTP_STATUS_OK) {
		    smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
		    return NULL;
		}
		if ((taillen + to_write) == chunklen)
		    smtp_puts(smtp, end);
		sptr += to_write;
		remain -= to_write;
		taillen = 0;
	    } while (remain >= chunklen);
	}
    }
    return sptr;
}

static SMTP_STATUS_CODE smtp_attachment_send(struct smtp * smtp, const val_t * attachement) {
    char	* base64, buf[1024];
    const char	* chunked = NULL;
    SOCK_T	fd;
    ssize_t	nread;

    if (smtp->status_code != SMTP_STATUS_OK)
	return smtp->status_code;

    if ((fd = open(val_file_get_name(attachement), O_RDONLY | O_EXCL | O_CLOEXEC)) < 0) {
	log_errno("Could not open file %s", val_file_get_name(attachement));
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);
    }

    while ((nread = sock_read(fd, buf, sizeof(buf), 100)) > 0) {
	if ((base64 = b64_encode((const char *)buf, (size_t)nread)) == NULL) {
	    sock_close(fd);
	    return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
	}
	chunked = smtp_chunk_split(smtp, chunked, base64, SMTP_LINE_MAX, "\r\n");
	ZFREE(base64);
	if (SEMPTY(chunked))
	    break;
    }
    sock_close(fd);

    return smtp->status_code;
}

void smtp_attachment_clear_all(struct smtp * smtp) {
    if (smtp->attachment_list != NULL) {
	val_free(smtp->attachment_list);
	smtp->attachment_list = NULL;
    }
}


/**
 * Print a MIME section containing an attachment.
 *
 * @param[in] smtp       SMTP client context.
 * @param[in] boundary   MIME boundary text.
 * @param[in] attachment See @ref smtp_attachment.
 * @return See @ref smtp_status_code.
 */
SMTP_STATUS_CODE smtp_print_mime_attachment(struct smtp * smtp, const char * boundary, const val_t * attachment) {
    smtp_printf(smtp, "--%s\r\nContent-Type: application/octet-stream\r\nContent-Disposition: attachment; filename=\"%s\"\r\nContent-Transfer-Encoding: base64\r\n\r\n", boundary, val_name(attachment));
    smtp_attachment_send(smtp, attachment);
    smtp_puts(smtp, "\r\n\r\n");
    return smtp->status_code;
}
