#include "smtp.h"

/**
 * Validate characters in the email address.
 *
 * The email address must consist only of printable characters excluding
 * the angle brackets (<) and (>).
 *
 * @param[in] email The email address of the party.
 * @retval TRUE Successful validation.
 * @retval FALSE Failed to validate.
 */
static BOOL smtp_address_validate_email(const char * email) {
    size_t i;
    unsigned char uc;

    for (i = 0; email[i]; i++) {
	uc = (unsigned char)email[i];
	if ((uc <= ' ') || (uc == 127) || (uc == '<') || (uc == '>'))
	    return FALSE;
    }
    return TRUE;
}

/**
 * Validate characters in the email name.
 *
 * Email user name must consist only of printable characters, excluding the
 * double quote character.
 *
 * @param[in] name Email name to validate.
 * @retval TRUE Successful validation.
 * @retval FALSE Failed to validate.
 */
static BOOL smtp_address_validate_name(const char *const name){
    size_t i;
    unsigned char uc;

    for (i = 0; name[i]; i++) {
	uc = (unsigned char)name[i];
	if ((uc < ' ') || (uc == 127) || (uc == '\"'))
	    return FALSE;
    }
    return TRUE;
}

SMTP_STATUS_CODE smtp_address_add(struct smtp * smtp,
                 SMTP_ADDRESS_TYPE type,
                 const char * email,
                 const char * name) {
    val_t	* new_address, * address_list;

    if (smtp->status_code != SMTP_STATUS_OK)
	return smtp->status_code;

    if ((type < 0) || (type > SMTP_ADDRESS_LAST))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);

    if (!smtp_address_validate_email(email))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);

    if (!SEMPTY(name) && !smtp_address_validate_name(name))
	return smtp_status_code_set(smtp, SMTP_STATUS_PARAM);

    address_list = smtp->address_list[type];
    if (address_list == NULL) {
	if ((address_list = val_new(NULL)) == NULL)
	    return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
	smtp->address_list[type] = address_list;
    }
    if ((new_address = val_new(email)) == NULL)
	return smtp_status_code_set(smtp, SMTP_STATUS_NOMEM);
    val_from_str(new_address, name);
    val_dict_add(address_list, new_address);

    return smtp->status_code;
}

void smtp_address_clear_all(struct smtp * smtp){
    int	i;

    for (i = 0; i < SMTP_ADDRESS_LAST; i++) {
	if (smtp->address_list[i] != NULL) {
	    val_free(smtp->address_list[i]);
	    smtp->address_list[i] = NULL;
	}
    }
}
