#include <plib/httpserver.h>
#include <plib/thread.h>
#include <plib/exec.h>
#include <plib/osutil.h>
#include "../httpint.h"
#ifdef HAVE_SHELL_LIB
#include "shell/webapp.h"
#endif

static WEB_METHOD web_load_method(HTTP_REQUEST * http, const char * str) {
    WEB_CONTEXT * web;
    WEB_METHOD	fn = NULL;

    if ((http != NULL) && ((web = http->app) != NULL)) {
	char	* cols[3];
	size_t	i, ncols = 3;
	
	if (http->plugin != NULL) {
	    close_plugin(http->plugin);
	    http->plugin = NULL;
	}
	for (i=0; i<3; i++)
	    cols[i] = NULL;
	split(str, cols, &ncols, ".");
	if (ncols != 3) {
	    for (i=0; i<ncols; i++) {
		if (cols[i] != NULL) {
		    ZFREE((void *)cols[i]);
		    cols[i] = NULL;
		}
	    }
	    split(web->validated_view, cols, &ncols, ".");
	}
	if (ncols == 3) {
	    char	buf[128];

#ifdef __WINDOWS__
	    snprintf(buf, sizeof(buf), "web\\%s", cols[0]);
#else
	    snprintf(buf, sizeof(buf), "web/%s", cols[0]);
#endif
	    if ((http->plugin = load_plugin(buf, cols[1], TRUE)) != NULL) {
		fn = (WEB_METHOD)get_plugin_function(http->plugin, cols[2]);
	    }
	}
	for (i=0; i < 3; i++) {
	    if (cols[i] != NULL)
		ZFREE((void *)cols[i]);
	}
    }
    return fn;
}

static BOOL web_validate(HTTP_REQUEST * http) {
    WEB_CONTEXT * web;
    BOOL	res = FALSE;

    if ((http != NULL) && ((web = http->app) != NULL)) {
	if (web->validator != NULL) {
	    char	buf[VAL_KEY_LEN];
	    WEB_METHOD	fn;

	    snprintf( buf, sizeof(buf), "sec.%s.validate", web->validator);
	    if ((fn = web_load_method(http, buf)) != NULL)
		res = (*fn)(http);
	}
    }
    return res;
}

PLEX HTTP_ERROR_CODES http_server_execute(HTTP_REQUEST * http) {
	const char	* page = NULL;

	if ((http != NULL) && (http->app != NULL) && !SEMPTY(http->app->validator)) {
	    page = http->app->not_validated_view;
	    if (web_validate(http)) {
		val_t * action = NULL;

		page = http->app->validated_view;
		if ((action = dict_find_node_near(http->request, "q", TRUE)) != NULL)
		    page = val_as_str_ptr(action);
	    }
	}
	if (page != NULL) {
	    WEB_METHOD	fn;
	
	    if ((fn = web_load_method(http, page)) != NULL) {
		if ((BOOL)(*fn)(http))
		    return HTTP_ERR_NONE;
		return HTTP_ERR_INTERNAL;
	    } else {
		log_err("Could not load method - %s", page);
	    }
	} else {
	    log_err("Not found what is to do");
	}
	return HTTP_ERR_NOT_FOUND;
}

PLEX HTTP_ERROR_CODES web_exec(WEB_CONTEXT * web, SOCK_T sock) {
    HTTP_ERROR_CODES	ret = HTTP_ERR_NONE;
    HTTP_REQUEST	* http = NULL;
    BOOL		testapp = FALSE, cgiapp = FALSE;

    if (web == NULL)
	return HTTP_ERR_INTERNAL;
    if ((strcmp(web->appname, "testfile") == 0) && (web->argc == 2)) {
	testapp = TRUE;
	web->mode |= WEB_APP_DEBUG;
	web->mode &= ~WEB_APP_ALLOW_SSL;
	if ((sock = open(web->argv[1], O_RDONLY|O_NONBLOCK|O_CLOEXEC)) < 0) {
	    log_errno("Could not open file %s", web->argv[1]);
	    return http_server_send_indication(http, HTTP_ERR_CANT_OPEN);
	}
    }
    cgiapp = (BOOL)(sock == STDIN_FILENO);
    if ((http = http_server_read_request(web, pstream_init(NULL, sock, NULL, web->connection_timeout), testapp || cgiapp)) != NULL) {
	if (testapp || cgiapp)
	    http->conn->fd = STDOUT_FILENO;
	ret = http_server_execute(http);
    }
    if (testapp)
	sock_close(sock);
    return ret;
}

PLEX WEB_CONTEXT * web_init(const char * appname, const char * domain, const char * sec, const char * invalid_page, const char * default_page, WEB_LANG default_lang, const char * cert, const char * ciphers) {
    WEB_CONTEXT * web = NULL;

    if ((web = web_alloc_ctx(domain, WEB_APP_SERVER, appname, cert, 0, ciphers)) != NULL) {
	web->validator = sec;
	web->not_validated_view = invalid_page;
	web->validated_view = default_page;
	web->lang = default_lang;
    }
    return web;
}

PLEX void http_server_destroy(struct http_server * srv) {
    PEER_ADDR	* p;

    if (srv == NULL) {
	log_err("NULLified pointer to http_server structure");
	return;
    }
    if (srv->thpool != NULL) {
	os_thpool_wait(srv->thpool);
	os_thpool_destroy(srv->thpool);
	srv->thpool = NULL;
    }
    if (!srv->external_watcher && (srv->pfd != NULL)) {
	sock_ev_close(srv->pfd);
	srv->pfd = NULL;
    }
    if (srv->http_ctx != NULL) {
	srv->http_ctx = web_free_ctx(srv->http_ctx);
    }
    if (srv->https_ctx != NULL) {
	srv->https_ctx = web_free_ctx(srv->https_ctx);
    }
    while ((p = TAILQ_FIRST(&srv->peer_addrs)) != NULL) {
	TAILQ_REMOVE(&srv->peer_addrs, p, entry);
	    pstream_close(&p->stream);
	ZFREE(p);
    }
    if (srv->uid >= 0)
	setuid(srv->uid);
    if (srv->gid >= 0)
	setgid(srv->uid);
}

PLEX BOOL http_server_run(struct http_server * srv, int * exit_flag) {
    if (srv == NULL) {
	log_err("NULLified pointer to http_server structure");
	return FALSE;
    }
    if (!SEMPTY(srv->sysuser)) {
	if (!os_change_exec_user_group(srv->sysuser, &srv->uid, srv->sysgroup, &srv->gid)) {
	    log_err("Could not chnage user/group to %s/%s", srv->sysuser, srv->sysgroup);
	    return FALSE;
	}
    }
    if (SEMPTY(srv->webroot)) {
	(void)getcwd(srv->webroot, sizeof(srv->webroot));
    } else {
	int	crx;

	if ((crx = chdir(srv->webroot)) != 0) {
	    log_errno("Could not change dir to %s", srv->webroot);
	    return FALSE;
	}
    }
    os_create_pidfile(srv->pidfile);
#ifdef HAVE_SHELL_LIB
    checkAndCreateShellHashMap(srv);
#endif
    if (!srv->external_watcher) {
	if ((srv->pfd = sock_init_read(&srv->peer_addrs, NULL, 1000, -1)) == NULL) {
	    log_err("Could not init sockets for listen");
	    return FALSE;
	}
	while (*exit_flag == 0) {
	    sock_loop_read(srv->pfd);
#ifdef HAVE_SHELL_LIB
	    finishOutdatedSessions(srv);
#endif
	}
    }
#ifdef HAVE_SHELL_LIB
    destroyShellHashMap(srv);
#endif
    fs_remove(srv->pidfile);
    return TRUE;
}
