#include <plib/httpserver.h>
#include <plib/fs.h>
#include <plib/basenet.h>
#include "../httpint.h"
#include "cgi.h"

HTTP_ERROR_CODES http_parse_cookies(HTTP_REQUEST *, const char *);

static HTTP_ERROR_CODES http_get_method(HTTP_REQUEST * ctx, const char * ptr) {
    if (!SEMPTY(ptr)) {
	if (strcmp(ptr, "CONNECT") == 0)
	    ctx->method = HTTP_REQ_CONNECT;
	else if (strcmp(ptr, "DELETE") == 0)
	    ctx->method = HTTP_REQ_DELETE;
	else if (strcmp(ptr, "HEAD") == 0)
	    ctx->method = HTTP_REQ_HEAD;
	else if (strcmp(ptr, "GET") == 0)
	    ctx->method = HTTP_REQ_GET;
	else if (strcmp(ptr, "OPTIONS") == 0)
	    ctx->method = HTTP_REQ_OPTIONS;
	else if (strcmp(ptr, "PATCH") == 0)
	    ctx->method = HTTP_REQ_PATCH;
	else if (strcmp(ptr, "POST") == 0)
	    ctx->method = HTTP_REQ_POST;
	else if (strcmp(ptr, "PUT") == 0)
	    ctx->method = HTTP_REQ_PUT;
	else if (strcmp(ptr, "TRACE") == 0)
	    ctx->method = HTTP_REQ_TRACE;
    }
    if (ctx->method != HTTP_REQ_UNKNOW) {
	if ((ctx->method & ctx->app->method_mask) == 0)
	    return HTTP_ERR_UNSUPPORTED_METHOD;
	return HTTP_ERR_NONE;
    } else if (IS_REQ_CGI_MODE(ctx))
	return HTTP_ERR_NONE;
    return HTTP_ERR_UNKNOWN_METHOD;
}

static HTTP_ERROR_CODES http_server_parse_header(HTTP_REQUEST * ctx) {
    HTTP_ERROR_CODES	retcode;
    val_t		** hdr = &ctx->req_headers;
    int64_t		cl;

    if (*hdr == NULL) {
	if ((*hdr = val_new("headers")) == NULL)
	    return HTTP_ERR_OUT_OF_MEMORY;
    }

    if (IS_REQ_CGI_MODE(ctx)) {
	const char	* ptr = getenv("REQUEST_METHOD");

	retcode = http_get_method(ctx, ptr);
	if (retcode != HTTP_ERR_NONE)
	    return retcode;

	ptr = getenv("QUERY_STRING");
	if (!SEMPTY(ptr))
	    ctx->query = ZSTRDUP(ptr);

	ptr = getenv("HTTP_REFERER");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new("Referer");

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("PATH_INFO");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new("Path-Info");

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("SERVER_NAME");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new("Server-Name");

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("SERVER_PORT");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new("Server-Port");

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("HTTP_USER_AGENT");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new(__http_ua_str);

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("REMOTE_ADDR");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new(__http_from_str);

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("CONTENT_TYPE");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new(__http_content_type_str);

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}

	ptr = getenv("CONTENT_LENGHT");
	if (!SEMPTY(ptr)) {
	    val_t	* v = val_new(__http_content_len_str);

	    if (v == NULL)
		return HTTP_ERR_OUT_OF_MEMORY;
	    val_dict_add(*hdr, v);
	    val_from_str(v, ptr);
	}
	ctx->flags |= HTTP_FLAG_CLOSE_CONNECTION;

    } else {
	/* We read from socket in server or client mode */
	BOOL	eof = FALSE;
	char	* body_start, * next_line;
	char	* fields[3];
	size_t	nf = 3;

	body_start = http_feed_next_part(ctx, __http_hdr_end, &eof);
	if (body_start == NULL)
//	    return HTTP_ERR_BAD_PACKET;
	    return HTTP_ERR_NOT_FULL_HEADER;
	body_start += 2;
	*body_start = 0;
	body_start ++;
	*body_start = 0;
	body_start ++;
	next_line = strstr((char *)blob_data(ctx->body), __http_lf);
	if (next_line == NULL)
	    return HTTP_ERR_BAD_PACKET;
	*next_line = 0;
	next_line += 2;

	if (IS_DEBUG_MODE(ctx->app))
	    fprintf(stderr, "%s\n", (char *)blob_data(ctx->body));

	split((char *)blob_data(ctx->body), fields, &nf, " ");
	if (nf < 2) {
	    retcode = HTTP_ERR_BAD_REQUEST;
	    goto hsph_out;
	}

	retcode = http_get_method(ctx, fields[0]);
	if (retcode != HTTP_ERR_NONE)
	    goto hsph_out;

	retcode = http_parse_protocol(ctx, fields[2]);
	if (retcode != HTTP_ERR_NONE)
	    goto hsph_out;

	if ((ctx->proto != HTTP_PROTO_V11) || (ctx->conn->ops != NULL))
	    ctx->flags |= HTTP_FLAG_CLOSE_CONNECTION;

	if (fields[1][0] == '/') {
	    char	* p = strstr(fields[1], "?");
	    if (p != NULL) {
		p ++;
		if ((ctx->query = ZSTRDUP(p)) == NULL) {
		    retcode = HTTP_ERR_OUT_OF_MEMORY;
		    goto hsph_out;
		}
	    }
	    if ((ctx->uri = uri_parse(fields[1])) == NULL) {
		retcode = HTTP_ERR_BAD_REQUEST;
		goto hsph_out;
	    }
	} else if ((strcmp(fields[1], "*") == 0) && (ctx->method == HTTP_REQ_OPTIONS)) {
	    if ((ctx->query = ZSTRDUP(fields[1])) == NULL) {
		    retcode = HTTP_ERR_OUT_OF_MEMORY;
		    goto hsph_out;
	    }
	} else {
	    retcode = HTTP_ERR_BAD_REQUEST;
	}

hsph_out:
	ZFREE(fields[0]);
	ZFREE(fields[1]);
	ZFREE(fields[2]);
	if (retcode != HTTP_ERR_NONE)
	    return retcode;

	blob_skip(ctx->body, next_line - (char *)blob_data(ctx->body));
	while ((next_line < body_start) && ((next_line = strstr((char *)blob_data(ctx->body), __http_lf)) != NULL)) {
	    char	key[VAL_KEY_LEN], val[1024];

	    *next_line = 0;
	    next_line += 2;
	    if (IS_DEBUG_MODE(ctx->app) && !IS_REQ_CGI_MODE(ctx))
		fprintf(stderr, "%s\n", (char *)blob_data(ctx->body));
	    if (sscanf((char *)blob_data(ctx->body), "%127[^:] : %1023[^#]", key, val) == 2) {

		if (strcmp(key, __http_cookie_str) == 0) {
		    retcode = http_parse_cookies(ctx, val);
		    if (retcode != HTTP_ERR_NONE)
			return retcode;
		} else {
		    val_t	* v = val_new(key);

		    if (v == NULL)
			return HTTP_ERR_OUT_OF_MEMORY;
		    val_dict_add(*hdr, v);
		    val_from_str(v, val);
		}
	    } else
		return HTTP_ERR_BAD_REQUEST;
	    blob_skip(ctx->body, next_line - (char *)blob_data(ctx->body));
	}
	if (ctx->proto == HTTP_PROTO_V11) {
	    if (dict_find_node_near(*hdr, __http_host_str, TRUE) == NULL)
		return HTTP_ERR_BAD_REQUEST;
	}
	blob_seek(ctx->body, body_start - (char *)blob_base(ctx->body), SEEK_SET);
    }

    if (val_as_int(dict_find_node_near(*hdr, __http_content_len_str, TRUE), &cl))
	ctx->cl = (size_t)cl;
    do {
	val_t	* v = dict_find_node_near(*hdr, "Connection", TRUE);

	if (v != NULL) {
	    const char	* cptr = val_as_str_ptr(v);

	    if (strcmp(cptr, "close") == 0)
		ctx->flags |= HTTP_FLAG_CLOSE_CONNECTION;
	}
    } while (0);
    http_parse_lang(ctx);
    http_debug_values(ctx, WEB_APP_REQUEST_HEADER);
    http_debug_values(ctx, WEB_APP_REQUEST_COOKIE);
    return HTTP_ERR_NONE;
}

static const char __http_content_lang[] = "Content-Language";

static ssize_t http_server_form_headers(HTTP_REQUEST * ctx, char * buf, size_t remain) {
    char	datetimebuf[VAL_KEY_LEN], * ptr = buf;
    ssize_t	written;
    int		i;

    if (IS_REQ_CGI_MODE(ctx))
	written = snprintf(ptr, remain, "Status: %d\r\n", (ctx->retcode == 0) ? 200 : ctx->retcode);
    else
	written = snprintf(ptr, remain, "HTTP/1.%d %d %s\r\n", ((ctx->proto == HTTP_PROTO_V11) ? 1 : 0), (ctx->retcode == 0) ? 200 : ctx->retcode, (ctx->replymsg == NULL) ? "OK" : ctx->replymsg);
    ptr += written;
    remain -= written;
    if (!SEMPTY(ctx->app->appname) && (remain > 0)) {
	written = snprintf(ptr, remain, "Server: %s\r\n", ctx->app->appname);
	ptr += written;
	remain -= written;
    }
    time_to_rfc1123str(datetimebuf, sizeof(datetimebuf), time(NULL));
    written = snprintf(ptr, remain, "Date: %s\r\n", datetimebuf);
    ptr += written;
    remain -= written;
    if (!(ctx->flags & HTTP_FLAG_CLOSE_CONNECTION) && (ctx->proto == HTTP_PROTO_V11) /*&& ((ctx->retcode < 300) || (ctx->retcode == 307) || (ctx->retcode == 401))*/)
	written = snprintf(ptr, remain, "Connection: keep-alive\r\nKeep-Alive: timeout=5; max=1000\r\n");
    else
	written = snprintf(ptr, remain, "Connection: close\r\n");
    ptr += written;
    remain -= written;

    FOREACH_IN_DICT(ctx->resp_headers, i) {
	val_t	* val = V_CHILD(ctx->resp_headers, i);
	char	* tptr = val_as_str_simple(val);

	if (!SEMPTY(tptr) && (remain > 0)) {
	    written = snprintf(ptr, remain, "%s: %s\r\n", val_name(val), tptr);
	    ptr += written;
	    remain -= written;
	}
	ZFREE(tptr);
    }
    if ((ctx->app->lang & WEB_LANG_EN) && (remain > 0)) {
	written = snprintf(ptr, remain, "%s: %s\r\n", __http_content_lang, "en-US");
	ptr += written;
	remain -= written;
    }
    if ((ctx->app->lang & WEB_LANG_RU) && (remain > 0)) {
	written = snprintf(ptr, remain, "%s: %s\r\n", __http_content_lang, "ru-RU");
	ptr += written;
	remain -= written;
    }
    if ((ctx->app->lang & WEB_LANG_FR) && (remain > 0)) {
	written = snprintf(ptr, remain, "%s: %s\r\n", __http_content_lang, "fr-FR");
	ptr += written;
	remain -= written;
    }
    if ((ctx->app->lang & WEB_LANG_ES) && (remain > 0)) {
	written = snprintf(ptr, remain, "%s: %s\r\n", __http_content_lang, "es-ES");
	ptr += written;
	remain -= written;
    }
    if ((ctx->app->lang & WEB_LANG_ZH) && (remain > 0)) {
	written = snprintf(ptr, remain, "%s: %s\r\n", __http_content_lang, "zh-ZH");
	ptr += written;
	remain -= written;
    }
    written = (ssize_t)http_print_server_cookies(ctx, ptr, remain);
    ptr += written;
    remain -= written;
    if (((ctx->retcode == 204) || (ctx->retcode == 401)) && (remain > 0)) {
	written = snprintf(ptr, remain, "%s: 0\r\n\r\n", __http_content_len_str);
	ptr += written;
	remain -= written;
    }
//    if (ptr != buf)
//	return http_write(ctx, buf, (size_t)(ptr - buf));
    return (ssize_t)(ptr - buf);
}

/* Assume that ctx->ptr has adjusted to start of POST body and eof is set if whole request has been read already */
static HTTP_ERROR_CODES parse_data_multipart(HTTP_REQUEST * ctx, const char * content_type, val_t * parent) {
    char	* ptr, * cptr;
    BOOL	eof = FALSE;
    char	boundary[512], filename[512];
    val_t	* node = NULL;
    size_t	len;

    ctx->content_type = HTTP_TYPE_MULTIPART_FORM_DATA;
    /* Determine the boundary string: */
    if ((ptr = strstr(content_type, __http_bnd_str)) == NULL)
	return HTTP_ERR_NO_BOUNDARY;
    ptr += strlen(__http_bnd_str);
    while (*ptr == '"')
	ptr ++;
    memset((void *)boundary, 0, sizeof(boundary));
    strncpy(boundary, __http_lf, sizeof(boundary));
    strncpy(boundary + 2, __http_bnd, sizeof(boundary) - 2);
    cptr = boundary + strlen(__http_bnd)+ 2;
    while ((*ptr != '\0') && (*ptr != '\r') && (*ptr != '\n') && (*ptr != '"'))
	*(cptr ++) = *(ptr ++);
    log_info("Boundary - %s", boundary + 2);
    len = strlen(boundary);
    blob_seek(ctx->body, -2, SEEK_CUR);
    memcpy(blob_data(ctx->body), __http_lf, 2);
    /* Read in until there's no more: */
    while (TRUE) {

	cptr = http_feed_next_part(ctx, boundary, &eof);


	if (cptr == NULL) {
	    if (eof || !V_IS_FILE(node)) {
		if (parent != NULL)
		    val_free(parent);
		return HTTP_ERR_BAD_REQUEST;
	    }
	    fs_add_contents_simple(val_file_get_name(node), blob_data(ctx->body), blob_left(ctx->body));
	} else {
	    char	* ccd;

	    /* We got a boundary! */
	    if (node != NULL) {
		*cptr = 0;
		if (V_IS_FILE(node)) {
		    fs_add_contents_simple(val_file_get_name(node), blob_data(ctx->body), cptr - (char *)blob_data(ctx->body));
		} else {
		    ptr = url_decode(NULL, 0, (const char *)blob_data(ctx->body), blob_left(ctx->body));
		    val_from_str(node, ptr);
		    ZFREE(ptr);
		}
		if (cptr[len+1] == '-') {
		    blob_seek(ctx->body , (cptr - (char *)blob_data(ctx->body)) + len + 2, SEEK_CUR);
		    /* We reach end of form! */
		    return HTTP_ERR_NONE;
		}
	    }
	    blob_seek(ctx->body , (cptr - (char *)blob_data(ctx->body)) + len + 2, SEEK_CUR);
	    if ((cptr = http_feed_next_part(ctx, __http_hdr_end, &eof)) == NULL) {
		return HTTP_ERR_BAD_REQUEST;
	    }
	    *cptr = 0;
	    cptr += 4; 
	    /* We should be at value or at included envelope at this moment */
	    /* We had just got a boundary, read headers: */
	    /* What kind of header is it? */
	    if ((ptr = strstr((char *)blob_data(ctx->body), __http_content_disp_str)) == NULL) {
		return HTTP_ERR_BAD_REQUEST;
	    }
	    ccd = ptr;
	    /* Content-disposition: */
	    filename[0] = 0;
	    /* Check for filename field */
	    if ((ptr = strstr(ptr, "filename=\"")) != NULL) {
		ptr += 10;
	    } else {
		ptr = ccd;
		if ((ptr = strstr(ptr, "file=\"")) != NULL)
		    ptr += 6;
	    }
	    if (ptr != NULL) {
		char	* tgt = filename;
		while ((tgt < (filename + sizeof(filename))) && (*ptr != '\0') && (*ptr != '"') && (*ptr != '\r') && (*ptr != '\n'))
		    *(tgt ++) = *(ptr ++);
		*tgt = 0;
	    }
	    if (((parent == NULL) || !V_IS_ARR(parent))) {
		char	name[VAL_KEY_LEN];
		char	* tgt = name;

	        /* For now, just look for "name=": */
		if ((ptr = strstr((char *)blob_data(ctx->body), "name=\"")) == NULL) {
		    return HTTP_ERR_BAD_REQUEST;
		}
		ptr += 6;
		while ((tgt < (name + sizeof(name))) && (*ptr != '\0') && (*ptr != '"') && (*ptr != '\r') && (*ptr != '\n'))
		    *(tgt ++) = *(ptr ++);
		*tgt = 0;
		if ((node = val_new(name)) == NULL)
		    return HTTP_ERR_OUT_OF_MEMORY;
		if (ctx->request == NULL) {
		    if ((ctx->request = val_new(NULL)) == NULL) {
			return HTTP_ERR_OUT_OF_MEMORY;
		    }
		}
		val_dict_add(ctx->request, node);
	    if (strstr((char *)blob_data(ctx->body), __http_content_type_str) != NULL) {
		if (strstr((char *)blob_data(ctx->body), __http_built_in_content_types[HTTP_TYPE_MULTIPART_MIXED]) != NULL) {
		    char	new_ct[512];
		    HTTP_ERROR_CODES	errn;

		    if (node == NULL)
			return HTTP_ERR_BAD_REQUEST;
		    strncpy(new_ct, (const char *)blob_data(ctx->body), sizeof(new_ct));
		    node->type = VAL_ARRAY;
		    blob_seek(ctx->body, cptr - (char *)blob_data(ctx->body), SEEK_CUR);
		    errn = parse_data_multipart(ctx, new_ct, node);
		    if (errn != HTTP_ERR_NONE)
			return errn;
		    node = NULL;
		    filename[0] = 0;
		    continue;
		}
	    }
		if (filename[0] != 0) {
		    val_t	* v;

		    node->type = VAL_ARRAY;
		    if ((v = val_new(filename)) == NULL)
			return HTTP_ERR_OUT_OF_MEMORY;
		    val_dict_add(node, v);
		    val_file(v, fs_tmp_name(NULL, NULL, 0), NULL);
		    node = v;
		}
	    } else {
		if (filename[0] == 0)
		    return HTTP_ERR_BAD_REQUEST;
		if ((node = val_new(filename)) == NULL)
		    return HTTP_ERR_OUT_OF_MEMORY;
		val_dict_add(parent, node);
		val_file(node, fs_tmp_name(NULL, NULL, 0), NULL);
	    }
	    blob_seek(ctx->body, cptr - (char *)blob_data(ctx->body), SEEK_CUR);
	}
    }
    return HTTP_ERR_BAD_REQUEST;
}

static HTTP_ERROR_CODES parse_data_urlencoded(HTTP_REQUEST * ctx, const char * query, size_t cl, size_t totallen) {
    char	name[VAL_KEY_LEN];
    char	* n = name;
    char	value[1024];
    char	* v = value;
    size_t	i;
    val_t	* node = NULL;

    if (cl > totallen)
	cl = totallen;
    ctx->content_type = HTTP_TYPE_APPLICATION_X_WWW_FORM_URLENCODED;
    memset(value, 0, sizeof(value));
    memset(name, 0, sizeof(name));
    while (TRUE) {
	totallen -= cl;
	log_debug("DATA size is %d and content is %s", cl, query);
	/* Go through the entire string of characters: */
	for (i = 0; i < cl; i++) {
	    switch(query[i]) {
		case '&':
		case '\0':
		case '\r':
		case '\n':
		    /* "&" represents the end of a name/value pair: */
		    if (node != NULL) {
			char	* p;

			str_unescape(value, sizeof(value), value);
			if ((p = url_decode(NULL, 0, value, sizeof(value))) == NULL)
			    return HTTP_ERR_OUT_OF_MEMORY;
			val_from_str(node, p);
			ZFREE(p);
			memset(value, 0, sizeof(value));
			v = value;
			node = NULL;
		    }
		break;
		case '=':
		    /* "=" is the end of the name half of a name/value pair: */
		    if (node != NULL)
			return HTTP_ERR_BAD_REQUEST;
		    str_unescape(name, sizeof(name), name);
		    node = val_new(name);
		    if (node == NULL)
			return HTTP_ERR_OUT_OF_MEMORY;
		    val_dict_add(ctx->request, node);
		    memset(name, 0, sizeof(name));
		    n = name;
		break;
		default:
		    if (node == NULL)
			*(n++) = query[i];
		    else
			*(v++) = query[i];
		break;
	    }
	}
	if (totallen > 0) {
	    ssize_t	nread;

	    nread = http_read(ctx, blob_base(ctx->body), MIN(totallen, blob_size(ctx->body)));
	    if (nread <= 0)
		return HTTP_ERR_CANT_READ;
	    cl = (size_t)nread;
	    query = (char *)blob_base(ctx->body);
	} else {
	    if (node != NULL) {
		char	* p;

		str_unescape(value, sizeof(value), value);
		if ((p = url_decode(NULL, 0, value, sizeof(value))) == NULL)
		    return HTTP_ERR_OUT_OF_MEMORY;
		val_from_str(node, p);
		ZFREE(p);
	    }
	    break;
	}
    }
    return HTTP_ERR_NONE;
}

PLEX HTTP_ERROR_CODES http_server_send_indication(HTTP_REQUEST * ctx, HTTP_ERROR_CODES errn) {
    log_debug("%s - %d", http_error_strings[errn], errn);
    switch (errn) {
	case HTTP_ERR_NOT_FULL_HEADER:
	case HTTP_ERR_BAD_PACKET:
	    goto drop;
	break;
	case HTTP_ERR_NO_CONTENT:
	    ctx->retcode = 204;
	    ctx->replymsg = "No content";
	break;
	case HTTP_ERR_REDIRECT:
	    ctx->retcode = 307;
	    ctx->replymsg = "Temporary redirect";
	break;
	case HTTP_ERR_NO_BOUNDARY:
	case HTTP_ERR_BAD_REQUEST:
	    ctx->retcode = 400;
	    ctx->replymsg = "Bad Request";
	break;
	case HTTP_ERR_NOT_AUTHORIZED:
	    ctx->retcode = 401;
	    ctx->replymsg = "Not authorized";
	break;
	case HTTP_ERR_NULL_QUERY_STRING:
	case HTTP_ERR_BAD_CONTENT_LENGTH:
	    ctx->retcode = 402;
	    ctx->replymsg = "Length Required";
	break;
	case HTTP_ERR_FORBIDDEN:
	    ctx->retcode = 403;
	    ctx->replymsg = "Forbidden";
	break;
	case HTTP_ERR_CANT_OPEN:
	case HTTP_ERR_NOT_FOUND:
	    ctx->retcode = 404;
	    ctx->replymsg = "Not Found";
	break;
	case HTTP_ERR_UNSUPPORTED_METHOD:
	    ctx->retcode = 405;
	    ctx->replymsg = "Method Not Allowed";
	break;
	case HTTP_ERR_INCORRECT_TYPE:
	    ctx->retcode = 406;
	    ctx->replymsg = "Not Acceptable";
	break;
	case HTTP_ERR_CANT_READ:
	case HTTP_ERR_OUT_OF_MEMORY:
	case HTTP_ERR_INTERNAL:
	    ctx->retcode = 500;
	    ctx->replymsg = "Internal Server Error";
	break;
	case HTTP_ERR_UNKNOWN_METHOD:
	    ctx->retcode = 501;
	    ctx->replymsg = "Not Implemented";
	break;
	case HTTP_ERR_UNKNOWN_PROTOCOL:
	    ctx->retcode = 505;
	    ctx->replymsg = "HTTP Version Not Supported";
	break;
	default:
	    ctx->retcode = 200;
	    ctx->replymsg = "OK";
	break;
    }
    if ((ctx->retcode != 200) && (ctx->retcode != 400) && (ctx->retcode != 301)) {
	val_t	* val = val_new(__http_cache_ctl_str);

	if (val != NULL) {
	    val_from_str(val, "no-cache,no-store");
	    http_set_v(ctx, val, WEB_APP_RESPONSE_HEADER);
	}
    }
/*
    if (ctx->retcode >= 300) {
	size_t	cl = blob_left(ctx->body);

	if (((ctx->method == HTTP_REQ_POST) || (ctx->method == HTTP_REQ_PUT) || IS_REQ_SSL_MODE(ctx)) && (ctx->cl > cl)) {
	    size_t	remain = ctx->cl - cl;

	    while (remain > 0) {
		ssize_t	nr = http_read(ctx, blob_base(ctx->body), MIN(remain, blob_size(ctx->body)));

		if (nr > 0) {
		    remain -= nr;
		} else if (remain > 0) {
		    log_err("Could not read more from HTTP connection. Remain %lu bytes. Read %ld. FD %d", remain, nr, ctx->conn->fd);
		    break;
		}
	    }
	}
    }
*/
    blob_rewind(ctx->body);
    blob_len(ctx->body) = 0;
    if ((ctx->retcode < 300) || (ctx->retcode == 401)) {
	http_write(ctx, blob_base(ctx->body), http_server_form_headers(ctx, (char *)blob_base(ctx->body), blob_size(ctx->body)));
    } else if (ctx->retcode != 400) {
	char	fname[512];

#ifdef __WINDOWS__
	snprintf(fname, sizeof(fname), ".\\codes\\%d.html", ctx->retcode);
#else
	snprintf(fname, sizeof(fname), "./codes/%d.html", ctx->retcode);
#endif
	if (fs_exists(fname)) {
	    http_write_file(ctx, fname, TRUE);
	} else {
	    blob_printf(ctx->body, "<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">\n\
<html lang=\"en\">\n<head>\n\
<meta http-equiv=\"Content-type\" content=\"text/html; charset=utf-8\">\n\
<meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">\n\
<title>%d %s</title></head>\n\
<body bgcolor=\"darkred\" text=\"white\">\n\
<style>\n.centered {\n\
font-family:arial, helvetica, freesans, sans-serif;font-size:3em;text-decoration:none;padding:4px;display:inline-block;\n\
position:fixed;top:50%;left:50%;transform:translate(-50%, -50%);text-align:center;background:aa6666\n\
}\n</style>\n\
<div class=\"centered\">\n%d %s\n</div></body></html>", ctx->retcode, ctx->replymsg, ctx->retcode, ctx->replymsg);
	    http_write_body_blob(ctx, __http_built_in_content_types[HTTP_TYPE_TEXT_HTML], ctx->body);
	}
    }
drop:
    http_free_request(ctx);
    return errn;
}

static HTTP_ERROR_CODES http_server_reconstruct_filename(HTTP_REQUEST * http, char * buf, size_t len) {
    size_t	sl;

    strncpy(buf, ".", len);
    if (http->uri) {
	if (!SEMPTY(http->uri->uri_path)) {
	    strlcat(buf, http->uri->uri_path, len);
	}
	if (!SEMPTY(http->uri->uri_filename)) {
	    strlcat(buf, http->uri->uri_filename, len);
	    if (fs_exists(buf))
		return HTTP_ERR_NONE;
	    else {
		/* Seems that uri_filename was directory */
		strlcat(buf, "/", len);
	    }
	}
    }
    sl = strlen(buf);
    if (http->app != NULL) {
	struct http_server	* srv = http->app->localcfg;

	if ((srv != NULL) && !SEMPTY(srv->indexes)) {
	    const char	* ptr = srv->indexes;

	    do {
		char	* fptr;

		ptr = pat_tokenize(ptr, NULL, &fptr);
		if (fptr != NULL) {
		    strncpy((char *)(buf + sl), fptr, len - sl);
		    ZFREE(fptr);
		    if (fs_exists(buf))
			return HTTP_ERR_NONE;
		}
	    } while (ptr != NULL);
	}
    }
    strncpy((char *)(buf + sl), "index.html", len - sl);
    if (fs_exists(buf))
	return HTTP_ERR_NONE;
    return HTTP_ERR_NOT_FOUND;
}

/* Grab data from the browser and prepare it for us. */

PLEX HTTP_REQUEST * http_server_read_request(WEB_CONTEXT * web, struct pstream_t * stream, BOOL diffrw) {
    HTTP_ERROR_CODES	errn = HTTP_ERR_NONE;
    HTTP_REQUEST	* ctx = NULL;
    const char		* cptr;

    if ((web == NULL) || (stream == NULL) || (stream->fd < 0))
	return NULL;
    if ((ctx = http_create_request(web, &errn)) == NULL)
	return NULL;
    ctx->form_headers = http_server_form_headers;
    ctx->conn = stream;
    if ((errn = http_server_parse_header(ctx)) != HTTP_ERR_NONE)
	goto http_read_err;
    if ((ctx->request = val_new("request")) == NULL) {
	errn = HTTP_ERR_OUT_OF_MEMORY;
	goto http_read_err;
    }
    cptr = http_get_content_type(ctx);
    /* Check for REQUEST_METHOD (set by HTTP server): */
    /* Determine the exact request method, and grab the data (if any)
     in the appropriate manner: */
    switch(ctx->method) {
	case HTTP_REQ_UNKNOW:
	    if ((web->argc > 1) && SVALUED(web->argv[1])) {
		ctx->method = HTTP_REQ_GET;
		/* Get a pointer to the data: */
		ctx->query = ZSTRDUP(web->argv[1]);
	    } else {
		errn = HTTP_ERR_UNKNOWN_METHOD;
		goto http_read_err;
	    }
	case HTTP_REQ_GET:
	    /* For now, assume Content Type of
	    "application/x-www-form-urlencoded"
	    (Is this a bad assumption?) */
	    /* GET method (data sent via "QUERY_STRING" env. variable): */
	    if (!SEMPTY(ctx->query)){
		/* Determine the content length by seeing how big the
		string is: */
		ctx->cl = strlen(ctx->query);
		if (ctx->cl > HTTP_MAX_QUERY_SIZE) {
		    errn = HTTP_ERR_BAD_CONTENT_LENGTH;
		    goto http_read_err;
		}
		log_info("QUERY String for GET method - %s:%lu", ctx->query, ctx->cl);
		errn = parse_data_urlencoded(ctx, ctx->query, ctx->cl, ctx->cl);
	    }
	break;
	case HTTP_REQ_PUT:
	    if (ctx->proto != HTTP_PROTO_V11) {
		errn = HTTP_ERR_UNKNOWN_METHOD;
		goto http_read_err;
	    }
	case HTTP_REQ_POST:
	    if (!SEMPTY(ctx->query)){
		/* Determine the content length by seeing how big the
		string is: */
		size_t	cl = strlen(ctx->query);
		if (cl > HTTP_MAX_QUERY_SIZE) {
		    errn = HTTP_ERR_BAD_CONTENT_LENGTH;
		    goto http_read_err;
		}
		log_info("QUERY String for POST method - %s:%lu", ctx->query, cl);
		errn = parse_data_urlencoded(ctx, ctx->query, ctx->cl, cl);
		if (errn != HTTP_ERR_NONE)
		    goto http_read_err;
	    }
	    /* Post method (data is sent to us via "stdin" in CGI case): */
	    errn = HTTP_ERR_INCORRECT_TYPE;
	    if (!SEMPTY(cptr)) {
		/* Content type is set */
		if (strstr(cptr, __http_built_in_content_types[HTTP_TYPE_APPLICATION_X_WWW_FORM_URLENCODED]) == cptr) {
		    if ((ctx->cl == 0) || (ctx->cl > HTTP_MAX_POST_SIZE))
			errn = HTTP_ERR_BAD_CONTENT_LENGTH;
		    else
			errn = HTTP_ERR_NONE;
		} else if (strstr(cptr, __http_built_in_content_types[HTTP_TYPE_MULTIPART_FORM_DATA]) == cptr) {
		    errn = HTTP_ERR_NONE;
		} else if (ctx->method == HTTP_REQ_PUT) {
		    if (ctx->cl > 0)
			errn = HTTP_ERR_NONE;
		    else
			errn = HTTP_ERR_BAD_CONTENT_LENGTH;
		}
	    }
	break;
	case HTTP_REQ_OPTIONS:
	    if (ctx->proto != HTTP_PROTO_V11) {
		errn = HTTP_ERR_UNKNOWN_METHOD;
	    } else {
		int	nmethod = 1;
		BOOL	need_comma = FALSE;
		char	buf[ASCIILINESZ];

		buf[0] = 0;
		while (nmethod <= HTTP_REQ_MAX) {
		    if (ctx->method & nmethod) {
			if (need_comma)
			    strlcat(buf, ", ", sizeof(buf));
			strlcat(buf, http_get_method_str(ctx), sizeof(buf));
			need_comma = TRUE;
		    }
		    nmethod <<= 1;
		}
		if (buf[0] != 0) {
		    val_t	* val = val_new("Allow");

		    if (val != NULL) {
			val_from_str(val, buf);
			http_set_v(ctx, val, WEB_APP_RESPONSE_HEADER);
		    }
		}
	    }
	case HTTP_REQ_HEAD:
	    errn = HTTP_ERR_NO_CONTENT;
	break;
	case HTTP_REQ_DELETE:
	    if (ctx->proto != HTTP_PROTO_V11)
		errn = HTTP_ERR_UNKNOWN_METHOD;
	break;
	default:
	    /* Something else? We can't handle it! */
	    errn = HTTP_ERR_UNKNOWN_METHOD;
	break;
    }
    if (errn == HTTP_ERR_NONE)
	errn = http_server_check_auth(ctx);
    if (errn == HTTP_ERR_NONE) {
	char	filename[MAXPATHLEN];
	BOOL	iscgireq = IS_REQ_CGI_MODE(ctx);

	if (diffrw)
	    ctx->conn->fd = STDOUT_FILENO;
	if (!iscgireq && !diffrw && ((errn = http_server_reconstruct_filename(ctx, filename, sizeof(filename))) == HTTP_ERR_NONE)) {
	    /* We found file */
	    if ((ctx->filename = ZSTRDUP(filename)) != NULL) {
		if (http_server_is_file_for_cgi(ctx, filename)) {
		    if ((ctx->method == HTTP_REQ_GET) || (ctx->method == HTTP_REQ_POST)) {
			if ((ctx->method == HTTP_REQ_POST) && ((ctx->cl == 0) || (ctx->cl > HTTP_MAX_POST_SIZE)))
			    errn = HTTP_ERR_BAD_CONTENT_LENGTH;
			else {
			    /* Need to run CGI */
			    ctx->flags |= HTTP_EXEC_CGI;
			    return ctx;
			}
		    } else
			errn = HTTP_ERR_UNKNOWN_METHOD;
		} else {
		    /* Just process it */
		    return ctx;
		}
	    } else
		errn = HTTP_ERR_OUT_OF_MEMORY;
	}
	if (errn == HTTP_ERR_NOT_FOUND) {
	    if (ctx->method == HTTP_REQ_PUT) {
		if ((strstr(cptr, __http_built_in_content_types[HTTP_TYPE_APPLICATION_X_WWW_FORM_URLENCODED]) == NULL)
		     && (strstr(cptr, __http_built_in_content_types[HTTP_TYPE_MULTIPART_FORM_DATA]) == NULL)) {

		    return ctx;
		}
	    }
	    if ((errn == HTTP_ERR_NOT_FOUND) && IS_ALLOW_VIRTUAL(ctx->app) && !SEMPTY(ctx->query) && (dict_find_node_near(ctx->request, "q", TRUE) != NULL))
		errn = HTTP_ERR_NONE;
	}
	if (errn == HTTP_ERR_NONE) {
	    if ((ctx->method == HTTP_REQ_POST) || (ctx->method == HTTP_REQ_PUT)) {
		if (strstr(cptr, __http_built_in_content_types[HTTP_TYPE_APPLICATION_X_WWW_FORM_URLENCODED]) == cptr) {
		    errn = parse_data_urlencoded(ctx, (const char *)blob_data(ctx->body), blob_left(ctx->body), ctx->cl);
		} else if (strstr(cptr, __http_built_in_content_types[HTTP_TYPE_MULTIPART_FORM_DATA]) == cptr) {
		    errn = parse_data_multipart(ctx, cptr, ctx->request);
		} else {
		    /* Content type is unrecognized! */
		    errn = HTTP_ERR_INCORRECT_TYPE;
		}
	    }
	    if (errn == HTTP_ERR_NONE)
		return ctx;
	}
    }
http_read_err:
    if (diffrw) {
	if (ctx->conn != NULL)
	    ctx->conn->fd = STDOUT_FILENO;
	else
	    ctx->conn = pstream_init(NULL, STDOUT_FILENO, NULL, -1);
    }
    if (((errn != HTTP_ERR_NONE) && (errn != HTTP_ERR_NOT_AUTHORIZED)) || (ctx->proto != HTTP_PROTO_V11))
	ctx->flags |= HTTP_FLAG_CLOSE_CONNECTION;
    http_server_send_indication(ctx, errn);
    return NULL;
}

PLEX BOOL http_write_body_blob(HTTP_REQUEST * ctx, const char * ct, blob_t * blob) {
    BOOL	res = FALSE;
    size_t	sz = blob_len(blob);

    if ((res = http_start_output(ctx, ct, sz))) {
	blob_rewind(blob);
	res = (BOOL)(pstream_write_blob(ctx->conn, blob) == (ssize_t)sz);
    }
    return res;
}

PLEX void http_server_send_redirect(HTTP_REQUEST * ctx, const char * newurl) {
    HTTP_ERROR_CODES	errcode = HTTP_ERR_REDIRECT;
    val_t	* val = val_new("Location");

    if (val != NULL) {
	val_from_str(val, newurl);
	http_set_v(ctx, val, WEB_APP_RESPONSE_HEADER);
    } else {
	errcode = HTTP_ERR_OUT_OF_MEMORY;
    }
    http_server_send_indication(ctx, errcode);
}

static HTTP_ERROR_CODES http_server_built_in_task(HTTP_REQUEST * http) {
	HTTP_ERROR_CODES	ret = http_server_execute(http);

	if (ret == HTTP_ERR_NONE) {
	    http_free_request(http);
	    return ret;
	} else
	    return http_server_send_indication(http, ret);
}

static HTTP_ERROR_CODES __http_send_file(HTTP_REQUEST * http) {
	HTTP_ERROR_CODES	errn = HTTP_ERR_NONE;

	if (!http_write_file(http, http->filename, TRUE))
	    errn = HTTP_ERR_CANT_OPEN;
	if (errn == HTTP_ERR_NONE) {
	    http_free_request(http);
	    return errn;
	} else
	    return http_server_send_indication(http, errn);
}

static HTTP_ERROR_CODES __http_delete_file(HTTP_REQUEST * http) {
	HTTP_ERROR_CODES	errn = HTTP_ERR_NO_CONTENT;

	if (!fs_remove(http->filename))
		errn = HTTP_ERR_CANT_READ;
	return http_server_send_indication(http, errn);
}

static HTTP_ERROR_CODES __http_put_file(HTTP_REQUEST * http) {
	HTTP_ERROR_CODES	errn = HTTP_ERR_CANT_OPEN;
	size_t			left = blob_left(http->body);
	ssize_t			remain = http->cl - left;

	if ((left > 0) || (remain > 0)) {
	    SOCK_T	filefd = open(http->filename, O_WRONLY|O_CREAT|O_CLOEXEC, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);

	    if (filefd >= 0) {
		char	buf[BUFSIZ];

		if (left > 0)
		    sock_write(filefd, blob_data(http->body), left, 100);
		while (remain > 0) {
		    ssize_t	n = http_read(http, buf, MIN(remain, sizeof(buf)));
		    if (n > 0) {
			sock_write(filefd, buf, n, 100);
			remain -= n;
		    } else
			break;
		}
		if (remain == 0)
		    errn = HTTP_ERR_NO_CONTENT;
		sock_close(filefd);
	    } else {
		log_errno("Could not create file %s", http->filename);
	    }
	}
	return http_server_send_indication(http, errn);
}

static void http_server_read_task(void * arg) {
    PEER_ADDR		* peer = arg;
    HTTP_REQUEST	* http;
    WEB_CONTEXT		* ctx;
    struct http_server	* srv;

    ctx = peer->arg;
    srv = ctx->localcfg;
    if ((http = http_server_read_request(ctx, &peer->stream, FALSE)) != NULL) {
	http->data = peer;
	http_debug_values(http, WEB_APP_REQUEST_BODY);
	if (http->flags & HTTP_EXEC_CGI) {
	    do_cgi(http);
	} else {
	    blob_rewind(http->body);
	    blob_len(http->body) = 0;
	    if (SEMPTY(http->filename)) {
		http_server_built_in_task(http);
	    } else {
		switch (http->method) {
		    case HTTP_REQ_GET:
			__http_send_file(http);
		    break;
		    case HTTP_REQ_DELETE:
			__http_delete_file(http);
		    break;
		    case HTTP_REQ_PUT:
			__http_put_file(http);
		    break;
		    default:
			http_server_send_indication(http, HTTP_ERR_UNKNOWN_METHOD);
		    break;
		}
	    }
	}
    }
}

PLEX SOCK_EV_ACTIONS web_server_exec(PEER_ADDR * peer) {
    WEB_CONTEXT		* ctx;
    struct http_server	* srv;

    ctx = peer->arg;
    srv = ctx->localcfg;

    if ((peer == NULL) || (peer->arg == NULL) || (peer->stream.fd < 0)) {
	log_err("Wrong PEER_ADDR data");
	return SOCK_EV_CANCEL;
    }
    os_thpool_add_work(srv->thpool, http_server_read_task, (void *)peer, NULL);

    return SOCK_EV_DELAY;
}
