#include <plib/httpserver.h>
#include <plib/fs.h>

static BOOL http_server_preinit(struct http_server * srv) {
    if (srv == NULL) {
	log_err("NULLified pointer to http_server structure");
	return FALSE;
    }
    memset(srv, 0, sizeof(*srv));
    TAILQ_INIT(&srv->peer_addrs);
    srv->servername = "PCORE HTTP server";
    srv->indexes = "index.htm";
    srv->connection_timeout = 30;
    srv->poolsize = 4;
    srv->mode = WEB_APP_ALLOW_VIRTUAL;
    srv->lang = WEB_LANG_EN;
    srv->method_mask = HTTP_REQ_GET | HTTP_REQ_POST | HTTP_REQ_HEAD | HTTP_REQ_OPTIONS | HTTP_REQ_DELETE | HTTP_REQ_PUT;
    srv->proto_mask = HTTP_PROTO_V10 | HTTP_PROTO_V11;
    srv->uid = -1;
    srv->gid = -1;
#ifdef POSIX
    strncpy(srv->pidfile, "/var/run/nu-httpd.pid", sizeof(srv->pidfile));
#endif
    return TRUE;
}

PLEX BOOL http_server_config(struct http_server * srv, int poolsize, int connection_timeout, const char * servername, const char * domain, const char * webroot, const char * indexes, WEB_LANG lang) {
    if (http_server_preinit(srv)) {
	sigset_t blockedSignal;
	sigemptyset(&blockedSignal);
	sigaddset(&blockedSignal, SIGPIPE);
	pthread_sigmask(SIG_BLOCK, &blockedSignal, NULL);
	if (poolsize > srv->poolsize)
	    srv->poolsize = poolsize;
	if (connection_timeout > srv->connection_timeout)
	    srv->connection_timeout = connection_timeout;
	srv->lang |= lang;
	if (!SEMPTY(servername))
	    srv->servername = servername;
	if (!SEMPTY(indexes))
	    srv->indexes = indexes;
	if (!SEMPTY(webroot))
	    strncpy(srv->webroot, webroot, sizeof(srv->webroot));
	if (!SEMPTY(domain))
	    strncpy(srv->domain, domain, sizeof(srv->domain));
	if ((srv->thpool = os_thpool_init(srv->poolsize)) == NULL) {
		log_err("Could not create thread pool");
		return FALSE;
	}
	return TRUE;
    }
    return FALSE;
}

static struct pstream_ops * http_client_socket_add_ssl(PEER_ADDR * peer, SOCK_T fd) {
	WEB_CONTEXT	* app = peer->arg;

	return pstream_init_ops_ssl(&app->app_ctx, fd);
}

static const char	__def_validator[] = "none";

PLEX BOOL http_server_add_listener(struct http_server * srv, struct addr * addr, uint16_t port, BOOL https) {
    PEER_ADDR	* la;

    if ((srv == NULL) || (SEMPTY(srv->certificate) && https)) {
	log_err("Wrong server param %p or empty ceritifcate when HTTPS is required", srv);
	return FALSE;
    }
    if ((addr == NULL) || ((addr->addr_type != ADDR_TYPE_IP) && (addr->addr_type != ADDR_TYPE_IP6))) {
	log_err("Wrong address param %p or address type", addr);
	return FALSE;
    }
    if ((la = ZALLOC(sizeof(*la))) == NULL) {
	log_err("Could not allocate listen structure");
	return FALSE;
    }
    memcpy(&la->addr, addr, sizeof(la->addr));
    la->option = SOCK_TYPE_TCP|SOCK_OPT_RECV_MIN|SOCK_OPT_SEND_MIN|SOCK_OPT_LISTEN;
    la->handler = (sock_ev_handler)web_server_exec;
    la->port = port;
    if (https) {
	if (srv->mode & WEB_APP_ALLOW_SSL) {
	    if (srv->https_ctx == NULL) {
		if ((srv->https_ctx = web_init(srv->servername, srv->domain, __def_validator, NULL, NULL, srv->lang, srv->certificate, srv->ciphers)) == NULL) {
		    log_err("Could not init application");
		    ZFREE(la);
		    srv->mode &= ~(WEB_APP_ALLOW_SSL);
		    return FALSE;
		}
		if (!IS_ALLOW_SSL(srv->https_ctx)) {
		    log_err("Could not init crypto context for HTTPS");
		    web_cleanup_ctx(srv->https_ctx);
		    srv->https_ctx = NULL;
		    ZFREE(la);
		    srv->mode &= ~(WEB_APP_ALLOW_SSL);
		    return FALSE;
		}
		srv->https_ctx->mode = (srv->mode | WEB_APP_SERVER | WEB_APP_ALLOW_SSL);
		srv->https_ctx->mode &= ~(WEB_APP_CLIENT);
		srv->https_ctx->method_mask = (srv->method_mask | HTTP_REQ_GET | HTTP_REQ_POST);
		srv->https_ctx->proto_mask = (srv->proto_mask | HTTP_PROTO_V10);
		srv->https_ctx->localcfg = srv;
		srv->https_ctx->connection_timeout = srv->connection_timeout;
	    }
	    la->arg = (void *)srv->https_ctx;
	    la->init = http_client_socket_add_ssl;
	} else {
	    log_err("Attempt to init HTTPS but certificate is not provided");
	    ZFREE(la);
	    return FALSE;
	}
    } else {
	if (srv->http_ctx == NULL) {
	    if ((srv->http_ctx = web_init(srv->servername, srv->domain, __def_validator, NULL, NULL, srv->lang, NULL /* cert */, NULL)) == NULL) {
		log_err("Could not init application");
		ZFREE(la);
		return FALSE;
	    }
	    srv->http_ctx->mode = (srv->mode | WEB_APP_SERVER);
	    srv->http_ctx->mode &= ~( WEB_APP_CLIENT | WEB_APP_ALLOW_SSL);
	    srv->http_ctx->method_mask = (srv->method_mask | HTTP_REQ_GET | HTTP_REQ_POST);
	    srv->http_ctx->proto_mask = (srv->proto_mask | HTTP_PROTO_V10);
	    srv->http_ctx->localcfg = srv;
	    srv->http_ctx->connection_timeout = srv->connection_timeout;
	}
	la->arg = (void *)srv->http_ctx;
    }
    TAILQ_INSERT_TAIL(&srv->peer_addrs, la, entry);
    return TRUE;
}

PLEX BOOL http_server_add_ssl(struct http_server * srv, const char * certificate, const char * ciphers) {
    if (srv == NULL) {
	log_err("Wrong server param %p", srv);
	return FALSE;
    }
    if (SEMPTY(certificate) || !fs_exists(certificate)) {
	log_err("Empty or not existing certificate file %s", certificate);
	return FALSE;
    }
    srv->mode |= WEB_APP_ALLOW_SSL;
    strncpy(srv->certificate, certificate, sizeof(srv->certificate));
    if (!SEMPTY(ciphers))
	strncpy(srv->ciphers, ciphers, sizeof(srv->ciphers));
    return TRUE;
}

PLEX BOOL http_server_add_debug(struct http_server * srv, BOOL debug) {
    if (srv == NULL) {
	log_err("Wrong server param %p", srv);
	return FALSE;
    }
    if (debug) {
	srv->mode |= WEB_APP_DEBUG;
	if (srv->http_ctx != NULL)
	    srv->http_ctx->mode |= WEB_APP_DEBUG;
	if (srv->https_ctx != NULL)
	    srv->https_ctx->mode |= WEB_APP_DEBUG;
    } else {
	srv->mode &= ~WEB_APP_DEBUG;
	if (srv->http_ctx != NULL)
	    srv->http_ctx->mode &= ~WEB_APP_DEBUG;
	if (srv->https_ctx != NULL)
	    srv->https_ctx->mode &= ~WEB_APP_DEBUG;
    }
    return TRUE;
}

PLEX BOOL http_server_add_masks(struct http_server * srv, HTTP_PROTO_TYPE proto_mask, HTTP_REQUEST_TYPE request_mask) {
    if (srv == NULL) {
	log_err("Wrong server param %p", srv);
	return FALSE;
    }
    srv->proto_mask = proto_mask;
    srv->method_mask = request_mask;
    return TRUE;
}

PLEX BOOL http_server_add_listener_any(struct http_server * srv, uint16_t port, BOOL https) {
    struct addr		addr;

    if (srv == NULL) {
	log_err("Wrong server param %p", srv);
	return FALSE;
    }
    memset(&addr, 0, sizeof(addr));
    addr.addr_type = ADDR_TYPE_IP;
    if (http_server_add_listener(srv, &addr, port, https)) {
#ifdef	HAVE_IPV6
	addr.addr_type = ADDR_TYPE_IP6;
	if (http_server_add_listener(srv, &addr, port, https))
	    return TRUE;
#else
	return TRUE;
#endif
    }
    return FALSE;
}

PLEX BOOL http_server_add_runinfo(struct http_server * srv, const char * username, const char * group, const char * pidfile) {
    if (srv == NULL) {
	log_err("Wrong server param %p", srv);
	return FALSE;
    }
    if (!SEMPTY(username))
	strncpy(srv->sysuser, username, sizeof(srv->sysuser));
    if (!SEMPTY(group))
	strncpy(srv->sysgroup, group, sizeof(srv->sysgroup));
    if (!SEMPTY(pidfile))
	strncpy(srv->pidfile, pidfile, sizeof(srv->pidfile));
    return TRUE;
}

PLEX char * http_get_peername(HTTP_REQUEST * http) {
    char	buf[256];
    PEER_ADDR	* peer;

    if ((http == NULL) || (http->data == NULL))
	return NULL;
    peer = http->data;
    snprintf(buf, sizeof(buf), "%s->%s", addr_ntoa(&peer->addr), (peer->parent != NULL ? addr_ntoa(&peer->parent->addr) : ""));
    return ZSTRDUP(buf);
}
