#include "httpint.h"

struct mime_entry {
    const char	* ext;
    size_t	ext_len;
    const char	* val;
    size_t	val_len;
};

static struct mime_entry enc_tab[] = {
#include "mime_encodings.h"
};

static const int n_enc_tab = sizeof(enc_tab) / sizeof(*enc_tab);

static struct mime_entry typ_tab[] = {
#include "mime_types.h"
};

static const int n_typ_tab = sizeof(typ_tab) / sizeof(*typ_tab);

static BOOL	__mime_initialized = FALSE;

/* qsort comparison routine - declared old-style on purpose, for portability. */
static inline int ext_compare(const void * v1, const void * v2) {
    const struct mime_entry * a = (const struct mime_entry *)v1, * b = (const struct mime_entry *)v2;
    return strcmp(a->ext, b->ext);
}


static void init_mime(void) {
    int i;

    if (__mime_initialized)
	return;

    /* Sort the tables so we can do binary search. */
    qsort(enc_tab, n_enc_tab, sizeof(*enc_tab), ext_compare);
    qsort(typ_tab, n_typ_tab, sizeof(*typ_tab), ext_compare);

    /* Fill in the lengths. */
    for ( i = 0; i < n_enc_tab; ++i ) {
	enc_tab[i].ext_len = strlen(enc_tab[i].ext);
	enc_tab[i].val_len = strlen(enc_tab[i].val);
    }
    for ( i = 0; i < n_typ_tab; ++i ) {
	typ_tab[i].ext_len = strlen(typ_tab[i].ext);
	typ_tab[i].val_len = strlen(typ_tab[i].val);
    }
    __mime_initialized = TRUE;
}


/* Figure out MIME encodings and type based on the filename.  Multiple
** encodings are separated by commas, and are listed in the order in
** which they were applied to the file.
*/
const char * http_get_mime_type(const char * name, char * me, size_t me_size) {
    const char	* prev_dot, * dot, * ext;
    int		me_indexes[100];
    size_t	ext_len, me_len, n_me_indexes;
    int		i, top, bot, mid, r;
    const char	* default_type = "text/plain; charset=%s";
    const char	* type;

    init_mime();
    /* Peel off encoding extensions until there aren't any more. */
    n_me_indexes = 0;
    for (prev_dot = &name[strlen(name)]; ; prev_dot = dot) {
	for (dot = prev_dot - 1; dot >= name && *dot != '.'; --dot) ;
	if (dot < name) {
	    /* No dot found.  No more encoding extensions, and no type
	    ** extension either.
	    */
	    type = default_type;
	    goto done;
	}
	ext = dot + 1;
	ext_len = prev_dot - ext;
	/* Search the encodings table.  Linear search is fine here, there
	** are only a few entries.
	*/
	for (i = 0; i < n_enc_tab; ++i) {
	    if (ext_len == enc_tab[i].ext_len && strncasecmp(ext, enc_tab[i].ext, ext_len ) == 0) {
		if (n_me_indexes < sizeof(me_indexes)/sizeof(*me_indexes)) {
		    me_indexes[n_me_indexes] = i;
		    ++n_me_indexes;
		}
		goto next;
	    }
	}
	/* No encoding extension found.  Break and look for a type extension. */
	break;

	next: ;
    }

    /* Binary search for a matching type extension. */
    top = n_typ_tab - 1;
    bot = 0;
    while (top >= bot) {
	mid = (top + bot) >> 1;
	r = strncasecmp(ext, typ_tab[mid].ext, ext_len);
	if (r < 0)
	    top = mid - 1;
	else if (r > 0)
	    bot = mid + 1;
	else if (ext_len < typ_tab[mid].ext_len)
	    top = mid - 1;
	else if (ext_len > typ_tab[mid].ext_len)
	    bot = mid + 1;
	else {
	    type = typ_tab[mid].val;
	    goto done;
	}
    }
    type = default_type;

    done:

    if ((me != NULL) && (me_size > 0)) {
	/* The last thing we do is actually generate the mime-encoding header. */
	me[0] = '\0';
	me_len = 0;
	for (i = n_me_indexes - 1; i >= 0; --i) {
	    if (me_len + enc_tab[me_indexes[i]].val_len + 1 < me_size) {
		if (me[0] != '\0') {
		    (void)strcpy(&me[me_len], "," );
		    ++me_len;
		}
		(void)strcpy(&me[me_len], enc_tab[me_indexes[i]].val);
		me_len += enc_tab[me_indexes[i]].val_len;
	    }
	}
    }
    return type;
}
