#include "../httpint.h"
#include <plib/crypt.h>

char * http_digest_auth_reply(const char *http_method, const char *url, 
	const char *header, const char *username, const char *password);

/**
 * make response to request authentication from the client
 *
 * @param fp
 * @param http_method
 * @param url
 * @param realm the realm presented to the client
 * @param opaque string to user for opaque value
 * @return not NULL on success, NULL otherwise
 */

static char * http_client_digest_auth_response(const char *url, 
	const char *realm, const char * nonce, const char *opaque, const char * username, const char * nc, 
	const char * cnonce, const char * response) {
	char	opq[HTTP_MAX_REALM_LENGTH];
	char	header[1024];

	if (SEMPTY(opaque))
	    opq[0] = 0;
	else
	    snprintf(opq, sizeof(opq), ",opaque=\"%s\"", opaque);
	/* Building the authentication header value */
	snprintf(header, sizeof(header),
		"Digest username=\"%s\",realm=\"%s\",qop=\"auth\",nonce=\"%s\",uri=\"%s\",nc=%s,cnonce=\"%s\",response=\"%s\"%s",
		username, realm, nonce, url, nc, cnonce, response, opq);
	return (char *)ZSTRDUP(header);
}

/**
 * Authenticates the authorization header sent by the client
 *
 * @param http_method
 * @param url
 * @param header: pointer to the position just after the string "Authorization: Digest "
 * @param realm The realm presented to the client
 * @param username The username needs to be authenticated
 * @param password The password used in the authentication
 * @param nonce_timeout The amount of time for a nonce to be
 *			invalid in seconds
 * @return MHD_YES if authenticated, MHD_NO if not,
 *			MHD_INVALID_NONCE if nonce is invalid
 */
char * http_digest_auth_reply(const char *http_method, const char *url, 
	const char *header, const char *username, const char *password) {
	size_t len;
	char realm[HTTP_MAX_REALM_LENGTH];
	char opaque[HTTP_MAX_REALM_LENGTH];
	char nonce[HTTP_MAX_NONCE_LENGTH];
	char cnonce[HTTP_MAX_NONCE_LENGTH];
	char qop[15]; /* auth,auth-int */
	char	ncount[15];
	const char *hentity = NULL; /* "auth-int" is not supported */
	char ha1[HASH_MD5_HEX_LEN + 1];
	char respexp[HASH_MD5_HEX_LEN + 1];
	char	* nptr;
	const char * pptr;
	BOOL	found = FALSE;

	len = http_lookup_sub_value(realm, sizeof(realm), header, "realm");
	if (0 == len) {
		log_err("Authentication failed, no \"realm\" in the header");
		return NULL;
	}

	if (0 == (len = http_lookup_sub_value(nonce, sizeof(nonce), header, "nonce"))) {
		log_err("Authentication failed, no \"nonce\" in the header");
		return NULL;
	}

	qop[0] = 0;
	if (0 != http_lookup_sub_value(qop, sizeof(qop), header, "qop")) {
		if (!SEMPTY(qop)) {
			pptr = qop;
			if (strcmp(qop, "auth") == 0)
				found = TRUE;
			while (!found && (pptr != NULL)) {
				pptr = pat_tokenize(pptr, ",", &nptr);
				if (nptr != NULL) {
					if (strcmp(nptr, "auth") == 0) {
						strncpy(qop, nptr, sizeof(qop));
						found = TRUE;
					}
					ZFREE(nptr);
				}
			}
			if (!found) {
				log_err("Authentication failed. Unsupported AUTH scheme");
				return NULL;
			}
		}
	}

	http_lookup_sub_value(opaque, sizeof(opaque), header, "opaque");

	snprintf(cnonce, sizeof(cnonce), "%X", (unsigned int)os_random());
	snprintf(ncount, sizeof(ncount), "%08X", 1);
	digest_calc_ha1("md5", username, realm, password, nonce, cnonce, ha1, sizeof(ha1));
	digest_calc_response(ha1, nonce, ncount, cnonce, qop, http_method, url, hentity, respexp, sizeof(respexp));
	return http_client_digest_auth_response(url, realm, nonce, opaque, username, ncount, cnonce, respexp);
}
