#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <plib/fs.h>
#include <plib/exec.h>
#include <plib/curl/pcurl.h>

#ifdef HAVE_CURL_LIB

static ssize_t __pcurl_blob_write(void * buffer, size_t size, size_t nmemb, blob_t * out) {
    if (out)
	return blob_write(out, buffer, size*nmemb);
    else
	log_err("BLOB is NULL");
    return -1;
}

static ssize_t __pcurl_blob_read(void * ptr, size_t size, size_t nmemb, blob_t * in) {
    return blob_read(in, ptr, size*nmemb);
}

static const char ua[] = "Mozilla/5.0 (X11; U; Linux i386; en-US) Gecko Firefox";

static void pcurl_destroy_ctx_final(pcurl_ctx * ctx) {
    if (ctx != NULL) {
	if (ctx->blob != NULL)
	    blob_free(ctx->blob);
	ZFREE(ctx);
    }
}

static pcurl_ctx * pcurl_alloc_ctx(const char * url, int options, pstream_t * stream, const char * cua, const char * cert, long timeout, long connection_timeout) {
    pcurl_ctx	* res = NULL;

    if (SEMPTY(url)) {
	log_err("Empty url");
	return res;
    }
    if ((res = ZALLOC(sizeof(*res))) == NULL) {
	log_err("Could not allocate pcurl_ctx structure");
	return res;
    }
    res->orig_url = url;
    if ((res->blob = blob_new()) == NULL) {
	log_err("Could not allocate blob for resposnse");
	res = ZFREE(res);
	return res;
    }
    if ((res->uri = uri_parse(res->orig_url)) == NULL) {
	log_err("Could not parse URL %s", res->orig_url);
	pcurl_destroy_ctx_final(res);
	return NULL;
    }
    res->options = options;
    res->stream = stream;
    res->ua = cua;
    res->ca = cert;
    if (res->ca != NULL)
	res->options |= PCURL_VERIFY_PEER;
    res->timeout = timeout;
    res->connection_timeout = connection_timeout;
    return res;
}

static void pcurl_ctx_cleanup(pcurl_ctx * ctx) {

    if (ctx->curl != NULL) {
	curl_easy_cleanup(ctx->curl);
	ctx->curl = NULL;
    }
    if (ctx->post_data != NULL) {
	curl_mime_free((curl_mime *)ctx->post_data);
	ctx->post_data = NULL;
    }
    if (ctx->header_data != NULL) {
	curl_slist_free_all((struct curl_slist *)ctx->header_data);
	ctx->header_data = NULL;
    }
    curl_global_cleanup();
    if (ctx->stream != NULL) {
	pstream_close(ctx->stream);
	ctx->stream = NULL;
    }
    if (ctx->uri) {
	uri_destroy(ctx->uri);
	ctx->uri = NULL;
    }
    if (ctx->url)
	ctx->url = ZFREE(ctx->url);
}

PLEX pcurl_ctx * pcurl_finalize(pcurl_ctx * ctx) {
    pcurl_ctx_cleanup(ctx);
    pcurl_destroy_ctx_final(ctx);
    return NULL;
}

static BOOL pcurl_init(pcurl_ctx * ctx) {
    BOOL	ret = FALSE;
    CURLcode	res;

    if ((res = curl_global_init(CURL_GLOBAL_DEFAULT)) != 0) {
	log_err("Could not init CURL library %d:%s", res, curl_easy_strerror(res));
	return ret;
    }
    if ((ctx->curl = curl_easy_init()) != NULL) {
	if (ctx->timeout > 0)
	    curl_easy_setopt(ctx->curl, CURLOPT_TIMEOUT, ctx->timeout);
	if (ctx->connection_timeout > 0)
	    curl_easy_setopt(ctx->curl, CURLOPT_CONNECTTIMEOUT, ctx->connection_timeout);
	curl_easy_setopt(ctx->curl, CURLOPT_NOPROGRESS, 1L);
	curl_easy_setopt(ctx->curl, CURLOPT_WRITEDATA, stdout);
	if (ctx->options & PCURL_WRITE_TO_BLOB) {
	    curl_easy_setopt(ctx->curl, CURLOPT_WRITEFUNCTION, __pcurl_blob_write);
	    curl_easy_setopt(ctx->curl, CURLOPT_WRITEDATA, ctx->blob);
	}
	if (ctx->options & PCURL_UPLOAD_REQUEST) {
	    /* enable uploading */
	    curl_easy_setopt(ctx->curl, CURLOPT_UPLOAD, 1L);
	    curl_easy_setopt(ctx->curl, CURLOPT_FTP_CREATE_MISSING_DIRS, 1L);
	    if (ctx->stream != NULL) {
		curl_easy_setopt(ctx->curl, CURLOPT_READFUNCTION, pstream_read_file);
		curl_easy_setopt(ctx->curl, CURLOPT_READDATA, ctx->stream);
		curl_easy_setopt(ctx->curl, CURLOPT_INFILESIZE_LARGE, (curl_off_t)fs_size(ctx->stream->filename, TRUE));
	    }
	} else if (ctx->stream != NULL) {
	    curl_easy_setopt(ctx->curl, CURLOPT_WRITEFUNCTION, pstream_write_file);
	    curl_easy_setopt(ctx->curl, CURLOPT_WRITEDATA, ctx->stream);
	}
	/* Switch on full protocol/debug output */
	curl_easy_setopt(ctx->curl, CURLOPT_VERBOSE, 0 /*1L*/);
	if ((ctx->uri != NULL) && (ctx->uri->uri_scheme != NULL)) {

	    if (strncmp(ctx->uri->uri_scheme, "smtp", 4) == 0) {
		ctx->options |= PCURL_SMTP_REQUEST;
		if ((ctx->uri->uri_port != 0) && (ctx->uri->uri_port != 25))
		    curl_easy_setopt(ctx->curl, CURLOPT_USE_SSL, CURLUSESSL_ALL);
		if (ctx->uri->uri_username != NULL) {
		    curl_easy_setopt(ctx->curl, CURLOPT_USERNAME, ctx->uri->uri_username);
		    ZFREE(ctx->uri->uri_username);
		    ctx->uri->uri_username = NULL;
		}
		if (ctx->uri->uri_password != NULL) {
		    curl_easy_setopt(ctx->curl, CURLOPT_PASSWORD, ctx->uri->uri_password);
		    ZFREE(ctx->uri->uri_password);
		    ctx->uri->uri_password = NULL;
		}
		/* we want to use our own read function */
		curl_easy_setopt(ctx->curl, CURLOPT_READFUNCTION, __pcurl_blob_read);
		curl_easy_setopt(ctx->curl, CURLOPT_UPLOAD, 1L);
	    } else if ((strncmp(ctx->uri->uri_scheme, "http", 4) == 0) || (strncmp(ctx->uri->uri_scheme, "ftp", 3) == 0)) {
		char	token[256];
		BOOL	need_auth = FALSE;

		curl_easy_setopt(ctx->curl, CURLOPT_USERAGENT, ((ctx->ua == NULL) ? ua : ctx->ua));
		if (strncmp(ctx->uri->uri_scheme, "ftp", 3) == 0)
		    ctx->options |= PCURL_FTP_REQUEST;
		else
		    ctx->options |= PCURL_HTTP_REQUEST;
		if (SVALUED(ctx->uri->uri_username) && SVALUED(ctx->uri->uri_password)) {
		    snprintf( token, sizeof(token), "%s:%s", ctx->uri->uri_username, ctx->uri->uri_password);
		    need_auth = TRUE;
		} else if (SVALUED(ctx->uri->uri_username)) {
		    snprintf( token, sizeof(token), "%s", ctx->uri->uri_username);
		    need_auth = TRUE;
		}
		if (need_auth) {
		    if (ctx->options & PCURL_HTTP_REQUEST)
			curl_easy_setopt(ctx->curl, CURLOPT_HTTPAUTH, CURLAUTH_ANY);
		    curl_easy_setopt(ctx->curl, CURLOPT_USERPWD, token);
		}
		if (ctx->uri->uri_username != NULL) {
		    ZFREE(ctx->uri->uri_username);
		    ctx->uri->uri_username = NULL;
		}
		if (ctx->uri->uri_password != NULL) {
		    ZFREE(ctx->uri->uri_password);
		    ctx->uri->uri_password = NULL;
		}
		if (strcmp(ctx->uri->uri_scheme, "https") == 0) {
		    curl_easy_setopt(ctx->curl, CURLOPT_SSL_VERIFYPEER, (long)(ctx->options & PCURL_VERIFY_PEER));
		    curl_easy_setopt(ctx->curl, CURLOPT_SSL_VERIFYHOST, (long)(ctx->options & PCURL_VERIFY_PEER));
		    if (((ctx->options & PCURL_VERIFY_PEER) != 0) && (ctx->ca != NULL))
			curl_easy_setopt(ctx->curl, CURLOPT_CAINFO, ctx->ca);
		}
	    }
	    ret = TRUE;
	}
    }
    return ret;
}

static int pcurl_exec(pcurl_ctx * ctx) {
    CURLcode	code;
    char	buf[1024];

    ctx->retcode = 404;
    uri_format(buf, sizeof(buf), ctx->uri);
    if ((ctx->url = ZSTRDUP(buf)) == NULL) {
	log_info("Could not allocate memory to duplicate URL %s", buf);
	return ctx->result;
    }
    curl_easy_setopt(ctx->curl, CURLOPT_URL, ctx->url);
    if ((code = curl_easy_perform(ctx->curl)) != CURLE_OK) {
	log_err("Error! Could not perform CURL request to %s (%d:%s)", ctx->url, code, curl_easy_strerror(code));
    } else {
	log_info("Successfully performed CURL request to %s", ctx->url);
	ctx->retcode = 200;
    }
    curl_easy_getinfo(ctx->curl, CURLINFO_RESPONSE_CODE, &ctx->response_code);
    curl_easy_getinfo(ctx->curl, CURLINFO_TOTAL_TIME, &ctx->elapsed);
    return ctx->retcode;
}

static BOOL prepare_http_headers(pcurl_ctx * ctx, const val_t * headers) {
    struct curl_slist	* header = (struct curl_slist *)ctx->header_data;
    int			i;

    if (header != NULL) {
	curl_slist_free_all(header);
	header = NULL;
    }    
    FOREACH_IN_DICT(headers, i) {
	char	* ptr = val_as_str_simple(V_CHILD(headers, i));

	if (ptr != NULL) {
	    char	buf[1024];

	    snprintf(buf, sizeof(buf), "%s: %s", val_name(V_CHILD(headers, i)), ptr);
	    ZFREE(ptr);
	    header = curl_slist_append(header, buf);
	}
    }
    if (header != NULL)
	curl_easy_setopt(ctx->curl, CURLOPT_HTTPHEADER, header);
    ctx->header_data = (void *)header;
    return TRUE;
}

static const char q[] = "q";
static const char amp[] = "&";
static const char eq[] = "=";

static BOOL prepare_http_get(pcurl_ctx * ctx, const val_t * fields) {
    BOOL	need_q = (BOOL)((ctx->uri != NULL) && (ctx->uri->uri_cmd == NULL)), 
		need_amp = (BOOL)(!need_q && (strstr(ctx->orig_url, amp) != NULL)),
		res = FALSE;
    blob_t	* blob = blob_new();
    int		i;
    char	* ptr;

    if (blob == NULL) {
	log_err("Could not allocate memory for blob");
	return res;
    }

    blob_write(blob, (const void *)ctx->url, strlen(ctx->url));
    ZFREE(ctx->url);
    if (need_q)
	blob_write(blob, (const void *)q, sizeof(q));
    FOREACH_IN_DICT(fields, i) {
	val_t		* child = V_CHILD(fields, i);
	const char	* name = val_name(child);

	ptr = val_as_str_simple(child);
	if (ptr != NULL) {
	    if (need_amp)
		blob_write(blob, (const void *)amp, sizeof(amp));
	    blob_write(blob, (const void *)name, strlen(name));
	    blob_write(blob, (const void *)eq, sizeof(eq));
	    blob_write(blob, (const void *)ptr, strlen(ptr));
	    ZFREE(ptr);
	    need_amp = TRUE;
	}
    }
    if ((ctx->url = (char *)ZALLOC(blob_left(blob)+1)) != NULL) {
	memmove((void *)ctx->url, (const void *)blob_base(blob), blob_len(blob));
	res = TRUE;
    }
    blob_free(blob);
    curl_easy_setopt(ctx->curl, CURLOPT_HTTPGET, 1L);
    return res;
}

PLEX pcurl_ctx * pcurl_http_get(const char * url, const val_t * hdr, const val_t * args, int options, 
				pstream_t * stream, const char * cua, const char * cert, long timeout, long connection_timeout) {
    pcurl_ctx	* ctx = pcurl_alloc_ctx(url, options, stream, cua, cert, timeout, connection_timeout);

    if (ctx != NULL) {
	if (pcurl_init(ctx) && prepare_http_headers(ctx, hdr) && prepare_http_get(ctx, args)) {
	    pcurl_exec(ctx);
	    pcurl_ctx_cleanup(ctx);
	    return ctx;
	}
	pcurl_ctx_cleanup(ctx);
	pcurl_destroy_ctx_final(ctx);
	ZFREE(ctx);
    }
    return NULL;
}

static BOOL prepare_http_post(pcurl_ctx * ctx, const val_t * fields) {
    curl_mime * form = NULL;
    int		i;

    if ((form = curl_mime_init(ctx->curl)) == NULL) {
	log_err("Could not allocate CURL mime handle");
	return FALSE;
    }

    FOREACH_IN_DICT(fields, i) {
	val_t		* child = V_CHILD(fields, i);
	const char	* name = val_name(child);
	curl_mimepart	* field = curl_mime_addpart(form);

	if (field) {
	    curl_mime_name(field, name);
	    if (V_IS_BLOB(child)) {
		curl_mime_data(field, (const char *)blob_base(val_as_blob(child)), blob_len(val_as_blob(child)));
	    } else {
		char	* ptr = val_as_str_simple(child);

		if (ptr != NULL) {
		    if (V_IS_FILE(child))
			curl_mime_filedata(field, ptr);
		    else
			curl_mime_data(field, ptr, CURL_ZERO_TERMINATED);
		    ZFREE(ptr);
		}
	    }
	} else {
	    log_err("Could not create CURL mime fileds");
	}
    }
    curl_easy_setopt(ctx->curl, CURLOPT_MIMEPOST, form);
    ctx->post_data = form;
    return TRUE;
}

PLEX pcurl_ctx * pcurl_http_post(const char * url, const val_t * hdr, const val_t * args, int options, 
				pstream_t * stream, const char * cua, const char * cert, long timeout, long connection_timeout) {
    pcurl_ctx	* ctx = pcurl_alloc_ctx(url, options, stream, cua, cert, timeout, connection_timeout);

    if (ctx != NULL) {
	if (pcurl_init(ctx) && prepare_http_headers(ctx, hdr) && prepare_http_post(ctx, args)) {
	    pcurl_exec(ctx);
	    pcurl_ctx_cleanup(ctx);
	    return ctx;
	}
	pcurl_ctx_cleanup(ctx);
	pcurl_destroy_ctx_final(ctx);
	ZFREE(ctx);
    }
    return NULL;
}

static BOOL pcurl_file(const char * url, const char * filename, BOOL upload) {
    pcurl_ctx		* ctx;
    BOOL		result = FALSE;
    pstream_t	* stream = pstream_init(filename, -1, NULL, 0);

    if (stream == NULL)
	return result;
    if ((ctx = pcurl_alloc_ctx(url, (upload ? PCURL_USE_STREAM_LOCAL_READ | PCURL_UPLOAD_REQUEST: PCURL_USE_STREAM_LOCAL_WRITE), stream, NULL, NULL, 0, 30)) != NULL) {
	if (pcurl_init(ctx))
	    result = (BOOL)(pcurl_exec(ctx) == 200);
	pcurl_finalize(ctx);
    } else pstream_close(stream);
    return result;
}

PLEX BOOL smtp_send_blob(const char * url, const char * fromaddr, const char * from,  const char * toaddr, const char * to, const char * subj, const blob_t * template) {
    pcurl_ctx	* ctx;
    BOOL	result = FALSE;
    val_t	* root = NULL;
    blob_t	* envelope = NULL;

    if (SEMPTY(url) || SEMPTY(fromaddr) || SEMPTY(from) || SEMPTY(toaddr) || SEMPTY(to) || SEMPTY(subj) || (template == NULL))
	return result;
    if ((root = val_new(NULL)) != NULL) {
	val_t * v = val_new("from");

	if (v == NULL)
	    return result;
	val_from_str(v, from);
	val_dict_add(root, v);
	if ((v = val_new("to")) == NULL)
	    goto ssb_end;
	val_from_str(v, to);
	val_dict_add(root, v);
	if ((v = val_new("subject")) == NULL)
	    goto ssb_end;
	val_from_str(v, subj);
	val_dict_add(root, v);
	if ((envelope = blob_xlat(envelope, (const char *)blob_data(template), blob_left(template), root)) == NULL)
	    goto ssb_end;
    } else return result;
    if ((ctx = pcurl_alloc_ctx(url, PCURL_USE_STREAM_LOCAL_WRITE|PCURL_SMTP_REQUEST|PCURL_UPLOAD_REQUEST, NULL, NULL, NULL, 0, 30)) != NULL) {
	if (pcurl_init(ctx)) {
	    struct curl_slist	* recipients = NULL;
	    const char		* ptr = toaddr;

	    do {
		char	* tto = NULL;

		ptr = pat_tokenize(ptr, ",", &tto);
		if (tto != NULL) {
		    recipients = curl_slist_append(recipients, tto);
		    ZFREE(tto);
		}
	    } while (ptr != NULL);
	    curl_easy_setopt(ctx->curl, CURLOPT_MAIL_FROM, fromaddr);
	    curl_easy_setopt(ctx->curl, CURLOPT_MAIL_RCPT, recipients);
	    /* we want to use our own read function */
	    curl_easy_setopt(ctx->curl, CURLOPT_READDATA, (void *)envelope);
	    result = (BOOL)(pcurl_exec(ctx) == 200);
	    curl_slist_free_all(recipients);
	}
	pcurl_finalize(ctx);
    }
ssb_end:
    if (root != NULL)
	val_free(root);
    if (envelope != NULL)
	blob_free(envelope);
    return result;
}

#else

PLEX pcurl_ctx * pcurl_finalize(pcurl_ctx * ctx) {
    web_cleanup_ctx(ctx);
    return NULL;
}

PLEX pcurl_ctx * pcurl_http_get(const char * url, const val_t * hdr, const val_t * args, int options, 
				pstream_t * stream, const char * cua, const char * cert, long timeout, long connection_timeout) {
    return http_client_get(url, hdr, args, options, stream, cua, cert, (int)connection_timeout);
}

PLEX pcurl_ctx * pcurl_http_post(const char * url, const val_t * hdr, const val_t * args, int options, 
				pstream_t * stream, const char * cua, const char * cert, long timeout, long connection_timeout) {
    return http_client_post(url, hdr, args, options, stream, cua, cert, (int)connection_timeout);
}

static BOOL pcurl_file(const char * url, const char * filename, BOOL upload) {
    if (upload)
	return FALSE;
    else
	return http_client_download_file(url, filename);
}

PLEX BOOL smtp_send_blob(const char * url, const char * fromaddr, const char * from,  const char * toaddr, const char * to, const char * subj, const blob_t * template) {
    return FALSE;
}
#endif

static BOOL scp_file(struct uri_t * uri, const char * filename, BOOL download, int bw) {
    char buf[1024], port[12], pass[128], ban[64];
    BOOL	res = FALSE;
    char	* cmd = NULL;
    char	* scpcmd = fs_find_exe(NULL, "scp");

    if (SEMPTY(scpcmd))
	goto sf_end;
    if (uri->uri_password != NULL) {
	char * scmd = fs_find_exe(NULL, "sshpass");

	if (SEMPTY(scmd))
	    goto sf_end;
	snprintf(pass, sizeof(pass), "%s -p %s", scmd, uri->uri_password);
	ZFREE(scmd);
	ZFREE(uri->uri_password);
	uri->uri_password = NULL;
    } else pass[0] = 0;
    if (uri->uri_scheme != NULL) {
	ZFREE( uri->uri_scheme);
	uri->uri_scheme = NULL;
    }
    if (uri->uri_port > 0)
	snprintf(port, sizeof(port), "-P %d", uri->uri_port);
    else port[0] = 0;
    uri->uri_port = (u_int16_t)-1;
    uri_format(buf, sizeof(buf), uri);
    if (bw > 0)
	snprintf( ban, sizeof(ban), "-l %d", bw);
    else ban[0] = 0;
    if (download)
	res = os_system("%s %s -q %s %s %s %s", pass, scpcmd, port, ban, buf, filename);
    else
	res = os_system("%s %s -q %s %s %s %s", pass, scpcmd, port, ban, filename, buf);
sf_end:
    if (cmd != NULL)
	ZFREE(cmd);
    if (scpcmd != NULL)
	ZFREE(scpcmd);
    return res;
}

static BOOL rsync_file(struct uri_t * uri, const char * filename, BOOL download, int bw) {
    BOOL res = FALSE;
    int	pid;
    char cmd[512], buf[256], ban[64];
    val_t * env = NULL, * val;
    os_exec_t	ose = { 0 };

    if (uri->uri_password != NULL) {
	if ((env = val_new("env")) != NULL) {
	    if ((val = val_new("RSYNC_PASSWORD")) != NULL) {
		val_str(val, uri->uri_password);
		val_dict_add(env, val);
		ZFREE(uri->uri_password);
		uri->uri_password = NULL;
	    }
	}
    }
    uri_format(buf, sizeof(buf), uri);
    if (bw > 0)
	snprintf(ban, sizeof(ban), "--bwlimit=%d", bw);
    else ban[0] = 0;
    if (download)
	snprintf(cmd, sizeof(cmd), "-v -q -r --delete %s %s %s", ban, buf, filename);
    else
	snprintf(cmd, sizeof(cmd), "-v -q -r --delete %s %s %s", ban, filename, buf);
    ose.envv = env;
    pid = os_exec("rsync", cmd, &ose);
    if (env != NULL)
	val_free(env);
    if (pid > 0) {
	int status;

	waitpid(pid, &status, 0);
	res = TRUE;
    }
    return res;
}

PLEX BOOL download_file(const char * url, const char * filename, int bw) {
    struct uri_t * uri;
    BOOL res = FALSE;

    if ((uri = uri_parse( url)) == NULL) {
	log_err("Couldnt parse URI '%s'", url);
	return FALSE;
    }
    if (strcmp(uri->uri_scheme, "scp") == 0)
	res = scp_file(uri, filename, TRUE, bw);
    else if (strcmp(uri->uri_scheme, "rsync") == 0)
	res = rsync_file(uri, filename, TRUE, bw);
    else
	res = pcurl_file(url, filename, FALSE);
    uri_destroy(uri);
    return res;
}

PLEX BOOL upload_file(const char * url, const char * filename, int bw) {
    struct uri_t * uri;
    char buf[1024];
    BOOL res = FALSE;

    if ((uri = uri_parse( url)) == NULL) {
	log_err("Couldnt parse URI '%s'\n", url);
	return FALSE;
    }
    parse_filename(filename, NULL, 0, buf, sizeof(buf));
    if (uri->uri_filename != NULL)
	ZFREE(uri->uri_filename);
    if ((uri->uri_filename = (char *)ZALLOC(strlen(buf)+sizeof(char))) != NULL)
	memcpy(uri->uri_filename, buf, strlen(buf));
    if (strcmp(uri->uri_scheme, "rsync") == 0)
	res = rsync_file(uri, filename, FALSE, bw);
    else if ( strcmp(uri->uri_scheme, "scp") == 0)
	res = scp_file(uri, filename, FALSE, bw);
    else {
	uri_format(buf, sizeof(buf), uri);
	res = pcurl_file(buf, filename, TRUE);
    }
    uri_destroy(uri);
    return res;
}

PLEX BOOL smtp_file(const char * url, const char * fromaddr, const char * from,  const char * toaddr, const char * to, const char * subj, const char * filename, BOOL del) {
    BOOL	res = FALSE;
    blob_t	* template = fs_read_to_blob(NULL, filename);

    if (template != NULL) {
	blob_rewind(template);
	res = smtp_send_blob(url, fromaddr, from,  toaddr, to, subj, template);
	blob_free(template);
    }
    if (del)
	unlink(filename);
    return res;
}
